/**
 * Copyright (c) 2025 Wunderkopf Technologies GmbH
 * All rights reserved.
 *
 * Permission is hereby granted, free  of charge, to any person obtaining
 * a  copy  of this  software  and  associated  documentation files  (the
 * "Software"), to  deal in  the Software without  restriction, including
 * without limitation  the rights to  use, copy, modify,  merge, publish,
 * distribute,  sublicense, and/or sell  copies of  the Software,  and to
 * permit persons to whom the Software  is furnished to do so, subject to
 * the following conditions:
 *
 * The  above  copyright  notice  and  this permission  notice  shall  be
 * included in all copies or substantial portions of the Software.
 *
 * THE  SOFTWARE IS  PROVIDED  "AS  IS", WITHOUT  WARRANTY  OF ANY  KIND,
 * EXPRESS OR  IMPLIED, INCLUDING  BUT NOT LIMITED  TO THE  WARRANTIES OF
 * MERCHANTABILITY,    FITNESS    FOR    A   PARTICULAR    PURPOSE    AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE,  ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

package de.wunderkopfservices.creditscore.client.internal;

import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.TypeAdapter;
import com.google.gson.TypeAdapterFactory;
import com.google.gson.internal.LinkedTreeMap;
import com.google.gson.internal.bind.ObjectTypeAdapter;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.JsonWriter;


public class JSON {

	private static final Logger logger = LoggerFactory.getLogger(JSON.class);
	
	public static final DateTimeFormatter ISO_8601 = new DateTimeFormatterBuilder()
			.appendPattern("yyyy-MM-dd'T'HH:mm:ss")
			.appendFraction(ChronoField.MICRO_OF_SECOND, 0, 6, true)
			.appendPattern("'Z'")
			.toFormatter();

	
	protected static Gson gson;
	
	static {
		gson = initializeGson();
	}
	
	protected static Gson initializeGson() {
		
		GsonBuilder builder = new GsonBuilder();
		builder.registerTypeAdapterFactory(NativeJavaObjectTypeAdapter.FACTORY);

		// Register custom type adapters (selection)
		builder.registerTypeAdapter(ZonedDateTime.class, new JsonSerializer<ZonedDateTime>() {
			@Override
			public JsonElement serialize(ZonedDateTime zonedDateTime, Type type,
					JsonSerializationContext context) {
				return new JsonPrimitive(zonedDateTime.withZoneSameInstant(ZoneOffset.UTC).format(ISO_8601));
			}
		});
		builder.registerTypeAdapter(ZonedDateTime.class, new JsonDeserializer<ZonedDateTime>() {
			
			@Override
			public ZonedDateTime deserialize(JsonElement json, Type type,
					JsonDeserializationContext context) {
				return ZonedDateTime.parse(json.getAsString());
			}
		});
		builder.registerTypeAdapter(LocalDate.class, new JsonSerializer<LocalDate>() {
			@Override
			public JsonElement serialize(LocalDate date, Type typeOfSrc,
					JsonSerializationContext context) {
				return new JsonPrimitive(date.format(DateTimeFormatter.ISO_DATE));
			}
		});
		builder.registerTypeAdapter(LocalDate.class, new JsonDeserializer<LocalDate>() {
			
			@Override
			public LocalDate deserialize(JsonElement json, Type type,
					JsonDeserializationContext context) {
				return LocalDate.parse(json.getAsString(), DateTimeFormatter.ISO_DATE);
			}
		});
		

		Gson gson = builder.create();
		
		// Remove type default objectType adapter: 
		try {
			
			Field f = gson.getClass().getDeclaredField("factories");
			f.setAccessible(true);
			List<TypeAdapterFactory> factories = (List<TypeAdapterFactory>)
					f.get(gson);
			factories = factories.stream()
					.filter(factory -> !factory.getClass().getName().contains(ObjectTypeAdapter.class.getName()))
					.collect(Collectors.toList());
			f.set(gson, Collections.unmodifiableList(factories));
			
		} catch (Exception e) {
			logger.error("unable to remove default object type adapter", e);
		}
		
		return gson;
	}
	
	public static <T> T parse(String s, Class<T> clazz){
		return gson.fromJson(s, clazz);
	}

	public static String stringify(Object object) {
		return gson.toJson(object);
	}
	
	public static Map<String, Object> parseMap(String string){
		 if (string != null && string.startsWith("{")) {
			 Map<String, Object> temp = gson.fromJson(string, Map.class);
			 if (temp != null) return temp;
		 }
		 return null;
	}
	
	public static Map<String, Object> mapify(Object obj){
		return parseMap(stringify(obj));
	}

	public static <T> T parse(Map<String, Object> map, Class<T> clazz) {
		return gson.fromJson(stringify(map), clazz);
	}
	
	public static class NativeJavaObjectTypeAdapter extends TypeAdapter<Object> {

		public static final TypeAdapterFactory FACTORY = new TypeAdapterFactory() {

			@Override
			public <T> TypeAdapter<T> create(Gson gson, TypeToken<T> type) {
				if (type.getRawType() == Object.class) {
					return (TypeAdapter<T>) new NativeJavaObjectTypeAdapter(gson);
				}
				return null;
			}
			
			
		};
		
		private final Gson gson;
		
		public NativeJavaObjectTypeAdapter(Gson gson) {
			this.gson = gson;
		}
		
		@Override
		public Object read(JsonReader in) throws IOException {
			JsonToken token = in.peek();
			switch (token) {
			case BEGIN_ARRAY :
				List<Object> list = new ArrayList<>();
				in.beginArray();
				while (in.hasNext()) {
					list.add(read(in));
				}
				in.endArray();
				return list;
				
			case BEGIN_OBJECT :
				Map<String, Object> map = new LinkedTreeMap<String, Object>();
				in.beginObject();
				while(in.hasNext()) {
					map.put(in.nextName(), read(in));
				}
				in.endObject();
				return map;
			
			case STRING:
				return in.nextString();
				
			case NUMBER:
				String number = in.nextString();
				try {
					return Long.valueOf(number);
				} catch (NumberFormatException e) {
					return Double.valueOf(number);
				}
			
			case BOOLEAN:
				return in.nextBoolean();
				
			case NULL:
				in.nextNull();
				return null;
			
			default:
				throw new IllegalStateException();
			}
		}

		@Override
		public void write(JsonWriter out, Object value) throws IOException {
			if (value == null) {
				out.nullValue();
				return;
			}
			
			TypeAdapter<Object> typeAdapter = (TypeAdapter<Object>) this.gson.getAdapter(value.getClass());
			if (typeAdapter instanceof NativeJavaObjectTypeAdapter) {
				out.beginObject();
				out.endObject();
				return;
			}
			typeAdapter.write(out, value);
		}
	}
}
