/**
 * Copyright (c) 2025 Wunderkopf Technologies GmbH
 * All rights reserved.
 *
 * Permission is hereby granted, free  of charge, to any person obtaining
 * a  copy  of this  software  and  associated  documentation files  (the
 * "Software"), to  deal in  the Software without  restriction, including
 * without limitation  the rights to  use, copy, modify,  merge, publish,
 * distribute,  sublicense, and/or sell  copies of  the Software,  and to
 * permit persons to whom the Software  is furnished to do so, subject to
 * the following conditions:
 *
 * The  above  copyright  notice  and  this permission  notice  shall  be
 * included in all copies or substantial portions of the Software.
 *
 * THE  SOFTWARE IS  PROVIDED  "AS  IS", WITHOUT  WARRANTY  OF ANY  KIND,
 * EXPRESS OR  IMPLIED, INCLUDING  BUT NOT LIMITED  TO THE  WARRANTIES OF
 * MERCHANTABILITY,    FITNESS    FOR    A   PARTICULAR    PURPOSE    AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE,  ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

package de.wunderkopfservices.creditscore.client;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpRequest.BodyPublishers;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.time.Duration;
import java.time.LocalDate;
import java.util.Base64;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.wunderkopfservices.creditscore.client.internal.CreditscoreAddress;
import de.wunderkopfservices.creditscore.client.internal.CreditscorePurpose;
import de.wunderkopfservices.creditscore.client.internal.CreditscoreRequest;
import de.wunderkopfservices.creditscore.client.internal.JSON;
import de.wunderkopfservices.creditscore.client.internal.Token;
import de.wunderkopfservices.creditscore.client.internal.WkApiAuthenticator;
import de.wunderkopfservices.creditscore.client.internal.WkApiClientException;

public class CreditscoreClient {

	private static final Logger logger = LoggerFactory.getLogger(CreditscoreClient.class);
	
	private static final String REGION = "eu-west-1";
	private static final String VERSION = "v1";
	private static final String API_ENDPOINT = 
			"https://creditscore." + REGION + ".wunderkopfservices.de/" + VERSION + "/";
		
	private String refreshToken = null;
	private String organizationId = null;
	private Environment environment = null;

	private HttpClient client;
	
	/**
	 * Create a new client
	 * 
	 * @param refreshToken The refreshToken is obtained from console.wunderkopf.technology
	 * @param organizationId The corresponding organizatonId you want to charge the request to.
	 * @param environment The selection if your 
	 */
	public CreditscoreClient(String refreshToken, String organizationId, Environment environment) {
		this.refreshToken = refreshToken;
		Objects.requireNonNull(this.refreshToken, "'refreshToken' is required");
		this.organizationId = organizationId;
		Objects.requireNonNull(this.organizationId, "'organizationId' is required");
		this.environment = environment;
		Objects.requireNonNull(this.environment, "'environment' is required");
		
		this.client = HttpClient.newBuilder().connectTimeout(Duration.ofMinutes(2)).build();
	}
	
	/**
	 * Create a new production client.
	 * 
	 * @param refreshToken The refreshToken is obtained from console.wunderkopf.technology
	 * @param organizationId The corresponding organizatonId you want to charge the request to.
	 * @return
	 */
	public static CreditscoreClient prod(String refreshToken, String organizationId) {
		return new CreditscoreClient(refreshToken, organizationId, Environment.PRODUCTION);
	}
	
	/**
	 * Create a new production client.
	 * 
	 * @param refreshToken The refreshToken is obtained from console.wunderkopf.technology
	 * @param organizationId The corresponding organizatonId you want to charge the request to.
	 * @return
	 */
	public static CreditscoreClient test(String refreshToken, String organizationId) {
		return new CreditscoreClient(refreshToken, organizationId, Environment.TEST);
	}
	
	public static enum Environment {
		
		PRODUCTION,
		
		TEST;
	}
	
	private String getAccessToken() throws WkApiClientException {
		Token accessToken = WkApiAuthenticator.getInstance().getAccessToken(refreshToken);
		return Base64.getEncoder().encodeToString(accessToken.getToken().getBytes());
	}
	
	
	/**
	 * Make a request to get the credit score of a person in Germany.
	 * 
	 * @param fristname
	 * @param surname
	 * @param street
	 * @param streetnr
	 * @param postcode
	 * @param city
	 * @param birthday (optional)
	 * @param purpose 
	 * @param referenceId (max 40 characters)
	 * 
	 * @return A wrapper with the score. 
	 * 
	 * @throws WkApiClientException
	 */
	public CreditscoreResponse fetchPersonScoreDe(String fristname,
												String surname,
												String street,
												String streetnr,
												String postcode,
												String city,
												LocalDate birthday,
												CreditscorePurpose purpose,
												String referenceId) throws WkApiClientException {
		
		CreditscoreAddress address = new CreditscoreAddress();
		address.firstname = fristname;
		address.surname = surname;
		address.street = street;
		address.streetnr = streetnr;
		address.postcode = postcode;
		address.city = city;
		
		return this.fetchPersonScoreDe(address, birthday, purpose, referenceId);
		
	}
	
	protected CreditscoreResponse fetchPersonScoreDe(CreditscoreAddress address,
												  LocalDate birthday,
												  CreditscorePurpose purpose,
												  String referenceId) throws WkApiClientException {

		CreditscoreRequest request = new CreditscoreRequest();
		request.setAddress(address);
		request.setBirthday(birthday);
		request.setPurpose(purpose);
		request.setCustomerReferenceId(referenceId);
		
		return this.fetchPersonScoreDe(request);
	}
	
	protected CreditscoreResponse fetchPersonScoreDe(CreditscoreRequest creditscoreRequest) 
	throws WkApiClientException {
		
		Objects.requireNonNull(creditscoreRequest, "'creditscoreRequest' is required");
		Objects.requireNonNull(creditscoreRequest.getCustomerReferenceId(), "customer reference is required");
		Objects.requireNonNull(creditscoreRequest.getAddress(), "'address' is required");
		Objects.requireNonNull(creditscoreRequest.getPurpose(), "'purpose' is required");
		
		if (creditscoreRequest.getCustomerReferenceId().length() > 40) {
			throw new IllegalStateException("Max reference length is 40");
		}
		
		String method = environment == Environment.PRODUCTION 
				? "person_score_de" 
				: "test_person_score_de";
		
		String payload = JSON.stringify(creditscoreRequest);
		System.out.println(payload);
		logger.trace("New fetchPersonScoreDe request [payload={}]", payload);
		
		HttpRequest request = HttpRequest.newBuilder()
				.uri(URI.create(API_ENDPOINT + method))
				.header("x-organization-id", this.organizationId)
				.header("Authorization", "Bearer " + this.getAccessToken())
				.header("Accept", "application/json")
				.header("Content-Type", "application/json")
				.POST(BodyPublishers.ofString(payload))
				.build();
		try {
			
			HttpResponse<String> response = this.client.send(request, BodyHandlers.ofString());
			if (response.statusCode() != 200) {
				
				String responseBody = response.body();
				int status = response.statusCode();
				logger.error("FetchPersonScoreDe failed [responseBody={}, status={}]",
						responseBody, status);
				
				throw WkApiClientException.ofApiError(
						"fetchPersonScoreDe", responseBody, status);
			}
			
			String responseBody = response.body();
			logger.trace("API response: {}", responseBody);
			
			return parseResponse(response.body());

			
		} catch (IOException | InterruptedException e) {
			if (e.getMessage() != null && (
					e.getMessage().contains("GOAWAY") ||
					e.getMessage().contains("Received RST_STREAM: Stream not processed"))) {
				try {	
					HttpResponse<String> response = this.client.send(request, BodyHandlers.ofString());
					if (response.statusCode() != 200) {
						
						String responseBody = response.body();
						int status = response.statusCode();
						logger.error("FetchPersonScoreDe failed [responseBody={}, status={}]",
								responseBody, status);
						
						throw WkApiClientException.ofApiError(
								"fetchPersonScoreDe", responseBody, status);
					}
					return parseResponse(response.body());
				} catch (IOException | InterruptedException | NullPointerException e2){
					// Ignore and handle first error
				}
			}
			
			logger.error("Network error [uri={}]",
					request.uri().toString());
			throw new WkApiClientException("Network error", e);
		}
	}
	
	private static CreditscoreResponse parseResponse(String json) {
		Map<String, Object> body = JSON.parseMap(json);
		return JSON.parse((Map<String, Object>) body.get("success"), CreditscoreResponse.class);
	}
}

