package de.wiejack.kreator.builder.processor.propertyinitializer

import com.google.devtools.ksp.KspExperimental
import com.google.devtools.ksp.getAnnotationsByType
import com.google.devtools.ksp.getClassDeclarationByName
import com.google.devtools.ksp.processing.Resolver
import com.google.devtools.ksp.symbol.KSPropertyDeclaration
import com.google.devtools.ksp.symbol.KSTypeReference
import com.squareup.kotlinpoet.ClassName
import com.squareup.kotlinpoet.FileSpec
import com.squareup.kotlinpoet.KModifier
import com.squareup.kotlinpoet.ParameterizedTypeName.Companion.parameterizedBy
import com.squareup.kotlinpoet.PropertySpec
import com.squareup.kotlinpoet.TypeName
import com.squareup.kotlinpoet.ksp.toTypeName
import de.wiejack.kreator.builder.api.InitializeCollectionCount
import de.wiejack.kreator.builder.processor.api.BuildContext
import de.wiejack.kreator.builder.processor.api.PropertyInitializer
import de.wiejack.kreator.builder.processor.api.resolveCached
import de.wiejack.kreator.builder.processor.propertyinitializer.CollectionPropertyInitializer.Companion.resolveCollectionInitializer
import kotlin.reflect.KClass

class CollectionPropertyInitializer : PropertyInitializer {

    companion object {
        fun resolveCollectionInitializer(
            property: KSPropertyDeclaration,
            resolver: Resolver,
        ): CollectionInitializer {
            val propertyStarProjection =
                property.type.resolveCached().makeNotNullable().starProjection()

            return when {
                starProjected(resolver, List::class).isAssignableFrom(propertyStarProjection) -> {
                    CollectionInitializer.ListInitialization
                }

                starProjected(resolver, Set::class).isAssignableFrom(propertyStarProjection) -> {
                    CollectionInitializer.SetInitialization
                }

                else -> {
                    CollectionInitializer.NoInitialization
                }
            }
        }

        private fun starProjected(resolver: Resolver, clazz: KClass<*>) =
            (resolver.getClassDeclarationByName(clazz.java.canonicalName)?.asStarProjectedType()
                ?: error("Expecting to resolve type: ${clazz.java.canonicalName}"))
    }

    override fun couldInitialize(
        property: KSPropertyDeclaration,
        buildContext: BuildContext,
    ): Boolean {
        return resolveCollectionInitializer(property, buildContext.resolver).couldInitialize()
    }

    override fun initializeProperty(
        propertyDeclaration: KSPropertyDeclaration,
        buildContext: BuildContext,
        targetFile: FileSpec.Builder,
        propertyConsumer: (PropertySpec) -> Unit,
    ) {
        resolveCollectionInitializer(propertyDeclaration, buildContext.resolver).initializeProperty(
            propertyDeclaration = propertyDeclaration,
            buildContext = buildContext,
            targetFile = targetFile,
            propertyConsumer = propertyConsumer
        )
    }

    override val order: Int
        get() = 11000
}

sealed class CollectionInitializer {

    abstract fun couldInitialize(): Boolean
    open fun initializeProperty(
        propertyDeclaration: KSPropertyDeclaration,
        buildContext: BuildContext,
        targetFile: FileSpec.Builder,
        propertyConsumer: (PropertySpec) -> Unit,
    ) {
        val collectionType = resolveCollectionInitializer(propertyDeclaration, buildContext.resolver)

        val privatePropertyName = getPrivatePropertyName(propertyDeclaration)

        val privateCollectionProperty =
            PropertySpec.builder(privatePropertyName, collectionType.mutableTypeName(propertyDeclaration))
                .mutable(false)
                .addModifiers(KModifier.INTERNAL)
                .initializer(collectionType.defaultInitializer(propertyDeclaration))

        val collectionDSLProperty =
            createCollectionDSLProperty(
                targetFile = targetFile,
                propertyDeclaration = propertyDeclaration,
                privatePropertyName = privatePropertyName,
                ignoreNullability = buildContext.initializeNullableProperties,
                initializeCollectionCount = buildContext.initializeCollectionsCount,
            )

        propertyConsumer.invoke(privateCollectionProperty.build())
        propertyConsumer.invoke(collectionDSLProperty.build())
    }

    abstract fun mutableTypeName(propertyDeclaration: KSPropertyDeclaration): TypeName

    abstract fun defaultInitializer(propertyDeclaration: KSPropertyDeclaration): String

    @OptIn(KspExperimental::class)
    private fun createCollectionDSLProperty(
        targetFile: FileSpec.Builder,
        propertyDeclaration: KSPropertyDeclaration,
        privatePropertyName: String,
        ignoreNullability: Boolean,
        initializeCollectionCount: Int,
    ): PropertySpec.Builder {

        val collectionCount =
            propertyDeclaration.getAnnotationsByType(InitializeCollectionCount::class).firstOrNull()?.count
                ?: initializeCollectionCount

        val prefilledInitializer = if (collectionCount > 0) {
            targetFile.addImport("de.wiejack.kreator.builder.api", "withRandomValues")
            ".apply { withRandomValues($collectionCount) }"
        } else {
            ""
        }
        return if (ignoreNullability or propertyDeclaration.type.resolveCached().isMarkedNullable.not()) {
            PropertySpec.builder(
                propertyDeclaration.simpleName.asString(),
                ClassName("de.wiejack.kreator.builder.api", "CollectionDsl").parameterizedBy(
                    genericTypeName(propertyDeclaration)
                )
            )
                .mutable(ignoreNullability)
                .initializer("CollectionDsl($privatePropertyName)$prefilledInitializer")

        } else {
            PropertySpec.builder(
                propertyDeclaration.simpleName.asString(),
                ClassName("de.wiejack.kreator.builder.api", "CollectionDsl").parameterizedBy(
                    genericTypeName(propertyDeclaration)
                ).copy(nullable = true)
            )
                .mutable(true)
                .initializer("null")
        }
    }

    private fun genericType(propertyDeclaration: KSPropertyDeclaration): KSTypeReference {
        return propertyDeclaration.type.element?.typeArguments!!.first().type!!
    }

    protected fun genericTypeName(propertyDeclaration: KSPropertyDeclaration): TypeName {
        return genericType(propertyDeclaration).toTypeName()
    }

    fun getPrivatePropertyName(property: KSPropertyDeclaration): String {
        return CollectionInitializer.getPrivatePropertyName(property)
    }

    data object NoInitialization : CollectionInitializer() {
        override fun couldInitialize() = false

        override fun initializeProperty(
            propertyDeclaration: KSPropertyDeclaration,
            buildContext: BuildContext,
            targetFile: FileSpec.Builder,
            propertyConsumer: (PropertySpec) -> Unit,
        ) {
            throw NotImplementedError("no implementation")
        }

        override fun mutableTypeName(propertyDeclaration: KSPropertyDeclaration): TypeName {
            throw NotImplementedError("no implementation")
        }

        override fun defaultInitializer(propertyDeclaration: KSPropertyDeclaration): String {
            throw NotImplementedError("no implementation")
        }
    }

    data object ListInitialization : CollectionInitializer() {
        override fun couldInitialize() = true
        override fun mutableTypeName(propertyDeclaration: KSPropertyDeclaration): TypeName {
            return ClassName("kotlin.collections", "MutableList").parameterizedBy(genericTypeName(propertyDeclaration))
        }

        override fun defaultInitializer(propertyDeclaration: KSPropertyDeclaration): String {
            val typesJoined =
                propertyDeclaration.type.element?.typeArguments?.joinToString {
                    it.type?.resolveCached()?.declaration?.qualifiedName?.asString() ?: ""
                } ?: ""
            return "mutableListOf<$typesJoined>()"
        }
    }

    data object SetInitialization : CollectionInitializer() {
        override fun couldInitialize() = true

        override fun mutableTypeName(propertyDeclaration: KSPropertyDeclaration): TypeName {
            return ClassName("kotlin.collections", "MutableSet").parameterizedBy(genericTypeName(propertyDeclaration))
        }

        override fun defaultInitializer(propertyDeclaration: KSPropertyDeclaration): String {
            val typesJoined =
                propertyDeclaration.type.element?.typeArguments?.joinToString {
                    it.type?.resolveCached()?.declaration?.qualifiedName?.asString() ?: ""
                } ?: ""
            return "mutableSetOf<$typesJoined>()"
        }
    }

    companion object {
        fun getPrivatePropertyName(propertyDeclaration: KSPropertyDeclaration) =
            "_${propertyDeclaration.simpleName.asString()}"
    }
}
