package de.wenzlaff.twrente;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.TextStyle;
import java.util.List;
import java.util.Locale;
import java.util.UUID;

/**
 * Kalender ICS Format.
 * 
 * @author Thomas Wenzlaff
 */
public class Kalender {

	/**
	 * Erzeuge ein ICS-File.
	 * 
	 * @param fileName        der Ziel Name des ICS-File
	 * @param targetDate      das Ziel Datum des Events
	 * @param fromDate        das ab Datum, ab welchen die Einträge erzeugt werden
	 * @param summaryText     der Text des Ereignis, wenn eine # enthalten ist wird
	 *                        es durch einen Counter ersetzt
	 * @param selectedWorkday die gewählten Wochentage
	 * @throws IOException bei Fehler
	 */
	public static void generateICSFile(File fileName, LocalDate targetDate, LocalDate fromDate, String summaryText, List<String> selectedWorkday) throws IOException {

		FileWriter writer = new FileWriter(fileName);
		writer.write("BEGIN:VCALENDAR\n");
		writer.write("VERSION:2.0\n");
		writer.write("PRODID:-//TWSoft//TWRente Kalender Generator//EN\n"); // weltweit eindeutig

		LocalDate currentDate = fromDate;

		int totalDays = (int) (targetDate.toEpochDay() - fromDate.toEpochDay());
		int daysRemaining = totalDays;

		while (!currentDate.isAfter(targetDate)) {

			DayOfWeek dayOfWeek = currentDate.getDayOfWeek();
			String currentWeekday = dayOfWeek.getDisplayName(TextStyle.FULL, Locale.forLanguageTag("DE"));

			if (selectedWorkday.contains(currentWeekday)) {
				// Hier wird die Aktion ausgeführt,
				// wenn der aktuelle Wochentag in der Liste der
				// ausgewählten Tage enthalten ist
				String formattedDate = currentDate.format(DateTimeFormatter.ofPattern("yyyyMMdd"));
				String uuid = UUID.randomUUID().toString(); // Generiere eine UUID
				String uid = uuid + "@wenzlaff.info"; // Ergänze die Domain

				writer.write("BEGIN:VEVENT\n");
				writer.write("UID:" + formattedDate + uid + "\n"); // UID mit Datum, UUID und Domain
				writer.write("DTSTAMP:" + formattedDate + "T000000Z\n"); // Zeitstempel bleibt unverändert
				writer.write("DTSTART;VALUE=DATE:" + formattedDate + "\n"); // Ganztägiges Ereignis
				writer.write("DTEND;VALUE=DATE:" + formattedDate + "\n"); // Enddatum für ganztägiges Ereignis
				// Ersetzen des Platzhalters # durch die verbleibenden Tage
				writer.write("SUMMARY:" + summaryText.replace("#", String.valueOf(daysRemaining)) + "\n");
				writer.write("END:VEVENT\n");
			}
			currentDate = currentDate.plusDays(1);
			daysRemaining--;
		}
		writer.write("END:VCALENDAR\n");
		writer.close();
	}
}
