package de.wenzlaff.twrente;

import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.border.EmptyBorder;
import javax.swing.filechooser.FileNameExtensionFilter;

/**
 * Kalender Ereignis-Generator.
 * 
 * @author Thomas Wenzlaff
 */
public class Gui {

	/**
	 * Das default Zeitformat für Deutschland.
	 */
	private static final String ZEITFORMAT = "dd.MM.yyyy";

	private static List<String> selectedDays = new ArrayList<>();

	/**
	 * Starten der Gui.
	 * 
	 * @param args keine Argumente.
	 */
	public static void main(String[] args) {
		SwingUtilities.invokeLater(Gui::createAndShowGUI);
	}

	private static void createAndShowGUI() {
		JFrame frame = new JFrame("Kalender Event-Generator");
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setSize(800, 400); // Größe angepasst, um Platz für Checkboxen zu machen
		frame.setLocationRelativeTo(null);

		JPanel panel = new JPanel();
		panel.setLayout(new GridLayout(12, 1)); // Layout angepasst

		JLabel labelDateVon = new JLabel("Start Datum (" + ZEITFORMAT + "):");
		labelDateVon.setBorder(new EmptyBorder(0, 10, 0, 0));
		panel.add(labelDateVon);

		LocalDate currentDate = LocalDate.now();
		String formattedDate = currentDate.format(DateTimeFormatter.ofPattern(ZEITFORMAT));

		JTextField dateFieldVon = new JTextField(formattedDate);
		panel.add(dateFieldVon);

		JLabel labelDate = new JLabel("Ziel-Ereignisdatum (" + ZEITFORMAT + "):");
		labelDate.setBorder(new EmptyBorder(0, 10, 0, 0));
		panel.add(labelDate);

		JTextField dateField = new JTextField(ZEITFORMAT);
		panel.add(dateField);

		JLabel labelSummary = new JLabel("Text für die Zusammenfassung (für Zähler ein #):");
		labelSummary.setBorder(new EmptyBorder(0, 10, 0, 0));
		panel.add(labelSummary);

		JTextField summaryField = new JTextField("# Tage bis zur Rente");
		panel.add(summaryField);

		// Label für die Checkboxen-Gruppe
		JLabel checkboxLabel = new JLabel("Ereignis ausgeben für:");
		checkboxLabel.setBorder(new EmptyBorder(0, 10, 0, 0));
		panel.add(checkboxLabel);

		// Checkboxen für die Tage der Woche
		JPanel checkboxPanel = new JPanel(new GridLayout(1, 7));

		addCheckbox(checkboxPanel, "Montag");
		addCheckbox(checkboxPanel, "Dienstag");
		addCheckbox(checkboxPanel, "Mittwoch");
		addCheckbox(checkboxPanel, "Donnerstag");
		addCheckbox(checkboxPanel, "Freitag");
		addCheckbox(checkboxPanel, "Samstag");
		addCheckbox(checkboxPanel, "Sonntag");

		panel.add(checkboxPanel);

		JButton generateButton = new JButton("Kalender generieren");
		panel.add(generateButton);

		JButton exitButton = new JButton("Programm beenden");
		panel.add(exitButton);

		frame.add(panel);
		frame.setVisible(true);

		generateButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				String inputDate = dateField.getText();
				String summaryText = summaryField.getText();

				try {
					DateTimeFormatter formatter = DateTimeFormatter.ofPattern(ZEITFORMAT);
					LocalDate targetDate = LocalDate.parse(inputDate, formatter);

					String vonDate = dateFieldVon.getText();
					LocalDate fromDate = LocalDate.parse(vonDate, formatter);

					if (!targetDate.isAfter(fromDate)) {
						JOptionPane.showMessageDialog(frame, "Das Datum muss in der Zukunft liegen!", "Fehler", JOptionPane.ERROR_MESSAGE);
						return;
					}

					JFileChooser fileChooser = getSaveDialog();

					int userSelection = fileChooser.showSaveDialog(frame);
					File selectedFile = null;
					if (userSelection == JFileChooser.APPROVE_OPTION) {
						selectedFile = fileChooser.getSelectedFile();
					} else if (userSelection == JFileChooser.CANCEL_OPTION) {
						JOptionPane.showMessageDialog(frame, "Speichern abgebrochen, nicht gespeichert!", "Info", JOptionPane.INFORMATION_MESSAGE);
						return;
					}

					// Übergeben des Parameter-Objekts an die Methode
					Kalender.generateICSFile(selectedFile, targetDate, fromDate, summaryText, selectedDays);

					JOptionPane.showMessageDialog(frame, "Kalender erfolgreich generiert nach\n" + selectedFile, "Erfolg", JOptionPane.INFORMATION_MESSAGE);
				} catch (Exception ex) {
					JOptionPane.showMessageDialog(frame, "Ungültiges Datum. Bitte verwenden Sie das Format " + ZEITFORMAT, "Fehler", JOptionPane.ERROR_MESSAGE);
				}
			}

			private JFileChooser getSaveDialog() {
				JFileChooser fileChooser = new JFileChooser();
				fileChooser.setDialogTitle("Kalender-Events speichern unter ...");
				FileNameExtensionFilter filter = new FileNameExtensionFilter("iCal Dateien (*.ics, *.ical)", "ics", "ical");
				fileChooser.setFileFilter(filter);
				fileChooser.setCurrentDirectory(new File("."));
				fileChooser.setSelectedFile(new File("calendar.ics"));
				return fileChooser;
			}
		});
		exitButton.addActionListener(e -> System.exit(0));
	};

	private static void addCheckbox(JPanel checkboxPanel, String wochentag) {
		JCheckBox checkBox = new JCheckBox(wochentag);
		checkBox.setSelected(true);
		checkboxPanel.add(checkBox);
		selectedDays.add(wochentag);
		checkBox.addActionListener(e -> {
			if (checkBox.isSelected()) {
				selectedDays.add(wochentag);
			} else {
				selectedDays.remove(wochentag);
			}
		});
	}
}
