package de.wenzlaff.twitmztransformer;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Paths;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Erzeugt aus einer *.mm eine Mermaid Datei.
 * 
 * https://mermaid.js.org/syntax/mindmap.html
 * 
 * @author Thomas Wenzlaff
 */
public class MapDataToMermaid {

	private static final String TAB = "  ";

	/**
	 * Beispiel src/main/resources/test-in/Mermaid.mm
	 * src/main/resources/test-out/Mermaid.mm
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		if (args.length != 2) {
			System.out.println("Usage: java -cp twitmztransformer.jar de.wenzlaff.twitmztransformer.MapDataToMermaid <input.xml> <output.mm>");
			return;
		}

		String inputFilePath = args[0];
		String outputFilePath = args[1];

		MapDataToMermaid mermaid = new MapDataToMermaid();
		try {
			String map = mermaid.convertXmlToMermaid(inputFilePath);
			System.out.println(map);

			Files.writeString(Paths.get(outputFilePath), map);
			System.out.println("Erzeuge Datei: " + outputFilePath);

		} catch (Exception e) {
			System.err.println(e.getLocalizedMessage());
		}
	}

	public String convertXmlToMermaid(String inputFilePath) throws Exception {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		Document doc = builder.parse(new File(inputFilePath));
		doc.getDocumentElement().normalize();

		NodeList topicList = doc.getElementsByTagName("topic");

		// Erstelle StringBuilder für Mermaid Output
		StringBuilder mermaidBuilder = new StringBuilder();
		mermaidBuilder.append("<pre class=\"mermaid\">\n" + "---\n" + "config:\n" + "  look: handDrawn\n" + "  theme: neutral\n" + "---\n" + "mindmap\n");

		prozess(topicList, mermaidBuilder);

		mermaidBuilder.append("</pre>\n" + "<script src=\"https://cdn.jsdelivr.net/npm/mermaid/dist/mermaid.min.js\"></script>\n"
				+ "<script>mermaid.initialize({ startOnLoad: true });</script>");

		return mermaidBuilder.toString();
	}

	private void prozess(NodeList topicList, StringBuilder mermaidBuilder) {
		// Text-Inhalte der <topic>-Elemente ausgeben
		for (int i = 0; i < topicList.getLength(); i++) {
			addNodes(topicList, mermaidBuilder, i);
		}
	}

	private void addNodes(NodeList topicList, StringBuilder mermaidBuilder, int i) {
		Node node = topicList.item(i);
		if (node.getNodeType() == Node.ELEMENT_NODE) {
			Element topic = (Element) node;
			String text = removeLeadingAndTrailingStarsUndKlammern(topic.getAttribute("text"));
			if (i == 0) {
				mermaidBuilder.append(TAB + "root(" + text + ")\n");
			} else if (!text.isEmpty()) {
				mermaidBuilder.append(TAB + TAB + text + "\n");
			}
		}
	}

	private static String removeLeadingAndTrailingStarsUndKlammern(String input) {
		if (input == null) {
			return ""; // Null-Schutz
		}
		// Finde die Position der ersten eckigen Klammer '[1]: '
		// Da Links im Text:
		// text="[&lt;u&gt;Markdown][1]&lt;/u&gt;-inspirierte&#10;&#10;[1]:
		// https://de.wikipedia.org/wiki/Markdown"
		int index = input.indexOf("[1]: ");

		// Schneide den String ab, falls die Klammer gefunden wurde
		String result = (index != -1) ? input.substring(0, index) : input;

		// Entfernt führende und endende Sterne und Klammern mit einem regulären
		// Ausdruck
		return result.replaceAll("^\\*+|\\*+$|\\(|\\)|\\[|\\]|\\|", "");
	}
}
