package de.wenzlaff.twitmztransformer;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/**
 * Entpackt alle Inhalte einer itmz in ein Verzeichnis.
 * 
 * @author Thomas Wenzlaff
 *
 */
public class FileExtractor {

	public static void main(String[] args) {
		// Überprüfen, ob die richtigen Parameter übergeben wurden
		if (args.length != 2) {
			System.out.println("Usage: java -jar twitmztransformer.jar de.wenzlaff.twitmztransformer.FileExtractor <sourceDir> <destinationDir>");
			System.exit(1);
		}

		Path sourceDir = Paths.get(args[0]);
		Path destinationDir = Paths.get(args[1]);

		// Überprüfen, ob das Quellverzeichnis existiert
		if (!Files.exists(sourceDir) || !Files.isDirectory(sourceDir)) {
			System.err.println("Das Quellverzeichnis existiert nicht oder ist kein Verzeichnis: " + sourceDir);
			System.exit(1);
		}

		// Erstellen des Zielverzeichnisses, falls es nicht existiert
		try {
			Files.createDirectories(destinationDir);
		} catch (IOException e) {
			System.err.println("Fehler beim Erstellen des Zielverzeichnisses: " + destinationDir);
			e.printStackTrace();
			System.exit(1);
		}

		// Verarbeitung starten
		try {
			Files.walk(sourceDir).filter(Files::isRegularFile).forEach(file -> {
				try {
					if (file.toString().endsWith(".itmz")) {
						unzipFile(file, destinationDir);
					}
				} catch (IOException e) {
					System.err.println("Fehler beim Verarbeiten der Datei: " + file);
					e.printStackTrace();
				}
			});
		} catch (IOException e) {
			System.err.println("Fehler beim Durchlaufen des Quellverzeichnisses: " + sourceDir);
			e.printStackTrace();
			System.exit(1);
		}

		System.out.println("Dateien wurden erfolgreich verarbeitet.");
	}

	private static void unzipFile(Path zipFilePath, Path destinationDir) throws IOException {
		System.out.println("verarbeite: unzip " + zipFilePath + " nach Ziel " + destinationDir);

		String dateiname = getFileNameWithoutExtension(zipFilePath);
		Path zielDir = destinationDir.resolve(dateiname);

		try (ZipInputStream zis = new ZipInputStream(Files.newInputStream(zipFilePath))) {
			ZipEntry entry;
			while ((entry = zis.getNextEntry()) != null) {
				Path newFilePath = zielDir.resolve(entry.getName());
				if (entry.isDirectory()) {
					Files.createDirectories(newFilePath);
				} else {
					Files.createDirectories(newFilePath.getParent());
					try (OutputStream os = Files.newOutputStream(newFilePath)) {
						byte[] buffer = new byte[1024];
						int length;
						while ((length = zis.read(buffer)) > 0) {
							os.write(buffer, 0, length);
						}
					}
				}
				zis.closeEntry();
			}
		}
	}

	private static String getFileNameWithoutExtension(Path path) {
		String fileName = path.getFileName().toString();
		int dotIndex = fileName.lastIndexOf('.');
		return (dotIndex > 0) ? fileName.substring(0, dotIndex) : fileName;
	}

	private static void copyFile(Path file, Path sourceDir, Path destinationDir) throws IOException {
		Path relativePath = sourceDir.relativize(file);
		Path targetPath = destinationDir.resolve(relativePath);

		// Sicherstellen, dass das Zielverzeichnis existiert
		Files.createDirectories(targetPath.getParent());

		// Datei kopieren
		Files.copy(file, targetPath, StandardCopyOption.REPLACE_EXISTING);
	}
}
