package de.wenzlaff.mathe;

import java.math.BigInteger;

/**
 * Eine Primzahl ist eine natürliche Zahl größer als 1, die nur zwei positive
 * Teiler hat: 1 und sich selbst. Definition im Detail:
 * 
 * Natürliche Zahl: Eine Zahl, die größer oder gleich 1 ist. Teiler: Eine Zahl,
 * die ohne Rest durch eine andere Zahl teilbar ist.
 * 
 * Eigenschaften von Primzahlen:
 * 
 * 2 ist die kleinste und gleichzeitig die einzige gerade Primzahl. Jede
 * Primzahl größer als 2 ist ungerade. Eine Zahl, die mehr als zwei Teiler hat,
 * nennt man zusammengesetzte Zahl.
 * 
 * Beispiele von Primzahlen:
 * 
 * Die ersten paar Primzahlen sind: 2, 3, 5, 7, 11, 13, 17, 19, 23, 29.
 * 
 * Nicht-Primzahlen:
 * 
 * Zahlen wie 4, 6, 8, 9, 10 sind keine Primzahlen, weil sie weitere Teiler
 * außer 1 und sich selbst haben. Beispiel: 6 hat die Teiler 1, 2, 3 und 6,
 * daher ist 6 keine Primzahl.
 * 
 * @author Thomas Wenzlaff
 */
public class Primzahlen {

	/**
	 * Aufruf für die Ausgabe von Primzahlen.
	 * 
	 * @param args Start und Endewert.
	 */
	public static void main(String[] args) {
		// Überprüfen, ob genügend Argumente übergeben wurden
		if (args.length != 2) {
			System.out.println("Bitte geben Sie einen Start- und Endwert als Parameter an.");
			return;
		}

		// Eingabeparameter konvertieren
		try {
			BigInteger start = new BigInteger(args[0]);
			BigInteger end = new BigInteger(args[1]);

			// Validierung der Eingabe
			if (start.compareTo(end) > 0) {
				System.out.println("Der Startwert muss kleiner oder gleich dem Endwert sein.");
				return;
			}

			findPrimzahlImBereich(start, end);

		} catch (NumberFormatException e) {
			System.out.println("Bitte geben Sie gültige Ganzzahlen als Start- und Endwert ein.");
		}
	}

	/**
	 * Methode zur Ausgabe der Primzahlen im angegebenen Bereich
	 * 
	 * @param start Startzahl
	 * @param end   Endezahl
	 */
	public static void findPrimzahlImBereich(BigInteger start, BigInteger end) {
		System.out.println("Primzahlen im Bereich von " + start + " bis " + end + ":");
		BigInteger anzahl = BigInteger.ZERO;
		for (BigInteger i = start; i.compareTo(end) <= 0; i = i.add(BigInteger.ONE)) {
			if (isPrime(i)) {
				System.out.print(i + "\n");
				anzahl = anzahl.add(BigInteger.ONE);
			}
		}
		System.out.println("Es gibt " + anzahl + " Primzahlen in dem Bereich.");
	}

	/**
	 * Methode zur Überprüfung, ob eine Zahl eine Primzahl ist (mit BigInteger)
	 * 
	 * @param number Zahl
	 * @return true wenn es eine Primzahl ist, false wenn nicht
	 */
	private static boolean isPrime(BigInteger number) {
		// Zahlen kleiner als 2 sind keine Primzahlen
		if (number.compareTo(BigInteger.TWO) < 0) {
			return false;
		}

		// Prüfen, ob die Zahl durch irgendeine Zahl von 2 bis sqrt(number) teilbar ist
		BigInteger sqrt = sqrt(number);
		for (BigInteger i = BigInteger.TWO; i.compareTo(sqrt) <= 0; i = i.add(BigInteger.ONE)) {
			if (number.mod(i).equals(BigInteger.ZERO)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Methode zur Berechnung der Quadratwurzel einer BigInteger-Zahl
	 * (Approximation)
	 * 
	 * @param number Zahl
	 * @return ergebnis
	 */
	private static BigInteger sqrt(BigInteger number) {
		BigInteger x0 = number.divide(BigInteger.TWO);
		BigInteger x1 = (x0.add(number.divide(x0))).divide(BigInteger.TWO);
		while (x1.compareTo(x0) < 0) {
			x0 = x1;
			x1 = (x0.add(number.divide(x0))).divide(BigInteger.TWO);
		}
		return x0;
	}
}
