package de.wenzlaff.mathe;

import java.util.stream.Stream;

/**
 * 
 * Leonardo Fibonacci, auch bekannt als Leonardo von Pisa, war ein bedeutender
 * Mathematiker des Mittelalters, der um 1170 in Pisa geboren wurde und nach
 * 1240 verstarb.
 * 
 * Er ist vor allem für die Einführung des indisch-arabischen Zahlensystems in
 * Europa und die nach ihm benannte Fibonacci-Folge bekannt.
 * 
 * Die Fibonacci-Folge ist eine unendliche Zahlenreihe, die mit 0 und 1 beginnt
 * und bei der jede nachfolgende Zahl die Summe der beiden vorherigen Zahlen
 * ist. Diese Folge wird oft verwendet, um Wachstumsprozesse in der Natur zu
 * modellieren und hat eine enge Verbindung zum Goldenen Schnitt.
 * 
 * @author Thomas Wenzlaff
 *
 */
public class FibonacciSequence {

	private static final double GOLDENER_WINKEL = 137.50776405003785; // in Grad

	public static void main(String[] args) {
		if (args.length != 1) {
			System.out.println("Bitte eine Zahl als Abbruchkriterium angeben.");
			return;
		}

		long limit;
		try {
			limit = Long.parseLong(args[0]);
			if (limit < 2) {
				System.out.println("Das Abbruchkriterium muss mindestens 2 sein.");
				return;
			}
		} catch (NumberFormatException e) {
			System.out.println("Bitte eine gültige ganze Zahl eingeben.");
			return;
		}

		berechneFibonacciMitGoldenenWinkel(limit);

		printFibonacciSequence(limit);
	}

	public static void printFibonacciSequence(long endCriterion) {
		Stream.iterate(new long[] { 0, 1 }, fib -> new long[] { fib[1], fib[0] + fib[1] }).limit(endCriterion).forEach(fib -> System.out.print(fib[0] + " "));
	}

	public static void berechneFibonacciMitGoldenenWinkel(long limit) {
		long a = 0, b = 1;
		System.out.printf("%d\n%d\n", a, b);

		for (long i = 2; i < limit; i++) {
			long next = a + b;
			double winkel = Math.toDegrees(Math.atan((double) b / a));
			double differenzGoldenerWinkel = Math.abs(GOLDENER_WINKEL - winkel);
			System.out.printf("%d (Winkel: %.2f Grad, Differenz zum Goldenen Winkel: %.2f Grad)\n", next, winkel, differenzGoldenerWinkel);
			a = b;
			b = next;
		}
	}
}
