package de.wenzlaff.mathe;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;

import org.json.JSONArray;
import org.json.JSONObject;

/**
 * Der Brier-Wert ist ein Maß zur Bewertung von Prognosen und
 * Wahrscheinlichkeiten.
 * 
 * Der Brier-Wert (englisch "Brier Score") ist eine wichtige Metrik zur
 * Bewertung der Genauigkeit von Wahrscheinlichkeitsprognosen, insbesondere im
 * Bereich maschinellen Lernens, der Statistik und bei Vorhersagemodellen. Der
 * Brier-Wert hilft dabei, die Genauigkeit einer Vorhersage zu quantifizieren,
 * indem er die Differenz zwischen vorhergesagten Wahrscheinlichkeiten und den
 * tatsächlich eingetretenen Ergebnissen misst. Dieser BrierScoreCalculator
 * liest eine JSon Datei in der Form:
 * 
 * <pre>
 
 {
  "forecasts": [0.8, 0.6, 0.2, 0.9, 0.1],
  "outcomes":  [1, 1, 0, 1, 0]
 }
 * 
 * </pre>
 * 
 * ein und gibt das berechnete Ergebnis des Brier-Wert und die Inputwert in
 * einer Tabelle aus. z.B.
 * 
 * <pre>

Index      Vorhersage      Ergebnis (1=eingetreten, 0=nicht eingetreten) 
-------------------------------------
1          0,80            1         
2          0,60            1         
3          0,20            0         
4          0,90            1         
5          0,10            0         
-------------------------------------
Brier-Wert: 0,0520
 * 
 * </pre>
 * <p>
 * Der Brier-Wert wird wie folgt berechnet:
 *
 * <pre>
 *     Brier-Wert = (1 / n) * Σ (forecastᵢ - outcomeᵢ)²
 * </pre>
 *
 * wobei:
 * <ul>
 * <li>n = Anzahl der Vorhersagen</li>
 * <li>forecastᵢ = Wahrscheinlichkeitsvorhersage für das Ereignis i</li>
 * <li>outcomeᵢ = tatsächliches Ergebnis des Ereignisses i (1 für eingetreten, 0
 * für nicht eingetreten)</li>
 * </ul>
 * 
 * @author Thomas Wenzlaff
 */
public class BrierScoreCalculator {

	public static void main(String[] args) {

		// Überprüfen, ob der Dateipfad als Argument übergeben wurde
		if (args.length != 1) {
			System.out.println("Bitte geben Sie den Pfad zur JSON-Datei als Argument an.");
			System.out.println("Beispiel: java BrierScoreCalculator input.json");
			return;
		}
		// Dateipfad zur JSON-Datei
		String jsonFilePath = args[0];

		try {
			// JSON-Datei laden und als String einlesen
			String content = new String(Files.readAllBytes(Paths.get(jsonFilePath)));

			// JSON-Objekt erstellen und Arrays extrahieren
			JSONObject jsonObject = new JSONObject(content);
			JSONArray forecastsArray = jsonObject.getJSONArray("forecasts");
			JSONArray outcomesArray = jsonObject.getJSONArray("outcomes");

			double[] forecasts = new double[forecastsArray.length()];// Vorhergesagte Wahrscheinlichkeiten
			int[] outcomes = new int[outcomesArray.length()]; // Tatsächliche Ergebnisse
																// (1 = Ereignis eingetreten, 0 = Ereignis nicht eingetreten
			for (int i = 0; i < forecastsArray.length(); i++) {
				forecasts[i] = forecastsArray.getDouble(i);
				outcomes[i] = outcomesArray.getInt(i);
			}

			double brierScore = berechneBrierScore(forecasts, outcomes);

			printResult(forecasts, outcomes, brierScore);

		} catch (IOException e) {
			System.out.println("Fehler beim Lesen der JSON-Datei: " + e.getMessage());
		} catch (Exception e) {
			System.out.println("Fehler: " + e.getMessage());
		}
	}

	/**
	 * Methode zur Berechnung des Brier-Wertes.
	 * 
	 * <p>
	 * Der Brier-Wert wird wie folgt berechnet:
	 *
	 * <pre>
	 *     Brier-Wert = (1 / n) * Σ (forecastᵢ - outcomeᵢ)²
	 * </pre>
	 *
	 * wobei:
	 * <ul>
	 * <li>n = Anzahl der Vorhersagen</li>
	 * <li>forecastᵢ = Wahrscheinlichkeitsvorhersage für das Ereignis i</li>
	 * <li>outcomeᵢ = tatsächliches Ergebnis des Ereignisses i (1 für eingetreten, 0
	 * für nicht eingetreten)</li>
	 * </ul>
	 * 
	 * @param forecasts Array von Wahrscheinlichkeitsvorhersagen
	 * @param outcomes  Array von tatsächlichen Ergebnissen (1 für Eintreten, 0 für
	 *                  Nichteintreten)
	 * @return Brier-Wert als double
	 */
	public static double berechneBrierScore(double[] forecasts, int[] outcomes) {
		if (forecasts.length != outcomes.length) {
			throw new IllegalArgumentException("Vorhersagen und Ergebnisse müssen die gleiche Länge haben.");
		}

		double totalError = 0.0;
		int n = forecasts.length;

		for (int i = 0; i < n; i++) {
			double error = forecasts[i] - outcomes[i];
			totalError += error * error;
		}
		return totalError / n; // Durchschnittlicher quadratischer Fehler
	}

	/**
	 * Methode zur Ausgabe der Vorhersagen und Ergebnisse als Tabelle
	 * 
	 * @param forecasts  Array mit Wahrscheinlichkeitsvorhersagen
	 * @param outcomes   Array mit tatsächlichen Ergebnissen (1 oder 0)
	 * @param brierScore der berechnete Brier-Wert
	 */
	public static void printResult(double[] forecasts, int[] outcomes, double brierScore) {
		if (forecasts.length != outcomes.length) {
			System.err.println("Fehler: Die Arrays 'forecasts' und 'outcomes' müssen dieselbe Länge haben.");
			return;
		}

		System.out.printf("%-10s %-15s %-10s%n", "Index", "Vorhersage", "Ergebnis");
		System.out.println("-------------------------------------");

		// Durch die Arrays iterieren und die Daten ausgeben
		for (int i = 0; i < forecasts.length; i++) {
			System.out.printf("%-10d %-15.2f %-10d%n", i + 1, forecasts[i], outcomes[i]);
		}

		System.out.println("-------------------------------------");
		System.out.printf("Brier-Wert: %.4f%n", brierScore);
	}
}
