package de.wenzlaff.crypto;

import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PublicKey;
import java.security.spec.ECGenParameterSpec;
import java.util.Arrays;

import javax.crypto.KeyAgreement;

/**
 * Beispiel zu ECDH, siehe auch http://blog.wenzlaff.de/?s=kurven
 * 
 * @author Thomas Wenzlaff
 *
 */
public class ECDH {

	public static void main(String[] args) throws Exception {

		// ein Generator für Schlüsselpaaren erzeugen, der den Algorithmus "EC"
		// (Elliptische Kurven) verwendet
		KeyPairGenerator kpg = KeyPairGenerator.getInstance("EC");
		// "secp256r1" ist der Name einer standardisierten elliptischen Kurve.
		// Diese Kurve ist sehr verbreitet und wird als sicher und effizient angesehen.
		ECGenParameterSpec ecSpec = new ECGenParameterSpec("secp256r1");
		kpg.initialize(ecSpec);

		// Alice erzeugt Schlüsselpaar
		KeyPair aliceKP = kpg.generateKeyPair();
		KeyAgreement aliceKeyAgree = KeyAgreement.getInstance("ECDH");
		System.out.println("\nAlice erzeugter public Key  = " + aliceKP.getPublic());
		System.out.println("\nAlice erzeugter private Key (Hex) = " + bytesToHex(aliceKP.getPrivate().getEncoded()));

		aliceKeyAgree.init(aliceKP.getPrivate());

		// Bob erzeugt Schlüsselpaar
		KeyPair bobKP = kpg.generateKeyPair();
		KeyAgreement bobKeyAgree = KeyAgreement.getInstance("ECDH");
		bobKeyAgree.init(bobKP.getPrivate());
		System.out.println("\nBob erzeugter public Key  = " + bobKP.getPublic());
		System.out.println("\nBob erzeugter private Key (Hex)  = " + bytesToHex(bobKP.getPrivate().getEncoded()));

		// Öffentliche Schlüssel austauschen
		PublicKey alicePubKey = aliceKP.getPublic();
		PublicKey bobPubKey = bobKP.getPublic();

		// Alice berechnet das gemeinsame Geheimnis
		// die elliptische Kurvenoperation durchführen
		aliceKeyAgree.doPhase(bobPubKey, true);
		byte[] aliceSharedSecret = aliceKeyAgree.generateSecret();

		// Bob berechnet das gemeinsame Geheimnis
		// die elliptische Kurvenoperation durchführen
		bobKeyAgree.doPhase(alicePubKey, true);
		byte[] bobSharedSecret = bobKeyAgree.generateSecret();

		// Vergleich: beide Schlüssel sollten gleich sein
		System.out.println("\nSind beide Shared Secrets identisch? " + Arrays.equals(aliceSharedSecret, bobSharedSecret));
		System.out.println("Shared Secret (Hex): " + bytesToHex(aliceSharedSecret));
	}

	// Hilfsmethode für hexadezimale Ausgabe
	private static String bytesToHex(byte[] bytes) {
		StringBuilder sb = new StringBuilder();
		for (byte b : bytes)
			sb.append(String.format("%02x", b));
		return sb.toString();
	}
}
