package de.wenzlaff.crypto;

import java.math.BigInteger;
import java.security.SecureRandom;

/**
 * Hier ist ein einfaches Java-Programm zur Demo, das den
 * Diffie-Hellman-Schlüsselaustausch demonstriert.
 * 
 * Es verwendet die Java-eigene Klasse  BigInteger  für große Zahlen und zeigt,
 * wie Alice und Bob ihre geheimen Schlüssel berechnen und austauschen, um zum
 * gleichen gemeinsamen Schlüssel zu kommen.
 * 
 * Dieses Programm zeigt die einzelnen Schritte des Diffie-Hellman-Verfahrens
 * mit exemplarisch kleinen Zahlen für bessere Verständlichkeit. In der Praxis
 * sollten die Werte von und deutlich größer sein (> 3000 Bit) für Sicherheit.
 * Der geheime Schlüssel ist das Ergebnis, das sowohl Alice als auch Bob
 * berechnen und das gleich ist. Der Wert wird hier nur ausgegeben, aber könnte
 * z.B. als symmetrischer Schlüssel für Verschlüsselung genutzt werden.
 * 
 * @author Thomas Wenzlaff
 *
 */
public class DiffieHellmanKeyTausch {

	public static void main(String[] args) {

		// Öffentliche Parameter: große Primzahl p und Generator g

		// p ist eine große Primzahl, die als Modul für alle Berechnungen dient. Sie
		// definiert die “Ringgröße” für die modularen Potenz-Rechnungen. Die Wahl von
		// ist entscheidend für die Sicherheit, denn sie muss so groß sein, dass der
		// diskrete Logarithmus (also das Auffinden geheimer Exponenten) praktisch nicht
		// lösbar ist. Typischerweise wird eine mit mehreren hundert oder tausend Bit
		// Länge gewählt.
		BigInteger p = new BigInteger("23"); // kleine Primzahl als Beispiel, in der Praxis viel größer

		// g ist eine sogenannte Primitivwurzel modulo, auch Generator genannt. Das
		// bedeutet, dass die Potenzen von modulo alle Zahlen von 1 bis p -1
		// erzeugen können.
		// Es ist also ein Basiswert, mit dem die geheimen Zahlen potenziert werden, um
		// öffentliche Schlüssel zu generieren.
		BigInteger g = new BigInteger("5"); // Generator (Basiswert für Potenzierung modulo )

		SecureRandom random = new SecureRandom();

		// Alice wählt ihre geheime Zahl a
		BigInteger a = new BigInteger(5, random); // zufällige Zahl kleiner 2^5 (als Beispiel)
		// Bob wählt seine geheime Zahl b
		BigInteger b = new BigInteger(5, random);

		System.out.println("Öffentliche Parameter:");
		System.out.println("p = " + p);
		System.out.println("g = " + g);

		System.out.println("\nAlice wählt geheime Zahl a = " + a);
		System.out.println("  Bob wählt geheime Zahl b = " + b);

		// Alice berechnet A = g^a mod p
		BigInteger A = g.modPow(a, p); // berechnet den modularen Exponent
		// Bob berechnet B = g^b mod p
		BigInteger B = g.modPow(b, p);

		System.out.println("\nAlice berechnet und sendet A = g^a mod p = " + g + "^" + a + " mod " + p + " = " + A);
		System.out.println("  Bob berechnet und sendet B = g^b mod p = " + g + "^" + b + " mod " + p + " = " + B);

		// Alice berechnet geheimen Schlüssel K = B^a mod p
		BigInteger keyAlice = B.modPow(a, p);
		// Bob berechnet geheimen Schlüssel K = A^b mod p
		BigInteger keyBob = A.modPow(b, p);

		System.out.println("\nAlice berechnet gemeinsamen Schlüssel: " + keyAlice);
		System.out.println("  Bob berechnet gemeinsamen Schlüssel: " + keyBob);

		// Prüfen, dass beide Schlüssel gleich sind
		System.out.println("\nSchlüssel stimmen überein: " + keyAlice.equals(keyBob));
	}
}
