package blended.updater.config

import com.typesafe.config.{Config, ConfigFactory}

import scala.collection.JavaConverters._
import scala.collection.immutable.Map
import scala.util.{Left, Right, Try}

/**
 * Helper for [[OverlayConfig]] containing common useful operations.
 */
final object OverlayConfigCompanion {

  object Properties {
    val JVM_MAX_MEM = "blended.launcher.jvm.xmx"
    val JVM_USE_MEM = "blended.launcher.jvm.xms"
  }

  def findCollisions(generatedConfigs: Seq[GeneratedConfig]): Seq[String] = {
    aggregateGeneratedConfigs(generatedConfigs) match {
      case Left(issues) => issues
      case _ => Nil
    }
  }

  def aggregateGeneratedConfigs(generatedConfigs: Iterable[GeneratedConfig]): Either[Seq[String], Map[String, Map[String, Object]]] = {
    // seen configurations per target file
    var fileToConfig: Map[String, Map[String, Object]] = Map()
    val issues = generatedConfigs.flatMap { gc =>
      val newConfig = GeneratedConfigCompanion.config(gc).root().unwrapped().asScala.toMap
      fileToConfig.get(gc.configFile) match {
        case None =>
          // no collision
          fileToConfig += gc.configFile -> newConfig
          Seq()
        case Some(existingConfig) =>
          // check collisions
          val collisions = existingConfig.keySet.intersect(newConfig.keySet)
          fileToConfig += gc.configFile -> (existingConfig ++ newConfig)
          collisions.map(c => s"Double defined config key found: ${c}")
      }
    }
    if (issues.isEmpty) Right(fileToConfig) else Left(issues.toList)
  }


  def read(config: Config): Try[OverlayConfig] = Try {

    def configAsMap(key: String, default: Option[() => Map[String, String]] = None): Map[String, String] =
      if (default.isDefined && !config.hasPath(key)) {
        default.get.apply()
      } else {
        config.getConfig(key).entrySet().asScala.map {
          entry => entry.getKey() -> entry.getValue().unwrapped().asInstanceOf[String]
        }.toMap
      }

    OverlayConfig(
      name = config.getString("name"),
      version = config.getString("version"),
      generatedConfigs = if (config.hasPath("configGenerator")) {
        config.getObjectList("configGenerator").asScala.map { gen =>
          val genConf = gen.toConfig()
          val fileName = genConf.getString("file")
          val genConfig = genConf.getObject("config").toConfig()
          GeneratedConfigCompanion.create(fileName, genConfig)
        }.toList
      } else Nil,
      properties = configAsMap("properties", Some(() => Map()))
    )
  }

  def toConfig(overlayConfig: OverlayConfig): Config = {
    val config = Map(
      "name" -> overlayConfig.name,
      "version" -> overlayConfig.version,
      "configGenerator" -> overlayConfig.generatedConfigs.map { genConfig =>
        Map(
          "file" -> genConfig.configFile,
          "config" -> GeneratedConfigCompanion.config(genConfig).root().unwrapped()
        ).asJava
      }.asJava,
      "properties" -> overlayConfig.properties.asJava
    ).asJava
    ConfigFactory.parseMap(config)
  }

}
