package blended.streams.dispatcher.internal.builder

import akka.NotUsed
import akka.stream.scaladsl.{Broadcast, Flow, GraphDSL, Merge}
import akka.stream.{FanOutShape2, Graph}
import blended.container.context.api.ContainerContext
import blended.streams.FlowProcessor
import blended.streams.dispatcher.internal._
import blended.streams.jms.JmsEnvelopeHeader
import blended.streams.message.{FlowEnvelope, FlowEnvelopeLogger}
import blended.streams.worklist.{WorklistEvent, WorklistStarted}
import blended.util.logging.LogLevel
import blended.util.RichTry._

import scala.concurrent.duration._
import scala.util.{Failure, Success, Try}

case class DispatcherFanout(
  dispatcherCfg : ResourceTypeRouterConfig,
  ctCtxt : ContainerContext,
  streamLogger : FlowEnvelopeLogger
)(implicit bs : DispatcherBuilderSupport) extends JmsEnvelopeHeader {

  /*-------------------------------------------------------------------------------------------------*/
  private[builder] val funFanoutOutbound : FlowEnvelope => Try[Seq[(OutboundRouteConfig, FlowEnvelope)]] = { env =>

    bs.withContextObject[ResourceTypeConfig, Seq[(OutboundRouteConfig, FlowEnvelope)]](bs.rtConfigKey, env, streamLogger) { rtCfg: ResourceTypeConfig =>
      Try {
        rtCfg.outbound.map { ob =>
          val obEnv =
            env
              .withContextObject(bs.outboundCfgKey, ob)
              .withHeader(bs.headerConfig.headerBranch, ob.id).unwrap
          (ob, outboundMsg(ob)(obEnv).unwrap)
        }
      }
    } match {
      case Right(s) => Success(s)
      case Left(t) =>
        t.exception match {
          case None => streamLogger.logEnv(env, LogLevel.Error, s"Exception in fan out step [${env.id}]")
          case Some(t) => streamLogger.logEnv(env.withException(t), LogLevel.Error, s"Exception in fan out step [${env.id}]")
        }
        Failure(t.exception.getOrElse(new Exception("Unexpected exception")))
    }
  }

  private[builder] lazy val fanoutOutbound =
    FlowProcessor.transform[Seq[(OutboundRouteConfig, FlowEnvelope)]]("fanoutOutbound", streamLogger)(funFanoutOutbound)

  /*-------------------------------------------------------------------------------------------------*/
  private lazy val outboundMsg : OutboundRouteConfig => FlowEnvelope => Try[FlowEnvelope] = { outCfg => env =>

    val useHeaderBlock : OutboundHeaderConfig => Try[Boolean] = { oh =>
      Try {
        oh.condition match {
          // if the block does not have a condition, the header block will be used
          case None => true
          case Some(c) =>
            val resolve = ctCtxt.resolveString(c, env.flowMessage.header.view.mapValues(_.value).toMap)
            streamLogger.logEnv(env, LogLevel.Debug, s"Resolved condition to [$resolve][${resolve.map(_.getClass().getName())}]")
            val use = resolve.map(_.asInstanceOf[Boolean]).unwrap

            val s = s"using header for [${env.id}]:[outboundMsg] block with expression [$c]"
            if (use) {
              streamLogger.logEnv(env, LogLevel.Debug, s)
            } else {
              streamLogger.logEnv(env, LogLevel.Debug, "Not " + s )
            }
            use
        }
      }
    }

    Try {

      outCfg.outboundHeader.filter(b => useHeaderBlock(b).unwrap).foldLeft(env) {
        case (current, oh) =>
          var newEnv : FlowEnvelope = current
            .withHeader(bs.headerConfig.headerMaxRetries, oh.maxRetries).unwrap
            .withHeader(bs.headerAutoComplete, oh.autoComplete).unwrap

          if (oh.timeToLive >= 0L) {
            newEnv = newEnv.withHeader(bs.headerTimeToLive, oh.timeToLive).unwrap
          } else {
            newEnv = newEnv.removeHeader(bs.headerTimeToLive)
          }

          newEnv = newEnv
            .withHeader(deliveryModeHeader(bs.headerConfig.prefix), oh.deliveryMode).unwrap

        oh.header.foreach { case (header, value) =>
          val resolved = ctCtxt.resolveString(value, env.flowMessage.header.view.mapValues(_.value).toMap).get
          streamLogger.logEnv(env, LogLevel.Trace,s"[${newEnv.id}]:[${outCfg.id}] - resolved property [$header] to [$resolved]")
          newEnv = newEnv.withHeader(header, resolved).unwrap
        }

          newEnv = if (oh.clearBody) {
            newEnv.copy(flowMessage = newEnv.flowMessage.clearBody())
          } else {
            newEnv
          }

          newEnv
            .withContextObject(bs.appHeaderKey, oh.applicationLogHeader)
            .withContextObject(bs.bridgeProviderKey, oh.bridgeProviderConfig)
            .withContextObject(bs.bridgeDestinationKey, oh.bridgeDestination)
      }
    }
  }

  private[builder] val toWorklist : Seq[(OutboundRouteConfig, FlowEnvelope)] => WorklistEvent = envelopes => {

    val timeout = envelopes.head._2.getFromContext[ResourceTypeConfig](bs.rtConfigKey) match {
      case Success(c) => c.map(_.timeout).getOrElse(10.seconds)
      case Failure(_) => 10.seconds
    }

    val wl = WorklistStarted(
      worklist = bs.worklist(envelopes.map(_._2) : _*).get,
      timeout = timeout
    )

    streamLogger.underlying.debug(s"Created worklist event [$wl]")
    wl
  }

  private lazy val decideRouting = DispatcherOutbound(dispatcherCfg, ctCtxt, streamLogger)

  def build() : Graph[FanOutShape2[FlowEnvelope, FlowEnvelope, WorklistEvent], NotUsed] = {
    GraphDSL.create() { implicit builder =>
      import GraphDSL.Implicits._

      val fanout = builder.add(fanoutOutbound)

      val errorFilter = builder.add(Broadcast[Either[FlowEnvelope, Seq[(OutboundRouteConfig, FlowEnvelope)]]](2))
      val withError = builder.add(Flow[Either[FlowEnvelope, Seq[(OutboundRouteConfig, FlowEnvelope)]]].collect{ case Left(l) => l })
      val noError = builder.add(Flow[Either[FlowEnvelope, Seq[(OutboundRouteConfig, FlowEnvelope)]]].collect{ case Right(r) => r })
      val mapDestination = builder.add(decideRouting)

      val createWorklist = builder.add(Broadcast[Seq[(OutboundRouteConfig, FlowEnvelope)]](2))

      val envelopes = builder.add(Flow[Seq[(OutboundRouteConfig, FlowEnvelope)]].mapConcat(_.toList).map(_._2))
      val worklist = builder.add(Flow[Seq[(OutboundRouteConfig, FlowEnvelope)]].map(toWorklist))

      val merge = builder.add(Merge[FlowEnvelope](2))

      fanout ~> errorFilter ~> withError ~> merge
      errorFilter ~> noError ~> createWorklist.in

      createWorklist.out(0) ~> envelopes ~> mapDestination ~> merge
      createWorklist.out(1) ~> worklist

      new FanOutShape2(
        fanout.in,
        merge.out,
        worklist.out
      )
    }
  }
}
