package blended.jms.utils

import blended.container.context.ContainerIdentifierService
import blended.jms.utils.ConnectionFactoryActivator.{CF_JNDI_NAME, DEFAULT_PWD, DEFAULT_USER, USE_JNDI}
import com.typesafe.config.Config

import scala.collection.JavaConverters._
import scala.util.{Failure, Success, Try}

object BlendedJMSConnectionConfig {

  val defaultConfig = BlendedJMSConnectionConfig(
    vendor = "",
    provider = "",
    enabled = true,
    jmxEnabled = true,
    pingTolerance = 5,
    pingInterval = 30,
    pingTimeout = 3,
    retryInterval = 5,
    minReconnect = 300,
    maxReconnectTimeout = -1,
    clientId = "$[[" + ContainerIdentifierService.containerId + "]]",
    defaultUser = None,
    defaultPassword  = None,
    pingDestination = "blended.ping",
    properties = Map.empty,
    useJndi = false,
    jndiName = None,
    cfEnabled = None,
    cfClassName = None,
    ctxtClassName = None,
    jmsClassloader  = None
  )

  def fromConfig(stringResolver : String => Try[String])(vendor: String, provider: Option[String], cfg: Config) : BlendedJMSConnectionConfig = {
    val prov = if (cfg.hasPath("provider")) cfg.getString("provider") else provider.getOrElse(defaultConfig.provider)
    val enabled = !cfg.hasPath("enabled") || cfg.getBoolean("enabled")
    val jmxEnabled = if (cfg.hasPath("jmxEnabled")) cfg.getBoolean("jmxEnabled") else defaultConfig.jmxEnabled
    val pingTolerance = if (cfg.hasPath("pingTolerance")) cfg.getInt("pingTolerance") else defaultConfig.pingTolerance
    val pingInterval = if (cfg.hasPath("pingInterval")) cfg.getInt("pingInterval") else defaultConfig.pingInterval
    val pingTimeout = if (cfg.hasPath("pingTimeout")) cfg.getInt("pingTimeout") else defaultConfig.pingTimeout
    val retryInterval = if (cfg.hasPath("retryInterval")) cfg.getInt("retryInterval") else defaultConfig.retryInterval
    val minReconnect = if (cfg.hasPath("minReconnect")) cfg.getInt("minReconnect") else defaultConfig.minReconnect
    val maxReconnectTimeout = if (cfg.hasPath("maxReconnectTimeout")) cfg.getInt("maxReconnectTimeout") else defaultConfig.maxReconnectTimeout
    val clientId = if (cfg.hasPath("clientId"))
      stringResolver(cfg.getString("clientId")) match {
        case Failure(t) => throw t
        case Success(id) => id
      }
    else
      defaultConfig.clientId
    val defaultUser = if (cfg.hasPath(DEFAULT_USER)) Some(cfg.getString(DEFAULT_USER)) else defaultConfig.defaultUser
    val defaultPasswd = if (cfg.hasPath(DEFAULT_PWD)) Some(cfg.getString(DEFAULT_PWD)) else defaultConfig.defaultPassword
    val destination = if (cfg.hasPath("destination")) cfg.getString("destination") else defaultConfig.pingDestination
    val properties : Map[String, String] = if (cfg.hasPath("properties")) {
      val resolved = cfg.getConfig("properties").entrySet().asScala.map{ e =>
        (e.getKey(), stringResolver(cfg.getConfig("properties").getString(e.getKey())))
      }.toMap

      resolved.find(_._2.isFailure).map(_._2.failed.get).map(throw _)

      resolved.map( p => p._1 -> p._2.get)
    } else defaultConfig.properties
    val jndiName = if (cfg.hasPath(CF_JNDI_NAME)) Some(cfg.getString(CF_JNDI_NAME)) else defaultConfig.jndiName
    val useJndi = if (cfg.hasPath(USE_JNDI)) cfg.getBoolean(USE_JNDI) else defaultConfig.useJndi

    BlendedJMSConnectionConfig(
      vendor = vendor,
      enabled = enabled,
      provider = prov,
      jmxEnabled = jmxEnabled,
      pingTolerance = pingTolerance,
      pingInterval = pingInterval,
      pingTimeout = pingTimeout,
      retryInterval = retryInterval,
      minReconnect = minReconnect,
      maxReconnectTimeout = maxReconnectTimeout,
      clientId = clientId,
      defaultUser = defaultUser,
      defaultPassword = defaultPasswd,
      pingDestination = destination,
      properties = properties,
      jndiName = jndiName,
      useJndi = useJndi,
      cfEnabled = None,
      jmsClassloader = defaultConfig.jmsClassloader,
      ctxtClassName = defaultConfig.ctxtClassName,
      cfClassName = defaultConfig.cfClassName
    )
  }
}

case class BlendedJMSConnectionConfig(
  vendor : String,
  provider : String,
  enabled : Boolean,
  jmxEnabled : Boolean,
  pingTolerance : Int,
  pingInterval : Int,
  pingTimeout : Int,
  retryInterval : Int,
  minReconnect : Int,
  maxReconnectTimeout: Int,
  clientId : String,
  defaultUser : Option[String],
  defaultPassword : Option[String],
  pingDestination : String,
  properties : Map[String, String],
  useJndi : Boolean,
  jndiName : Option[String] = None,
  cfEnabled : Option[BlendedJMSConnectionConfig => Boolean],
  cfClassName: Option[String],
  ctxtClassName : Option[String],
  jmsClassloader : Option[ClassLoader]
)
