package de.voize.reaktnativetoolkit.ksp.processor

import com.google.devtools.ksp.KspExperimental
import com.google.devtools.ksp.getAnnotationsByType
import com.google.devtools.ksp.processing.CodeGenerator
import com.google.devtools.ksp.processing.KSPLogger
import com.google.devtools.ksp.symbol.ClassKind
import com.google.devtools.ksp.symbol.KSAnnotation
import com.google.devtools.ksp.symbol.KSClassDeclaration
import com.google.devtools.ksp.symbol.KSDeclaration
import com.google.devtools.ksp.symbol.KSFile
import com.google.devtools.ksp.symbol.KSFunctionDeclaration
import com.google.devtools.ksp.symbol.KSPropertyDeclaration
import com.google.devtools.ksp.symbol.KSType
import com.google.devtools.ksp.symbol.KSTypeAlias
import com.google.devtools.ksp.symbol.KSTypeParameter
import com.google.devtools.ksp.symbol.KSTypeReference
import io.outfoxx.typescriptpoet.CodeBlock
import io.outfoxx.typescriptpoet.CodeBlock.Companion.joinToCode
import io.outfoxx.typescriptpoet.EnumSpec
import io.outfoxx.typescriptpoet.FileSpec
import io.outfoxx.typescriptpoet.FunctionSpec
import io.outfoxx.typescriptpoet.InterfaceSpec
import io.outfoxx.typescriptpoet.Modifier
import io.outfoxx.typescriptpoet.ModuleSpec
import io.outfoxx.typescriptpoet.NameAllocator
import io.outfoxx.typescriptpoet.ParameterSpec
import io.outfoxx.typescriptpoet.PropertySpec
import io.outfoxx.typescriptpoet.SymbolSpec
import io.outfoxx.typescriptpoet.TypeAliasSpec
import io.outfoxx.typescriptpoet.TypeName
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import java.util.UUID

internal class TypescriptModelsGenerator(
    private val codeGenerator: CodeGenerator,
    private val config: TypescriptConfig,
    private val logger: KSPLogger,
) {
    private fun getTypescriptTypeName(
        ksType: KSType,
        annotations: Sequence<KSAnnotation>? = null,
    ) = getTypescriptTypeName(
        ksType,
        annotations,
        config.externalTypeMapping,
        config.defaultInstantJSType,
        logger,
    )

    internal fun createTypescriptModelsFile(
        rootNamespace: TypescriptModelsNamespaceTree.NamespaceNode,
        originatingKSFiles: List<KSFile>,
    ) {
        val typescriptModelsFileBuilder = FileSpec.builder("$generatedTsFilePath$modelsModule")
        typescriptModelsFileBuilder.addComment("This file is generated by ReaktNativeToolkit. Do not edit.")
        val rootTypeDeclarationsModuleSpec =
            createTypescriptNamespaceWithTypeDeclarations(rootNamespace)
        rootTypeDeclarationsModuleSpec.members.forEach {
            when (it) {
                is InterfaceSpec -> {
                    typescriptModelsFileBuilder.addInterface(it)
                }

                is EnumSpec -> {
                    typescriptModelsFileBuilder.addEnum(it)
                }

                is TypeAliasSpec -> {
                    typescriptModelsFileBuilder.addTypeAlias(it)
                }

                is ModuleSpec -> {
                    typescriptModelsFileBuilder.addModule(it)
                }

                is FunctionSpec -> {
                    typescriptModelsFileBuilder.addFunction(it)
                }

                else -> error("Unsupported member ${it::class.simpleName}")
            }
        }
        val typescriptModelsFile = typescriptModelsFileBuilder.build()
        typescriptModelsFile.writeTo(codeGenerator, kspDependencies(true, originatingKSFiles))
    }

    private fun createTypescriptNamespaceWithTypeDeclarations(namespace: TypescriptModelsNamespaceTree.NamespaceNode): ModuleSpec {
        return ModuleSpec.builder(namespace.name)
            .apply {
                addModifier(Modifier.EXPORT)
                namespace.declarations.forEach { declaration ->
                    createTypescriptTypeDeclaration(declaration, this)
                    createTypescriptTypeMapping(declaration, this)
                }
                namespace.children.map(::createTypescriptNamespaceWithTypeDeclarations).forEach {
                    addModule(it)
                }
            }.build()
    }

    private fun KSClassDeclaration.getSealedSubclassTypeEnumSymbol(sealedClassDeclaration: KSDeclaration): SymbolSpec {
        val typeEnumName = sealedClassDeclaration.getTypescriptNameWithNamespace() + "Type"
        val typeEnumTypeSymbol = getSymbol(typeEnumName, config.externalTypeMapping)
        return typeEnumTypeSymbol.nested(getTypescriptName())
    }

    private fun KSClassDeclaration.getSealedTypescriptBaseType(): TypeName? {
        val sealedClassDeclaration = getSealedSuperclass()
        return if (sealedClassDeclaration != null) {
            val baseTypeName = sealedClassDeclaration.getTypescriptNameWithNamespace() + "Base"
            val baseTypeTypeName = getTypeName(baseTypeName, config.externalTypeMapping)
            baseTypeTypeName.parameterized(
                TypeName.standard(getSealedSubclassTypeEnumSymbol(sealedClassDeclaration))
            )
        } else null
    }

    private fun createTypescriptTypeDeclaration(
        declaration: KSDeclaration,
        typescriptFileBuilder: ModuleSpec.Builder,
    ) {
        when (declaration.qualifiedName) {
            else -> when (declaration) {
                is KSClassDeclaration -> {
                    val sealedBaseType = declaration.getSealedTypescriptBaseType()

                    when (declaration.classKind) {
                        ClassKind.INTERFACE -> error("Interfaces are not supported")
                        ClassKind.CLASS -> {
                            if (com.google.devtools.ksp.symbol.Modifier.DATA in declaration.modifiers) {
                                // data class
                                declaration.assertSerializable {
                                    "Data classes must be annotated with @Serializable: $declaration"
                                }
                                val interfaceSpec =
                                    InterfaceSpec.builder(declaration.getTypescriptName()).apply {
                                        addModifiers(Modifier.EXPORT)
                                        if (sealedBaseType != null) {
                                            addSuperInterface(sealedBaseType)
                                        }
                                        addTSDoc(
                                            "Data class generated from {@link %N}\n",
                                            declaration.qualifiedName!!.asString()
                                        )
                                        addProperties(
                                            declaration.getAllProperties()
                                                .map { toTypescriptPropertySpec(it) }.toList()
                                        )
                                    }.build()
                                typescriptFileBuilder.addInterface(interfaceSpec)
                            } else if (com.google.devtools.ksp.symbol.Modifier.SEALED in declaration.modifiers) {
                                if (sealedBaseType != null) {
                                    error("Sealed classes as direct members of other sealed classes are not supported: $declaration")
                                }
                                declaration.assertSerializable {
                                    "Sealed classes must be annotated with @Serializable: $declaration"
                                }
                                val subclasses = declaration.getSealedSubclasses()
                                val subclassesToDiscriminator =
                                    subclasses.associateWith { it.getSealedSubclassDiscriminatorValue() }
                                val typeEnumName = declaration.getTypescriptName() + "Type"
                                val typeEnumTypeName = getTypeName(typeEnumName, config.externalTypeMapping)
                                EnumSpec.builder(typeEnumName).apply {
                                    addModifiers(Modifier.EXPORT)
                                    subclassesToDiscriminator.forEach { (subclassDeclaration, discriminator) ->
                                        addConstant(
                                            subclassDeclaration.getTypescriptName(),
                                            CodeBlock.of("%S", discriminator)
                                        )
                                    }
                                }.build().let {
                                    typescriptFileBuilder.addEnum(it)
                                }
                                val baseTypeName = declaration.getTypescriptName() + "Base"

                                val discriminatorKey =
                                    declaration.getDiscriminatorKeyForSealedClass()

                                InterfaceSpec.builder(baseTypeName).apply {
                                    val typeVariable =
                                        TypeName.typeVariable(
                                            "T",
                                            TypeName.bound(typeEnumTypeName)
                                        )
                                    addTypeVariable(typeVariable)
                                    addProperty(
                                        PropertySpec.builder(
                                            discriminatorKey,
                                            typeVariable,
                                        ).build()
                                    )
                                }.build().let {
                                    typescriptFileBuilder.addInterface(it)
                                }

                                val sealedTypeUnion = TypeAliasSpec.builder(
                                    declaration.getTypescriptName(),
                                    TypeName.unionType(*subclasses.map { getTypeName(it.getTypescriptNameWithNamespace(), config.externalTypeMapping) }
                                        .toList().toTypedArray())
                                ).addModifiers(Modifier.EXPORT)
                                    .addTSDoc(
                                        "Sealed class generated from {@link %N}\n",
                                        declaration.qualifiedName!!.asString()
                                    ).build()
                                typescriptFileBuilder.addTypeAlias(sealedTypeUnion)
                            } else {
                                error("Only data classes and sealed classes are supported, found: $declaration")
                            }
                        }

                        ClassKind.ENUM_CLASS -> {
                            declaration.assertSerializable {
                                "Enums must be annotated with @Serializable: $declaration"
                            }
                            val enumSpec = EnumSpec.builder(declaration.getTypescriptName()).apply {
                                addModifiers(Modifier.EXPORT)
                                addTSDoc(
                                    "Enum generated from {@link %N}\n",
                                    declaration.qualifiedName!!.asString()
                                )
                                declaration.declarations.filterIsInstance<KSClassDeclaration>()
                                    .filter { it.classKind == ClassKind.ENUM_ENTRY }
                                    .forEach { enumEntry ->
                                        val name = enumEntry.simpleName.asString()
                                        addConstant(name, CodeBlock.of("%S", name))
                                    }
                            }.build()
                            typescriptFileBuilder.addEnum(enumSpec)
                        }

                        ClassKind.ENUM_ENTRY -> error("Enum entries are not supported")
                        ClassKind.OBJECT -> {
                            declaration.assertSerializable {
                                "Objects must be annotated with @Serializable: $declaration"
                            }
                            val interfaceSpec =
                                InterfaceSpec.builder(declaration.getTypescriptName()).apply {
                                    addModifiers(Modifier.EXPORT)
                                    addTSDoc(
                                        "Object generated from {@link %N}\n",
                                        declaration.qualifiedName!!.asString()
                                    )
                                    if (sealedBaseType != null) {
                                        addSuperInterface(sealedBaseType)
                                    }
                                }.build()
                            typescriptFileBuilder.addInterface(interfaceSpec)
                        }

                        ClassKind.ANNOTATION_CLASS -> error("Annotation classes are not supported")
                    }
                }

                is KSFunctionDeclaration -> {
                    error("Function declarations are not supported")
                }

                is KSTypeAlias -> {
                    val type = declaration.type.resolve()
                    val typeAliasSpec = TypeAliasSpec.builder(
                        declaration.getTypescriptName(),
                        getTypescriptTypeName(type)
                    )
                        .addModifiers(Modifier.EXPORT)
                        .addTSDoc(
                            "Type alias generated from {@link %N}\n",
                            declaration.qualifiedName!!.asString()
                        )
                        .build()

                    typescriptFileBuilder.addTypeAlias(typeAliasSpec)
                }

                is KSPropertyDeclaration -> {
                    error("Property declarations are not supported")
                }

                is KSTypeParameter -> {
                    error("Type parameter declarations are not supported")
                }

                else -> {
                    error("Unsupported declaration: $declaration")
                }
            }
        }
    }

    /**
     * Create js mappings from and to json for a given declaration.
     */
    private fun createTypescriptTypeMapping(
        declaration: KSDeclaration,
        typescriptFileBuilder: ModuleSpec.Builder,
    ) {
        when (declaration.qualifiedName) {
            else -> when (declaration) {
                is KSClassDeclaration -> {
                    when (declaration.classKind) {
                        ClassKind.INTERFACE -> error("Interfaces are not supported")
                        ClassKind.CLASS -> {
                            if (com.google.devtools.ksp.symbol.Modifier.DATA in declaration.modifiers) {
                                typescriptFileBuilder.createTypescriptTypeMappingForDataClass(
                                    declaration
                                )
                            } else if (com.google.devtools.ksp.symbol.Modifier.SEALED in declaration.modifiers) {
                                typescriptFileBuilder.createTypescriptTypeMappingForSealedClass(
                                    declaration
                                )
                            } else {
                                error("Only data classes and sealed classes are supported, found: $declaration")
                            }
                        }

                        ClassKind.ENUM_CLASS -> {
                            typescriptFileBuilder.createTypescriptTypeMappingForEnumClass(
                                declaration
                            )
                        }

                        ClassKind.ENUM_ENTRY -> error("Enum entries are not supported")
                        ClassKind.OBJECT -> {
                            typescriptFileBuilder.createTypescriptTypeMappingForObject(declaration)
                        }

                        ClassKind.ANNOTATION_CLASS -> error("Annotation classes are not supported")
                    }
                }

                is KSFunctionDeclaration -> {
                    error("Function declarations are not supported")
                }

                is KSTypeAlias -> {
                    typescriptFileBuilder.createTypescriptTypeMappingForTypeAlias(declaration)
                }

                is KSPropertyDeclaration -> {
                    error("Property declarations are not supported")
                }

                is KSTypeParameter -> {
                    error("Type parameter declarations are not supported")
                }

                else -> {
                    error("Unsupported declaration: $declaration")
                }
            }
        }
    }

    private fun ModuleSpec.Builder.createTypescriptTypeMappingForTypeAlias(
        declaration: KSTypeAlias
    ) {
        val type = declaration.type.resolve()

        val fromJson = FunctionSpec.builder(declaration.getTypescriptFromJsonFunctionName())
            .apply {
                val nameAllocator = NameAllocator()
                addModifiers(Modifier.EXPORT)
                addTSDoc(
                    "Mapping generated from {@link %N}\n",
                    declaration.qualifiedName!!.asString()
                )
                val jsonParameterTag = UUID.randomUUID()
                nameAllocator.newName("json", jsonParameterTag)

                addParameter(
                    ParameterSpec.builder(
                        nameAllocator[jsonParameterTag],
                        TypeName.ANY,
                    ).build()
                )
                addCode(
                    returnStatement(
                        convertJsonToType(
                            nameAllocator[jsonParameterTag].asCodeBlock(),
                            type,
                            nameAllocator.copy(),
                            config.externalTypeMapping,
                            config.defaultInstantJSType,
                            logger,
                        )
                    )
                )
                returns(getTypescriptTypeName(type))
            }.build()

        val toJson = FunctionSpec.builder(declaration.getTypescriptToJsonFunctionName())
            .apply {
                val nameAllocator = NameAllocator()
                addModifiers(Modifier.EXPORT)
                addTSDoc(
                    "Mapping generated from {@link %N}\n",
                    declaration.qualifiedName!!.asString()
                )
                val valueParameterTag = UUID.randomUUID()
                nameAllocator.newName("value", valueParameterTag)
                addParameter(
                    ParameterSpec.builder(
                        nameAllocator[valueParameterTag],
                        getTypescriptTypeName(type),
                    ).build()
                )
                addCode(
                    returnStatement(
                        convertTypeToJson(
                            nameAllocator[valueParameterTag].asCodeBlock(),
                            type,
                            nameAllocator.copy(),
                            config.externalTypeMapping,
                            config.defaultInstantJSType,
                            logger,
                        )
                    )
                )
                returns(recordType(TypeName.STRING, TypeName.ANY))
            }.build()
        addFunction(fromJson)
        addFunction(toJson)
    }

    private fun ModuleSpec.Builder.createTypescriptTypeMappingForEnumClass(
        declaration: KSClassDeclaration
    ) {
        val fromJson = FunctionSpec.builder(declaration.getTypescriptFromJsonFunctionName())
            .apply {
                val returnType = getTypescriptTypeName(declaration.asStarProjectedType())

                addModifiers(Modifier.EXPORT)
                addTSDoc(
                    "Mapping generated from {@link %N}\n",
                    declaration.qualifiedName!!.asString()
                )
                val jsonParameterName = "json"
                addParameter(
                    ParameterSpec.builder(
                        jsonParameterName,
                        TypeName.STRING,
                    ).build()
                )
                addCode(
                    returnStatement(
                        jsonParameterName.asCodeBlock().castTo(returnType)
                    )
                )
                returns(returnType)
            }.build()

        val toJson = FunctionSpec.builder(declaration.getTypescriptToJsonFunctionName())
            .apply {
                addModifiers(Modifier.EXPORT)
                addTSDoc(
                    "Mapping generated from {@link %N}\n",
                    declaration.qualifiedName!!.asString()
                )
                val valueParameterName = "value"
                addParameter(
                    ParameterSpec.builder(
                        valueParameterName,
                        getTypescriptTypeName(declaration.asStarProjectedType()),
                    ).build()
                )
                addCode(
                    returnStatement(
                        valueParameterName.asCodeBlock()
                    )
                )
                returns(TypeName.STRING)
            }.build()
        addFunction(fromJson)
        addFunction(toJson)
    }

    private fun ModuleSpec.Builder.createTypescriptTypeMappingForObject(
        declaration: KSClassDeclaration
    ) {
        val fromJson =
            FunctionSpec.builder(declaration.getTypescriptFromJsonFunctionName())
                .apply {
                    addModifiers(Modifier.EXPORT)
                    addTSDoc(
                        "Mapping generated from {@link %N}\n",
                        declaration.qualifiedName!!.asString()
                    )
                    val jsonParameterName = "_"
                    addParameter(
                        ParameterSpec.builder(
                            jsonParameterName,
                            recordType(TypeName.STRING, TypeName.ANY),
                        ).build()
                    )
                    addCode(
                        returnStatement(
                            CodeBlock.of("{}")
                        )
                    )
                    returns(getTypescriptTypeName(declaration.asStarProjectedType()))
                }.build()

        val toJson =
            FunctionSpec.builder(declaration.getTypescriptToJsonFunctionName())
                .apply {
                    addModifiers(Modifier.EXPORT)
                    addTSDoc(
                        "Mapping generated from {@link %N}\n",
                        declaration.qualifiedName!!.asString()
                    )
                    val valueParameterName = "_"
                    addParameter(
                        ParameterSpec.builder(
                            valueParameterName,
                            getTypescriptTypeName(declaration.asStarProjectedType()),
                        ).build()
                    )
                    addCode(
                        returnStatement(
                            CodeBlock.of("{}")
                        )
                    )
                    returns(recordType(TypeName.STRING, TypeName.ANY))
                }.build()
        addFunction(fromJson)
        addFunction(toJson)
    }

    private fun ModuleSpec.Builder.createTypescriptTypeMappingForDataClass(
        declaration: KSClassDeclaration
    ) {
        val fromJson = FunctionSpec.builder(declaration.getTypescriptFromJsonFunctionName())
            .apply {
                val nameAllocator = NameAllocator()

                addModifiers(Modifier.EXPORT)
                addTSDoc(
                    "Mapping generated from {@link %N}\n",
                    declaration.qualifiedName!!.asString()
                )
                val jsonParameterTag = "json"
                nameAllocator.newName("json", jsonParameterTag)
                addParameter(
                    ParameterSpec.builder(
                        nameAllocator[jsonParameterTag],
                        recordType(TypeName.STRING, TypeName.ANY),
                    ).build()
                )
                addCode(
                    codeBlock(brackets = false) {
                        add(
                            returnStatement(
                                CodeBlock.of(
                                    "{%>%L%<}",
                                    declaration.getAllProperties()
                                        .map {
                                            val name = it.getJSName()
                                            property(
                                                name,
                                                convertJsonToType(
                                                    CodeBlock.of(
                                                        "%N[%S]",
                                                        nameAllocator[jsonParameterTag],
                                                        name,
                                                    ),
                                                    it.type.resolve(),
                                                    nameAllocator.copy(),
                                                    config.externalTypeMapping,
                                                    config.defaultInstantJSType,
                                                    logger,
                                                )
                                            )
                                        }.toList().joinToCode(
                                            separator = ",\n",
                                            prefix = "\n",
                                            suffix = "\n"
                                        ),
                                )
                            )
                        )
                    }
                )
                returns(getTypescriptTypeName(declaration.asStarProjectedType()))
            }.build()

        val toJson = FunctionSpec.builder(declaration.getTypescriptToJsonFunctionName())
            .apply {
                val nameAllocator = NameAllocator()
                addModifiers(Modifier.EXPORT)
                addTSDoc(
                    "Mapping generated from {@link %N}\n",
                    declaration.qualifiedName!!.asString()
                )
                val valueParameterTag = "value"
                nameAllocator.newName("value", valueParameterTag)
                addParameter(
                    ParameterSpec.builder(
                        nameAllocator[valueParameterTag],
                        getTypescriptTypeName(declaration.asStarProjectedType()),
                    ).build()
                )
                addCode(
                    codeBlock(brackets = false) {
                        add(
                            returnStatement(
                                CodeBlock.of(
                                    "{%>%L%<}",
                                    declaration.getAllProperties()
                                        .map {
                                            val name = it.getJSName()
                                            property(
                                                name,
                                                convertTypeToJson(
                                                    CodeBlock.of(
                                                        "%N[%S]",
                                                        nameAllocator[valueParameterTag],
                                                        name,
                                                    ),
                                                    it.type.resolve(),
                                                    nameAllocator.copy(),
                                                    config.externalTypeMapping,
                                                    config.defaultInstantJSType,
                                                    logger,
                                                )
                                            )
                                        }.toList().joinToCode(
                                            separator = ",\n",
                                            prefix = "\n",
                                            suffix = "\n"
                                        ),
                                )
                            )
                        )
                    }
                )
                returns(recordType(TypeName.STRING, TypeName.ANY))
            }.build()
        addFunction(fromJson)
        addFunction(toJson)
    }

    private fun ModuleSpec.Builder.createTypescriptTypeMappingForSealedClass(
        declaration: KSClassDeclaration
    ) {
        val subclasses = declaration.getSealedSubclasses()
        val fromJson = FunctionSpec.builder(declaration.getTypescriptFromJsonFunctionName())
            .apply {
                addModifiers(Modifier.EXPORT)
                addTSDoc(
                    "Mapping generated from {@link %N}\n",
                    declaration.qualifiedName!!.asString()
                )
                val jsonParameterName = "json"
                addParameter(
                    ParameterSpec.builder(
                        jsonParameterName,
                        recordType(TypeName.STRING, TypeName.ANY),
                    ).build()
                )
                addCode(
                    codeBlock(brackets = false) {
                        val discriminatorKey = declaration.getDiscriminatorKeyForSealedClass()
                        val typeVariableType = "type"
                        add(
                            const(
                                typeVariableType,
                                CodeBlock.of(
                                    "%N[%S]",
                                    jsonParameterName,
                                    discriminatorKey
                                )
                            )
                        )
                        add(
                            returnStatement(
                                CodeBlock.of(
                                    "{%>%L%<}",
                                    buildList {
                                        add(
                                            CodeBlock.of(
                                                "...%L", lambda(
                                                    args = emptyList(),
                                                    body = block(
                                                        CodeBlock.of(
                                                            "switch (%N) {%>%L%<}",
                                                            typeVariableType,
                                                            buildList {
                                                                addAll(
                                                                    subclasses.map {
                                                                        val typeEnum =
                                                                            it.getSealedSubclassTypeEnumSymbol(
                                                                                declaration
                                                                            )
                                                                        CodeBlock.of(
                                                                            "case %Q: return %Q(%N);",
                                                                            typeEnum,
                                                                            it.getTypescriptFromJsonFunctionNameWithNamespace(config.externalTypeMapping),
                                                                            jsonParameterName,
                                                                        )
                                                                    }
                                                                )
                                                                add(
                                                                    CodeBlock.of(
                                                                        "default: throw new %T(%S + %N);",
                                                                        TypescriptErrorTypeName,
                                                                        "Unknown discriminator value: ",
                                                                        typeVariableType,
                                                                    )
                                                                )
                                                            }.joinToCode(
                                                                separator = "\n",
                                                                prefix = "\n",
                                                                suffix = "\n"
                                                            ),
                                                        )
                                                    )
                                                ).inParentheses()
                                                    .invoke()
                                            ),
                                        )
                                        add(
                                            property(
                                                discriminatorKey,
                                                typeVariableType.asCodeBlock(),
                                            )
                                        )
                                    }.joinToCode(
                                        separator = ",\n",
                                        prefix = "\n",
                                        suffix = "\n"
                                    ),
                                )
                            )
                        )
                    }
                )
                returns(getTypescriptTypeName(declaration.asStarProjectedType()))
            }.build()

        val toJson = FunctionSpec.builder(declaration.getTypescriptToJsonFunctionName())
            .apply {
                addModifiers(Modifier.EXPORT)
                addTSDoc(
                    "Mapping generated from {@link %N}\n",
                    declaration.qualifiedName!!.asString()
                )
                val valueParameterName = "value"
                addParameter(
                    ParameterSpec.builder(
                        valueParameterName,
                        getTypescriptTypeName(declaration.asStarProjectedType()),
                    ).build()
                )
                addCode(
                    codeBlock(brackets = false) {
                        val discriminatorKey = declaration.getDiscriminatorKeyForSealedClass()
                        val typeVariableType = "type"
                        add(
                            const(
                                typeVariableType,
                                CodeBlock.of(
                                    "%N[%S]",
                                    valueParameterName,
                                    discriminatorKey
                                )
                            )
                        )
                        add(
                            returnStatement(
                                CodeBlock.of(
                                    "{%>%L%<}",
                                    buildList {
                                        add(
                                            CodeBlock.of(
                                                "...%L", lambda(
                                                    args = emptyList(),
                                                    body = block(
                                                        CodeBlock.of(
                                                            "switch (%N) {%>%L%<}",
                                                            typeVariableType,
                                                            buildList {
                                                                addAll(
                                                                    subclasses.map {
                                                                        val typeEnum =
                                                                            it.getSealedSubclassTypeEnumSymbol(
                                                                                declaration
                                                                            )
                                                                        CodeBlock.of(
                                                                            "case %Q: return %Q(%N);",
                                                                            typeEnum,
                                                                            it.getTypescriptToJsonFunctionNameWithNamespace(config.externalTypeMapping),
                                                                            valueParameterName,
                                                                        )
                                                                    }
                                                                )
                                                                add(
                                                                    CodeBlock.of(
                                                                        "default: throw new %T(%S + %N);",
                                                                        TypescriptErrorTypeName,
                                                                        "Unknown discriminator value: ",
                                                                        typeVariableType,
                                                                    )
                                                                )
                                                            }.joinToCode(
                                                                separator = "\n",
                                                                prefix = "\n",
                                                                suffix = "\n"
                                                            ),
                                                        )
                                                    )
                                                ).inParentheses()
                                                    .invoke()
                                            )
                                        )
                                        add(
                                            property(
                                                discriminatorKey,
                                                typeVariableType.asCodeBlock(),
                                            )
                                        )
                                    }.joinToCode(
                                        separator = ",\n",
                                        prefix = "\n",
                                        suffix = "\n"
                                    ),
                                )
                            )
                        )
                    }
                )
                returns(recordType(TypeName.STRING, TypeName.ANY))
            }.build()
        addFunction(fromJson)
        addFunction(toJson)
    }

    @OptIn(KspExperimental::class)
    private fun KSDeclaration.assertSerializable(message: () -> String) =
        getAnnotationsByType(Serializable::class).singleOrNull() ?: error(message)

    @OptIn(KspExperimental::class)
    private fun KSDeclaration.getSerialNameAnnotationOrNull(): SerialName? =
        getAnnotationsByType(SerialName::class).singleOrNull()


    private fun typeReferenceToTypescriptTypeName(typeReference: KSTypeReference): TypeName {
        val resolvedType = typeReference.resolve()
        return getTypescriptTypeName(resolvedType)
    }

    private fun toTypescriptPropertySpec(propertyDeclaration: KSPropertyDeclaration): PropertySpec {
        return PropertySpec.builder(
            propertyDeclaration.getJSName(),
            typeReferenceToTypescriptTypeName(propertyDeclaration.type)
        ).build()
    }

    private fun KSPropertyDeclaration.getJSName(): String {
        return getSerialNameAnnotationOrNull()?.value ?: simpleName.asString()
    }

    private fun KSClassDeclaration.getSealedSubclassDiscriminatorValue() =
        (getSerialNameAnnotationOrNull()?.value
            ?: error("Sealed subclasses must be annotated with @SerialName: ${this}"))
}

private val TypescriptErrorTypeName = TypeName.implicit("Error")