package de.viaboxx.markdown

import groovy.json.JsonSlurper
import groovy.util.slurpersupport.Node
import net.sourceforge.plantuml.SourceStringReader
import org.apache.commons.io.FileUtils
import org.apache.commons.lang.StringUtils

import javax.xml.bind.DatatypeConverter

import static de.viaboxx.markdown.Confluence2MD.Mode.*

/**
 * Description: convert a confluence Json page(s)/space to a markdown for pandoc<br>
 * <p>
 * Date: 15.09.14<br>
 * </p>
 * @see Confluence2MD#usage()
 */
class Confluence2MD implements Walker {
    boolean verbose
    /**
     * InputStream, URL or File (or something a JsonSlurper can parse)
     */
    def input
    PrintStream out = System.out
    String userPassword

    static String usage() {
        """
 Usage:
     java Confluence2MD -m wiki &lt;pageId&gt;  &gt; [outputfile]
     java Confluence2MD -m file &lt;input file&gt;  &gt; [outputfile]
     java Confluence2MD -m url  &lt;input URL&gt;   &gt; [outputfile]

 options:
 -m wiki|file|url specify input format/processing mode (default: wiki)
 -o file specify output format, charset=UTF-8  (default: stdout, charset=file.encoding of plaform)
 -oa file specify output format, charset=UTF-8 - open for append!
 -v true for verbose output       (default: false)
 -u user:password to use HTTP-Basic-Auth to request the URL (default: no auth)
 -depth -1..n the depth to follow down the child-pages hierarchy. -1=infinte, 0=no children (default: -1)
 -server URL of confluence server. used in wiki-mode (default: https://viaboxx.atlassian.net/wiki)
 -plantuml  turn off integrated run of PlantUML to render diagrams (default is to call PlantUML automatically)
 -a download folder for attachments (default: attachments)
 last parameter: the file to read (-m file) or the URL to get (-m url) or the pageId to start with (-m wiki)
 +H true/false true: document hierarchy used to generate page header format type (child document => h2 etc) (default: true)
 +T true/false true: title transformation ON (cut everything before first -) (default: true)
 +RootPageTitle true/false true: generate header for root page, false: omit header of root page (default: true)
"""
    }


    enum Mode {
        Default, Panel, BlockQuote, CodeBlock, Table
    }

    private int listIndent = 0, blockIndent = 0
    private Integer itemNumber
    private Mode mode = Default
    private Table table
    private Map pageIdCache = [:]  // key = page.title, value = page.id
    private Map attachmentCache = [:] // key = page.id, value = attachments (jsonSlurper)
    private String wikiServerUrl = "https://viaboxx.atlassian.net/wiki"
    private String rootPageId
    private int maxDepth = -1
    private int depth = 0
    private def currentPage
    private def written = null
    private File downloadFolder = new File("attachments")

    private int MAX_LEVEL = 5 // docx does not support more!

    private final def GET_PAGE_BODY = { pageId -> "/rest/api/content/${pageId}?expand=body.storage" }
    private final def GET_CHILD_PAGES = { pageId -> "/rest/api/content/${pageId}/child/page?limit=300" }
    private final def GET_CHILD_ATTACHMENTS = { pageId -> "/rest/api/content/${pageId}/child/attachment?limit=300" }
    private final def QUERY_PAGE_BY_TITLE = { title -> "/rest/api/content?title=$title" }

    private int imageCounter = 0
    private boolean runPlantUml = true
    private boolean docHierarchy = true, newLine = true, blankLine = true, strong = false
    private boolean titleTransformation = true, titleRootPage = true

    private def File outFile

    Confluence2MD() {}

    Confluence2MD(String[] args) {
        String format = "wiki"
        for (int i = 0; i < args.length - 1; i++) {
            String arg = args[i]
            switch (arg) {
                case "-m":
                    format = args[++i]
                    break
                case "-o":
                    outFile = new File(args[++i])
                    log("Creating output file '${outFile}'")
                    if (outFile.parentFile && !outFile.parentFile.exists()) outFile.parentFile.mkdirs()
                    out = new PrintStream(new FileOutputStream(outFile), false, "UTF-8")
                    break
                case "-oa":
                    outFile = new File(args[++i])
                    log("Appending to output file '${outFile}'")
                    if (outFile.parentFile && !outFile.parentFile.exists()) outFile.parentFile.mkdirs()
                    out = new PrintStream(new FileOutputStream(outFile, true), false, "UTF-8")
                    newLine = false
                    blankLine = false
                    break
                case "-u":
                    userPassword = args[++i]
                    break
                case "-v":
                    verbose = true
                    break
                case "-server":
                    wikiServerUrl = args[++i]
                    break
                case "-depth":
                    maxDepth = Integer.parseInt(args[++i])
                    break
                case "-plantuml":
                    runPlantUml = false
                    break
                case "-a":
                    downloadFolder = new File(args[++i])
                    break
                case "+H":
                    docHierarchy = Boolean.parseBoolean(args[++i])
                    break
                case "+T":
                    titleTransformation = Boolean.parseBoolean(args[++i])
                    break
                case "+RootPageTitle":
                    titleRootPage = Boolean.parseBoolean(args[++i])
                    break
            }
        }
        switch (format) {
            case "wiki":
                rootPageId = args[-1]
                log("Starting page '${rootPageId}'")
                input = openInput(wikiServerUrl + GET_PAGE_BODY(rootPageId), rootPageId + ".body")
                break
            case "url":
                input = openInput(args[-1])
                log("Input url '${args[-1]}'")
                break
            case "file":
                input = new File(args[-1])
                log("Input file '$input'")
                break
            default:
                throw new IllegalArgumentException("unknown format (-f) $format")
        }
    }

    static void main(String[] args) {
        if (!args) {
            println usage()
        } else {
            new Confluence2MD(args).run()
        }
    }

    void run() {
        try {
            def page = new JsonSlurper().parse(input)
            fillPageIdCache(page)
            log("Generating markdown")
            parsePages(page)
        } finally {
            close()
        }
        log("Done")
    }

    void parsePages(page) {
        withPages(page) { each ->
            currentPage = each
            if (!each.body) {
                def childInput = openInput(wikiServerUrl + GET_PAGE_BODY(each.id), each.id + ".body")
                try {
                    parsePage(new JsonSlurper().parse(childInput))
                } finally {
                    close(childInput)
                }
            } else {
                parsePage(each)
            }
        }
    }

    private void withPages(def page, Closure processor) {
        processor(page)
        if (rootPageId && goDeeper()) {
            def queryChildren = openInput(wikiServerUrl + GET_CHILD_PAGES(page.id), page.id + ".children")
            try {
                def children = new JsonSlurper().parse(queryChildren)
                depth++
                children.results.each { child ->
                    withPages(child, processor)
                }
                depth--
            } finally {
                close(queryChildren)
            }
        }
    }

    void parsePage(page) {
        log("${depth} - Processing page $page.id '$page.title'")
        if (depth > 0 || titleRootPage) {
            writeHeader(1, {
                write("${pageTitle(page.title)}")
            }, "PAGE_${page.id}")
        }
        parseBody(page.body.storage.value)
    }

    private String pageTitle(String realTitle) {
        if (!titleTransformation) return realTitle
        String tok = ' - '
        int pos = realTitle.indexOf(tok)
        if (pos < 0) return realTitle
        return realTitle.substring(pos + tok.length())
    }

    private void fillPageIdCache(def page) {
        withPages(page) { each ->
            log("${depth} - Found page $each.id '$each.title'")
            pageIdCache.put(each.title, each.id)
        }
    }

    private boolean goDeeper() {
        return (maxDepth < 0 || maxDepth > depth)
    }

    void parseBody(String body) {
        def xmlSlurper = new XmlSlurper(HTMLParser.build())
        def html = xmlSlurper.parseText("<html><body>" + body + "</body></html>")
        html.BODY.childNodes().each { Node node ->
            if (!handle(node)) {
                walkThrough(node)
            }
        }
    }

    String intoString(Closure process) {
        def baos = new ByteArrayOutputStream()
        def out = new PrintStream(baos)
        withOut(out, process)
        return baos.toString()
    }

    void withOut(final PrintStream out, Closure process) {
        def old = this.out
        this.out = out
        process()
        this.out = old
    }

    private static final String INTERPUNKTION = ".,;:\"'"

    void walkThrough(def parent) {
        parent.children().each { def nodeOrText ->
            if (nodeOrText instanceof Node) {
                if (!handle(nodeOrText as Node)) {
                    walkThrough(nodeOrText as Node)
                }
            } else {
                def text = nodeOrText as String
                if (written == 1 && text) { // workaround, because neko-html parser does not detect spaces between <span>
                    /* example 1: write the space
                    <span class=\"hps\">is a locker</span> <span class=\"hps\">system</span>
                    example 2: do not write the space because there it no space before a ,
                    <span class=\"hps\">system</span><span>,</span> where parcels
                    */
                    int lastChar = text.charAt(text.length() - 1)
                    if (INTERPUNKTION.indexOf(lastChar) < 0) writeRaw(' ')
                }
                write(text)
                written = text
            }
        }
    }

    private void writeHeader(int level, def node) {
        if (node.text()) {
            writeHeader(level) {
                write(node.text().trim())
            }
        } else {
            writeHeader(level) {
                walkThrough(node)
            }
        }
    }

    private void writeHeader(int level, Closure processor, String ref = null) {
        if (!blankLine) {
            if (!newLine) writeln()
            writeln()
        }
        final int effectiveLevel
        if (depth == 0) {
            if (titleRootPage) {
                effectiveLevel = docHierarchy ? depth + level : level
            } else {
                effectiveLevel = docHierarchy ? depth + level - 1 : level
            }
        } else {
            if (titleRootPage) {
                effectiveLevel = docHierarchy ? depth + level : level
            } else {
                effectiveLevel = docHierarchy ? depth + level - 1 : level
            }
        }
        writeHeaderBeginTags(effectiveLevel)
        processor()
        writeHeaderEndTags(effectiveLevel)
        if (ref) writeRaw(" {#${ref}}")
        assertBlankLine()
    }

    private void assertBlankLine() {
        if (!blankLine) {
            if (!newLine) writeln()
            writeln()
        }
    }

    private void writeHeaderBeginTags(int level) {
        writeHeaderTags(level)
    }

    private void writeHeaderEndTags(int level) {
        writeHeaderTags(level)
    }

    private void writeHeaderTags(int level) {
        if (mode != Default || level > MAX_LEVEL) {
            writeRaw("__")
        } else {
            level.times { writeRaw('#') }
        }
    }

    private boolean handle(Node node) {
        switch (node.name()) {
            case "H1":
                writeHeader(2, node)
                return true
            case "H2":
                writeHeader(3, node)
                return true
            case "H3":
                writeHeader(4, node)
                return true
            case "H4":
                writeHeader(5, node)
                return true
            case "H5":
                writeHeader(6, node)
                return true
            case "H6":
                writeHeader(7, node)
                return true
            case "P":
                if (mode != Mode.Table && listIndent == 0) {
                    assertBlankLine()
                }
                walkThrough(node)
                if (mode != Mode.Table) writeln()
                return true
            case "A":
                def href = node.attributes()["href"]
                if (href) {
                    writeRaw("[")
                    walkThrough(node)
                    writeRaw("]")
                    writeRaw("(")
                    writeRaw(href as String)
                    writeRaw(")")
                } // else <a name=\"BACKUP-FILE\"></a> --> anchor not yet supported
                return true
            case "AC:LINK":
                // <ac:link><ri:page ri:content-title=\"Nachricht - PrinterSignalState\" />
                // <ac:link><ac:link-body>PrinterSignalState</ac:link-body></ac:link>
                // <ac:link><ri:attachment ri:filename=\"Anwendungsüberwachung_DHL_NL.xls\" /></ac:link>
                String linkText = null
                String linkUrl = null
                def child
                child = getFirstChildNamed(node, "RI:PAGE")
                if (child) {
                    linkUrl = child.attributes()["ri:content-title"]
                    if (pageIdCache.get(linkUrl)) {
                        linkText = linkUrl
                        linkUrl = "PAGE_" + pageIdCache.get(linkUrl)
                    } else {
                        log("Link out of scope to page: \'$linkUrl\'")
                        linkText = linkUrl
                        linkUrl = null
                    }
                }
                child = getFirstChildNamed(node, "AC:LINK-BODY")
                if (child) {
                    linkText = child.text()
                } else {
                    child = getFirstChildNamed(node, "AC:PLAIN-TEXT-LINK-BODY")
                    if (child) {
                        linkText = child.text()
                    }
                }
                if (!linkText) {
                    child = getFirstChildNamed(node, "RI:ATTACHMENT")
                    if (child) {
                        linkText = child.attributes()["ri:filename"]
                    }
                }
                if (linkUrl) {
                    writeRaw("[")
                    write(linkText ?: linkUrl)
                    writeRaw("](#")
                    writeRaw(linkUrl)
                    writeRaw(")")
                } else if (linkText) {
                    writeRaw("_")
                    write(linkText)
                    writeRaw("_")
                }
                return true
            case "AC:STRUCTURED-MACRO":
                def macroName = node.attributes().get('ac:name')
                switch (macroName) {
                    case "code":
                    case "panel":
                    case "info":
                    case "noformat":
                    case "warning":
                        // ignore
                        break
                    case "section": // Inhaltsverzeichnis
                    default:
                        log("WARN: '$macroName' structured-macro not supported")
                }
                break
            case "AC:PARAMETER":
                return true // skip
            case "BLOCKQUOTE":
                if (mode != Mode.Table) {
                    withMode(BlockQuote) {
                        blockIndent++
                        walkThrough(node)
                        blockIndent--
                    }
                    return true
                }
                break
            case "AC:RICH-TEXT-BODY":
            case "PRE":
                if (mode != Mode.Table) {
                    String text = intoString {
                        if (mode != Panel) {
                            withMode(BlockQuote) {
                                walkThrough(node)
                            }
                        } else {
                            walkThrough(node)
                        }
                    }
                    if (text.contains("~~~~~~~")) { // avoid | and ~~~~~~~ together
                        writeRaw(text)
                    } else if (text) {
                        if (!newLine) writeln()
                        writeRaw(text)
                        assertBlankLine()
                    }
                    return true
                }
                break
            case "CODE":
            case "AC:PLAIN-TEXT-BODY": // codeblock
                /**
                 * Problem codeblock inside table currently not supported,
                 * see http://comments.gmane.org/gmane.text.pandoc/5170
                 */
                /*
     +-----------------------+------------------------+
     | ~~~~                  |                        |
     | This is a code block! | This is ordinary text! |
     | ~~~~                  |                        |
     +-----------------------+------------------------+
                 */
                if (mode != Default) {
                    log("WARN code block nested in $mode currently not supported")
                    write(node.text())
                } else {
                    withMode(CodeBlock) {
                        writeRaw("\n\n~~~~~~~\n")
                        write(node.text())
                        writeRaw("\n~~~~~~~\n")
                    }
                }
                return true
            case "UL":
                withList {
                    itemNumber = null
                    walkThrough(node)
                }
                return true
            case "OL":
                withList {
                    itemNumber = 1
                    walkThrough(node)
                }
                return true
            case "LI":
                assertBlankLine()
                ((listIndent - 1) * 2).times { writeRaw(' ') }
                if (itemNumber != null) {
                    write("${itemNumber}. ")
                    itemNumber = itemNumber + 1
                } else {
                    writeRaw("+ ")
                }
                break
            case "TABLE":
                if (mode == Mode.Table) {
                    // nested tables not supported by pandoc
                    def table = intoString {
                        assertBlankLine()
                        withMode(Mode.Table) {
                            def oldTable = table
                            table = new Table()
                            walkThrough(node)
                            table = oldTable
                        }
                        assertBlankLine()
                    }
                    log("WARN nested table not supported = $table")
                    writeRaw("{table}" + table.replace("|", ",").replace('\n', ';') + "{/table}")
                    return true
                } else {
                    assertBlankLine()
                    withMode(Mode.Table) {
                        def oldTable = table
                        table = new Table()
                        walkThrough(node)
                        table = oldTable
                    }
                    assertBlankLine()
                    return true
                }
                break
            case "TBODY": // ignore
                break
            case "TR":
                table.rows << new Row()
                walkThrough(node)
                writeRaw("|\n")
                if (table.rows.size() == 1) {
                    table.row.renderSeparator(this)
                }
                return true
            case "TD":
                writeRaw("|")
                table.row.cells << new Cell(node)
                table.row.cell.render(this)
                return true
            case "TH":
                writeRaw("|")
                table.row.cells << new Cell(node)
                table.row.cell.render(this)
                return true
            case "BR":
                writeln()
                return true
            case "SPAN": // ignore
                walkThrough(node)
                if (written instanceof String) {  /* char(160) &nbsp; */
                    if (!written.endsWith(' ') && !written.endsWith("\u00A0")) {
                        written = 1  // space maybe to be written
                    }
                }
                return true
            case "I": // italic = emphasis
            case "U": // underline: not yet supported. using italic
                writeRaw("_")
                walkThrough(node)
                writeRaw("_")
                return true
                break
            case "EM":
            case "STRONG":
                if (strong) { // avoid duplication of ** because **** would not work, this can happen when <strong><em>... is nested
                    return false
                } else {
                    strong = true
                    def markdown = intoString {
                        walkThrough(node)
                    }
                    int idx = 0
                    while (markdown.length() > idx && markdown.charAt(idx) == ' ') {
                        idx++
                        writeRaw(" ") // write spaces before ** because after ** must not follow a direct space
                    }
                    if (idx > 0) {
                        markdown = markdown.substring(idx)
                    }
                    writeRaw("**")
                    writeRaw(markdown)
                    writeRaw("**")
                    strong = false
                    return true
                }
                break
            case "AC:IMAGE":
                /*
                <ac:image><ri:attachment ri:filename=\"Resequencer.png\" /></ac:image>
                 */
                String title = null
                String url = null
                def child = getFirstChildNamed(node, "RI:ATTACHMENT")
                if (child) {    // attached image
                    title = child.attributes()["ri:filename"]
                    def attachments = getAttachments(currentPage.id)
                    def attachment = findAttachmentTitled(attachments, title)
                    if (!attachment) {
                        log("WARN: Cannot find attachment $title")
                    } else {
                        url = downloadedFile(attachment).path
                    }
                } else {
                    child = getFirstChildNamed(node, "RI:URL")
                    if (child) { // image by URL
                        url = child.attributes()["ri:value"]
                    }
                }
                writeRaw("![")
                write(title ?: url)
                writeRaw("](")
                write(url)
                writeRaw(")")
                return true
            case "S":  // strikeout
                writeRaw("~~")
                def text = intoString { walkThrough(node) }
                writeRaw(text.trim())
                writeRaw("~~")
                return true
            case "HR":
                writeRaw("\n---\n")
                return true
            case "AC:EMOTICON":
                def icon = node.attributes()["ac:name"]
                switch (icon) {
                    case "minus":
                        write(" (-) ")
                        break
                    case "smile":
                        write(" :-) ")
                        break
                    case "sad":
                        write(" :-( ")
                        break
                    case "cheeky":
                        write(" :-P ")
                        break
                    case "laugh":
                        write(" :-D ")
                        break
                    case "wink":
                        write(" ;-) ")
                        break
                    case "thumbs-up":
                        write(" (^.^) ")
                        break
                    case "thumbs-down":
                        write(" (:-[) ")
                        break
                    case "tick":
                        write(" (ok) ")
                        break
                    case "cross":
                        write(" (x) ")
                        break
                    case "warning":
                        write(" (!) ")
                        break
                    case "question":
                        write(" (?) ")
                        break
                    default:
                        write("($icon)")
                }
                return true
            case "DIV":
                break // ignore
            case "AC:MACRO":  // e.g. plantUML
                def macro = node.attributes()['ac:name']
                switch (macro) {
                    case "plantuml":
                        plantUML(node)
                        break
                    default:
                        log("Unknown macro tag ${node.name()} = ${macro}")
                }
                return true
            case "COL":
            case "COLGROUP":
            case "CITE":
            default:
                log("Unhandled tag ${node.name()} = ${node.text()}")

        }
        return false
    }

    private void plantUML(Node node) {
        def text = node.text()
        imageCounter++
        String img = new File(downloadFolder, "plantuml${imageCounter}.png").path
        if (runPlantUml) {
            if (text.trim().startsWith("!include ")) {
                log("Looking for plantUML-!include for $img with $text")
                def pumlAttachment = text.trim().substring("!include ".length())
                def page
                def idx = pumlAttachment.indexOf("^")
                if (idx >= 0) {
                    page = pumlAttachment.substring(0, idx)
                    pumlAttachment = pumlAttachment.substring(idx + 1)
                } else {
                    page = null
                }
                def attachments
                if (page) {
                    def pageId = queryPageIdByTitle(page)
                    if (pageId) {
                        attachments = getAttachments(pageId)
                    }
                } else {
                    attachments = getAttachments(currentPage.id)
                }
                def attachment
                if (attachments) {
                    attachment = findAttachmentTitled(attachments, pumlAttachment)
                }
                if (attachment) {
                    File pumlFile = downloadedFile(attachment)
                    if (pumlFile) {
                        text = FileUtils.readFileToString(pumlFile)
                    }
                }
            }
            if (!text.contains("@startuml") && !text.contains("@startdot")) text = "@startuml\n" + text
            if (!text.contains("@enduml") && !text.contains("@enddot")) text += "\n@enduml"
            log("Running PlantUml on $img with \n$text")
            def reader = new SourceStringReader(text)
            FileOutputStream file = new FileOutputStream(img)
            reader.generateImage(file);
            file.close()
        } else {
            writeRaw("<!--\n")
            writeRaw(text)
            writeRaw("\n-->\n")
        }
        writeRaw("![Image generated by PlantUML]($img)\n");
    }

    private String queryPageIdByTitle(String title) {
        def pageId = pageIdCache.get(title)
        if (pageId) return pageId
        def input = openInput(wikiServerUrl + QUERY_PAGE_BY_TITLE(title), title + ".title.query")
        def json = new JsonSlurper().parse(input)
        def page = json.results.find { it.title == title }
        close(input)
        if (page) {
            pageId = page.id
            pageIdCache.put(title, pageId)
        }
        return pageId
    }

    private def getAttachments(def pageId) {
        def attachments = attachmentCache[pageId]
        if (attachments == null) {
            def url = wikiServerUrl + GET_CHILD_ATTACHMENTS(pageId)
            def stream = openInput(url, pageId + ".attachments")
            try {
                attachments = new JsonSlurper().parse(stream)
                attachmentCache[pageId] = attachments
            } finally {
                close(stream)
            }
        }
        return attachments
    }

    private File downloadedFile(def attachment) {
        if (!downloadFolder.exists()) downloadFolder.mkdirs()
        File targetFile = new File(downloadFolder, attachment.id + "_" + attachment.title)
        if (!targetFile.exists()) { // speed up - use existing file
            def downloadUrl = wikiServerUrl + attachment._links.download
            log("Downloading '${targetFile.name}' from '$downloadUrl'")
            def stream = openStream(downloadUrl)
            FileUtils.copyInputStreamToFile(stream, targetFile)
            stream.close()
        } else {
            log("Found downloaded file ${targetFile.name}")
        }
        return targetFile
    }

    private Node getFirstChildNamed(Node node, String name) {
        return node.children().find { child ->
            (child instanceof Node && name == child.name())
        }
    }

    private def findAttachmentTitled(def attachments, String title) {
        return attachments.results.find { it.title == title }
    }

    private def openInput(String urlString, String cache = null) {
        URL url = new URL(urlString)
        File cacheFile = cache ? new File(downloadFolder, "." + cache + ".json") : null
        if (cacheFile?.exists()) {
            log("Found cached file $cacheFile.name")
            return new FileInputStream(cacheFile)
        }
        log("Requesting $urlString")
        if (userPassword) {
            def conn = url.openConnection()
            String basicAuth = "Basic " + DatatypeConverter.printBase64Binary(userPassword.getBytes())
            conn.setRequestProperty("Authorization", basicAuth)
            if (!cache) return conn.inputStream
            else {
                def stream = conn.inputStream
                FileUtils.copyInputStreamToFile(stream, cacheFile)
                stream.close()
                return new FileInputStream(cacheFile)
            }
        } else {
            if (!cache) return url
            else {
                def stream = url.openStream()
                FileUtils.copyInputStreamToFile(stream, cacheFile)
                stream.close()
                return new FileInputStream(cacheFile)
            }
        }
    }

    private InputStream openStream(String urlString) {
        def stream = openInput(urlString)
        if (stream instanceof URL) stream = stream.openStream()
        return stream
    }

    void writeln() {
        boolean newLineBefore = newLine
        if (mode == Mode.Table) {
            out.println('\\')
            newLine = true
        } else {
            out.println()
            newLine = true
            if (mode == Panel) {
                writeRaw("\n| ")
            } else if (mode == BlockQuote) {
                writeRaw("\n")
                blockIndent.times { writeRaw("> ") }
            }
        }
        blankLine = newLine && newLineBefore
    }

    private void log(String text) {
        if (verbose) {
            (depth * 2).times { print(" ") }
            println(text)
        }
    }

    void write(String text) {
        if (text) {
            writeRaw(transform(text))
        }
    }

    private void computeLineStatus(String text) {
        if (mode == Panel) {
            newLine = text.endsWith("\n ")
            blankLine = text.endsWith("\n \n ")
        } else if (mode == BlockQuote) {
            StringBuffer buf = new StringBuffer(2 * blockIndent)
            blockIndent.times { buf.append("> ") }
            def nl = "\n " + buf.toString()
            newLine = text.endsWith(nl)
            blankLine = text.endsWith(nl + nl)
        } else if (mode == Mode.Table) {
            text = text.replace('\n', '\\\n')
            newLine = text.endsWith('\\\n')
            blankLine = text.endsWith('\\\n\\\n')
        } else {
            newLine = text.endsWith('\n')
            blankLine = text.endsWith('\n\n')
        }
    }

    private String transform(String text) {
        // escape unwanted footnotes etc.
        if (mode != CodeBlock) { // CodeBlock: do not replace most of the things
            String[] search = ['\\', '<', '#', "^[", "*", "`", "{", "}", "[", "]", ">", "#", "+", "-", ".", "!"] as String[]
            String[] replace = ['\\\\', '\\<', '\\#', "^\\[", "\\*", "\\`", "\\{", "\\}", "\\[", "\\]", "\\>", "\\#", "\\+", "\\-", "\\.", "\\!"] as String[]
            text = StringUtils.replaceEach(text, search, replace)
        }
        if (mode == Panel) {
            text = text.replace("\n", "\n ")
        } else if (mode == BlockQuote) {
            StringBuffer buf = new StringBuffer(2 * blockIndent)
            blockIndent.times { buf.append("> ") }
            text = text.replace("\n", "\n " + buf.toString())
        } else if (mode == Mode.Table) {
            text = text.replace('\n', '\\\n')
        }
        // \`*_{}[]()>#+-.!
        return text
    }

    void writeRaw(String text) {
        if (text) {
            out.print(text)
            computeLineStatus(text)
        }
    }

    protected void withList(Closure processor) {
        listIndent++
        def oldItemNumber = itemNumber
        processor()
        itemNumber = oldItemNumber
        listIndent--
        assertBlankLine()
    }

    protected void withMode(final Mode mode, Closure processor) {
        def old = this.mode
        this.mode = mode
        processor()
        this.mode = old
    }

    void close() {
        close(input)
        if (out instanceof Closeable) out.close()
    }

    void close(def inputThing) {
        if (inputThing instanceof Closeable) input.close()
    }

}
