
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2011, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.util.collections;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import de.unkrig.commons.nullanalysis.NotNullByDefault;
import de.unkrig.commons.nullanalysis.Nullable;

/**
 * Various utility methods for map processing.
 */
public final
class MapUtil {

    private
    MapUtil() {}

    /**
     * Returns a {@link Map} that is composed of <var>map1</var> and <var>map2</var>. <var>Map2</var> is
     * never modified.
     */
    @NotNullByDefault(false) public static <K, V> Map<K, V>
    combine(final Map<K, V> map1, final Map<K, V> map2) {
        return new Map<K, V>() {

            @Override public void
            clear() { throw new UnsupportedOperationException("clear"); }

            @Override public Set<Map.Entry<K, V>>
            entrySet() { throw new UnsupportedOperationException("entrySet"); }

            @Override public Set<K>
            keySet() { throw new UnsupportedOperationException("keySet"); }

            @Override public V
            remove(Object key) { throw new UnsupportedOperationException("remove"); }

            @Override public Collection<V>
            values() { throw new UnsupportedOperationException("values"); }

            @Override public boolean
            containsKey(Object key) {
                return map1.containsKey(key) || map2.containsKey(key);
            }

            @Override public boolean
            containsValue(Object value) {
                return map1.containsValue(value) || map2.containsValue(value);
            }

            @Override public V
            get(Object key) {
                V value = map1.get(key);
                return value != null || map1.containsKey(key) ? value : map2.get(key);
            }

            @Override public boolean
            isEmpty() {
                return map1.isEmpty() && map2.isEmpty();
            }

            @Override public V
            put(K key, V value) {
                return map1.put(key, value);
            }

            @Override public void
            putAll(Map<? extends K, ? extends V> map) {
                map1.putAll(map);
            }

            @Override public int
            size() {
                return map1.size() + map2.size();
            }
        };
    }

    /**
     * Returns a {@link Map} that is composed of the {@code delegate} and one extra entry.
     */
    @NotNullByDefault(false) public static <K, V> Map<K, V>
    augment(final Map<K, V> delegate, final K extraKey, final V extraValue) {
        return new Map<K, V>() {

            @Override public void
            clear() { throw new UnsupportedOperationException("clear"); }

            @Override public Set<Map.Entry<K, V>>
            entrySet() { throw new UnsupportedOperationException("entrySet"); }

            @Override public Set<K>
            keySet() { throw new UnsupportedOperationException("keySet"); }

            @Override public V
            remove(Object key) { throw new UnsupportedOperationException("remove"); }

            @Override public Collection<V>
            values() { throw new UnsupportedOperationException("values"); }

            @Override public V
            put(K key, V value) { throw new UnsupportedOperationException("put"); }

            @Override public void
            putAll(Map<? extends K, ? extends V> map) { throw new UnsupportedOperationException("putAll"); }

            @Override public boolean
            containsKey(Object key) {
                return MapUtil.equal(key, extraKey) || delegate.containsKey(key);
            }

            @Override public boolean
            containsValue(Object value) {
                return MapUtil.equal(value, extraValue) || delegate.containsValue(value);
            }

            @Override public V
            get(Object key) {
                return MapUtil.equal(key, extraKey) ? extraValue : delegate.get(key);
            }

            @Override public boolean
            isEmpty() {
                return false;
            }

            @Override public int
            size() {
                return delegate.size() + 1;
            }
        };
    }

    /**
     * The often-needed "equal" method that handles {@code null} references.
     * <p>
     * Redeemed by {@code java.util.Objects.equals(Object, Object)} which is available since Java 1.7.
     */
    public static boolean
    equal(@Nullable Object o1, @Nullable Object o2) {
        return o1 == null ? o2 == null : o1.equals(o2);
    }

    /**
     * @param keysAndValues An even-sized array filled alternatingly with objects of type {@code K} and {@code V}
     * @return              An unmodifiable {@link Map} populated from the given key/value pairs
     */
    @SuppressWarnings("unchecked") public static <K, V> Map<K, V>
    fromMappings(Object... keysAndValues) {
        final Map<K, V> m = new HashMap<K, V>();
        for (int i = 0; i < keysAndValues.length;) {
            m.put((K) keysAndValues[i++], (V) keysAndValues[i++]);
        }
        return Collections.unmodifiableMap(m);
    }
}
