
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2012, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.io;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;

import de.unkrig.commons.nullanalysis.NotNullByDefault;

/**
 * Decodes the data bytes (assuming ISO-8859-1 encoding) into characters, and writes them to the given delegate writer.
 * In a sense, this class is the complement of the {@link java.io.InputStreamReader}.
 */
@NotNullByDefault(false) public
class WriterOutputStream extends OutputStream {

    private final Writer delegate;

    /**
     * @see WriterOutputStream
     */
    public
    WriterOutputStream(Writer delegate) {
        this.delegate = delegate;
    }

    @Override public void
    close() throws IOException {
        this.delegate.close();
    }

    @Override public void
    flush() throws IOException {
        this.delegate.flush();
    }

    @Override public void
    write(byte[] buf, int off, int len) throws IOException {
        for (; len > 0; len--) {
            this.delegate.write(0xff & buf[off++]);
        }
    }

    @Override public void
    write(int c) throws IOException {
        this.delegate.write(c);
    }
}
