
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2013, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.util.time;

import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.unkrig.commons.nullanalysis.Nullable;

/**
 * Representation of the length of time between two points of time, with a resolution of 1 millisecond.
 */
public final
class Duration {

    // SUPPRESS CHECKSTYLE WrapAndIndent:16
    // SUPPRESS CHECKSTYLE LineLength:15
    private static final Pattern
    TI_MS        = Pattern.compile("(\\d++) *ms"),
    TI_S         = Pattern.compile("(\\d++|\\d+\\.\\d*|\\.\\d+) *(?:s|secs?|\")?"),
    TI_M         = Pattern.compile("(\\d++|\\d+\\.\\d*|\\.\\d+) *(?:mins?|')?"),
    TI_X_X       = Pattern.compile("(\\d+):(\\d+)"),
    TI_M_S       = Pattern.compile("(\\d+)(?::| *(mins?|') *)(\\d+(?:\\.\\d*)?) *(?:s|secs?|\")?"),
    TI_H         = Pattern.compile("(\\d++|\\d+\\.\\d*|\\.\\d+) *h"),
    TI_H_M_S     = Pattern.compile("(\\d+)(?::| *h *)(\\d+)(?::| *mins? *)(\\d+(?:\\.\\d*)?) *(s|secs?)?"),
    TI_D         = Pattern.compile("(\\d++|\\d+\\.\\d*|\\.\\d+) *d"),
    TI_D_H_M     = Pattern.compile("(\\d+) *d *(\\d+)(?::| *h *)(\\d+) *(?:mins?)?"),
    TI_D_H_M_S   = Pattern.compile("(\\d+) *d *(\\d+)(?::| *h *)(\\d+)(?::| *mins? *)(\\d+(?:\\.\\d*)?)(?: *s(?:ecs?)?)"),
    TI_W         = Pattern.compile("(\\d++|\\d+\\.\\d*|\\.\\d+) *wk?"),
    TI_W_D       = Pattern.compile("(\\d+) *wk? *(\\d+) *d"),
    TI_W_D_H_M   = Pattern.compile("(\\d+) *wk? *(\\d+) *d *(\\d+):(\\d+)"),
    TI_W_D_H_M_S = Pattern.compile("(\\d+) *wk? *(\\d+) *d *(\\d+)(?::| *h *)(\\d+)(?::| *mins? *)(\\d+(?:\\.\\d*)?)(?: *s(?:ecs?)?)");

    private final long ms;

    public
    Duration(long ms) { this.ms = ms; }

    public
    Duration(double seconds) { this.ms = (long) (1000.0 * seconds); }

    public
    Duration(String s) {
        Matcher m;

        m = Duration.TI_MS.matcher(s);
        if (m.matches()) {
            this.ms = Long.parseLong(m.group(1));
            return;
        }

        m = Duration.TI_S.matcher(s);
        if (m.matches()) {
            this.ms = (long) (1000.0 * Double.parseDouble(m.group(1)));
            return;
        }

        m = Duration.TI_M.matcher(s);
        if (m.matches()) {
            this.ms = (long) (60.0 * 1000.0 * Double.parseDouble(m.group(1)));
            return;
        }

        m = Duration.TI_X_X.matcher(s);
        if (m.matches()) {
            throw new IllegalArgumentException(
                "'"
                + s
                + "' could mean 'hh:mm' or 'mm:ss'; please use either '"
                + s
                + ":00' or '0:"
                + s
                + "'"
            );
        }

        m = Duration.TI_M_S.matcher(s);
        if (m.matches()) {
            this.ms = (60 * 1000) * Long.parseLong(m.group(1)) + (long) (1000.0 * Double.parseDouble(m.group(2)));
            return;
        }

        m = Duration.TI_H.matcher(s);
        if (m.matches()) {
            this.ms = (long) (60.0 * 1000.0 * Double.parseDouble(m.group(1)));
            return;
        }

        m = Duration.TI_H_M_S.matcher(s);
        if (m.matches()) {
            this.ms = (
                0
                + (3600 * 1000) * Long.parseLong(m.group(1))
                +   (60 * 1000) * Long.parseLong(m.group(2))
                + (long) (1000.0 * Double.parseDouble(m.group(3)))
            );
            return;
        }

        m = Duration.TI_D.matcher(s);
        if (m.matches()) {
            this.ms = (long) (24.0 * 3600.0 * 1000.0 * Double.parseDouble(m.group(1)));
            return;
        }

        m = Duration.TI_D_H_M.matcher(s);
        if (m.matches()) {
            this.ms = (
                0
                + (24 * 3600 * 1000) * Long.parseLong(m.group(1))
                +      (3600 * 1000) * Long.parseLong(m.group(2))
                +        (60 * 1000) * Long.parseLong(m.group(3))
            );
            return;
        }

        m = Duration.TI_D_H_M_S.matcher(s);
        if (m.matches()) {
            this.ms = (
                0
                + (24 * 3600 * 1000) * Long.parseLong(m.group(1))
                +      (3600 * 1000) * Long.parseLong(m.group(2))
                +        (60 * 1000) * Long.parseLong(m.group(3))
                + (long) (1000.0 * Double.parseDouble(m.group(4)))
            );
            return;
        }

        m = Duration.TI_W.matcher(s);
        if (m.matches()) {
            this.ms = (long) (7.0 * 24.0 * 3600.0 * 1000.0 * Long.parseLong(m.group(1)));
            return;
        }

        m = Duration.TI_W_D.matcher(s);
        if (m.matches()) {
            this.ms = (
                0
                + (7 * 24 * 3600 * 1000) * Long.parseLong(m.group(1))
                +     (24 * 3600 * 1000) * Long.parseLong(m.group(2))
            );
            return;
        }

        m = Duration.TI_W_D_H_M.matcher(s);
        if (m.matches()) {
            this.ms = (
                0
                + (7 * 24 * 3600 * 1000) * Long.parseLong(m.group(1))
                +     (24 * 3600 * 1000) * Long.parseLong(m.group(2))
                +          (3600 * 1000) * Long.parseLong(m.group(3))
                +            (60 * 1000) * Long.parseLong(m.group(4))
            );
            return;
        }

        m = Duration.TI_W_D_H_M_S.matcher(s);
        if (m.matches()) {
            this.ms = (
                0
                + (7 * 24 * 3600 * 1000) * Long.parseLong(m.group(1))
                +     (24 * 3600 * 1000) * Long.parseLong(m.group(2))
                +          (3600 * 1000) * Long.parseLong(m.group(3))
                +            (60 * 1000) * Long.parseLong(m.group(4))
                + (long) (1000.0 * Double.parseDouble(m.group(5)))
            );
            return;
        }

        throw new IllegalArgumentException("Time interval '" + s + "' cannot be parsed");
    }

    /** @return The number of milliseconds represented by this object */
    public long milliseconds() { return this.ms; }

    /** @return The length of time in seconds represented by this object */
    public double toSeconds() { return this.ms / 1000.0; }

    @Override public String
    toString() {
        return (
            this.ms < 1000 ? this.ms + "ms" :
            this.ms < 60 * 1000 ? String.format(
                Locale.US,
                "%.3fs",
                this.ms / 1000.0
            ) :
            this.ms < 24 * 60 * 60 * 1000 ? String.format(
                Locale.US,
                "%d:%02d:%06.3f",
                (this.ms / (60 * 60 * 1000)),
                (this.ms /      (60 * 1000)) % 60,
                (this.ms % (60 * 1000)) / 1000.0
            ) :
            this.ms < 7 * 24 * 60 * 60 * 1000 ? String.format(
                Locale.US,
                "%dd %d:%02d:%06.3f",
                (this.ms / (24 * 60 * 60 * 1000)),
                (this.ms /      (60 * 60 * 1000)) % 24,
                (this.ms /           (60 * 1000)) % 60,
                (this.ms % (60 * 1000)) / 1000.0
            ) :
            String.format(
                Locale.US,
                "%dw %dd %d:%02d:%06.3f",
                (this.ms / (7 * 24 * 60 * 60 * 1000)),
                (this.ms /     (24 * 60 * 60 * 1000)) % 7,
                (this.ms /          (60 * 60 * 1000)) % 24,
                (this.ms /               (60 * 1000)) % 60,
                (this.ms % (60 * 1000)) / 1000.0
            )
        );
    }

    /** @return A duration that represents the sum of {@code this} and {@code other} */
    public Duration
    add(Duration other) { return new Duration(this.ms + other.ms); }

    /** @return A duration that is {@code factor} as long as this duration */
    public Duration
    multiply(double factor) { return new Duration((long) (this.ms * factor)); }

    /** @return A duration which is one {@code divisor}th of this duration long */
    public Duration
    divide(double divisor) { return new Duration((long) (this.ms / divisor)); }

    /** @return Whether this object represents the zero-length interval */
    public boolean
    isZero() { return this.ms == 0; }

    @Override public int
    hashCode() { return (int) (this.ms ^ (this.ms >>> 32)); }

    @Override public boolean
    equals(@Nullable Object obj) { return obj == this || (obj instanceof Duration && ((Duration) obj).ms == this.ms); }
}

