
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2012, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.util.collections;

import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Various {@link Iterable}-related utility methods.
 */
public final
class IterableUtil {

    private
    IterableUtil() {}

    /**
     * @return An {@link Iterable} which can't be modified through the {@link Iterator#remove()} method
     */
    public static <T> Iterable<T>
    unmodifiableIterable(final Iterable<T> i) {
        return new Iterable<T>() {

            @Override public Iterator<T>
            iterator() {
                final Iterator<T> it = i.iterator();

                return new Iterator<T>() {

                    @Override public boolean
                    hasNext() { return it.hasNext(); }

                    @Override public T
                    next() { return it.next(); }

                    @Override public void
                    remove() { throw new UnsupportedOperationException("remove"); }
                };
            }
        };
    }

    /**
     * @return An {@link Iterable} producing the  given {@code subject} {@code n} times
     */
    public static <T> Iterable<T>
    repeat(final T subject, final int n) {
        if (n < 0) throw new IllegalArgumentException(Integer.toString(n));

        return new Iterable<T>() {

            @Override public Iterator<T>
            iterator() {
                return new Iterator<T>() {

                    final AtomicInteger index = new AtomicInteger();

                    @Override public boolean
                    hasNext() {
                        return this.index.intValue() < n;
                    }

                    @Override public T
                    next() {
                        if (this.index.getAndIncrement() >= n) throw new NoSuchElementException();

                        return subject;
                    }

                    @Override public void
                    remove() {
                        throw new UnsupportedOperationException("remove");
                    }
                };
            }
        };
    }

    /**
     * @return An {@link Iterable} that traverses the elements of the given {@link List} in reverse order.
     */
    public static <E> Iterable<E>
    reverseList(final List<E> list) {
        return new Iterable<E>() {

            @Override public Iterator<E>
            iterator() {
                final ListIterator<E> li = list.listIterator(list.size());
                return new Iterator<E>() {

                    @Override public boolean
                    hasNext() { return li.hasPrevious(); }

                    @Override public E
                    next() { return li.previous(); }

                    @Override public void
                    remove() { li.remove(); }
                };
            }
        };
    }
}
