
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2015, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.text;

import de.unkrig.commons.lang.protocol.RunnableWhichThrows;


/**
 * Manages the "context printer".
 *
 * <p>
 *   Use the {@link #error(String)}, {@link #warn(String)}, {@link #info(String)}, {@link #verbose(String)} and
 *   {@link #debug(String)} methods to print to the context printer.
 * </p>
 * <p>
 *   The default context printer prints errors and warnings to {@link System#err} and infos to {@link System#out},
 *   and discards verbose and debug messages.
 * </p>
 * <p>
 *   To switch to a different printer, use the {@link #withPrinter(Printer, Runnable)} methods.
 * </p>.
 */
public final
class Printers {

    private Printers() {}

    private static final LevelFilteredPrinter FILTERED_CONSOLE = new LevelFilteredPrinter(new StreamPrinter());

    private static final ThreadLocal<Printer> THREAD_LOCAL_PRINTER = new InheritableThreadLocal<Printer>() {
        @Override protected Printer initialValue() { return Printers.FILTERED_CONSOLE; }
    };

    /**
     * @return the context printer for this thread
     */
    private static Printer get() { return Printers.THREAD_LOCAL_PRINTER.get(); }

    /**
     * Sets the context printer for this thread and all its (existing and future) child threads (unless they have set
     * their own context printer, or until they set their own context printer).
     */
    private static void set(Printer printer) { Printers.THREAD_LOCAL_PRINTER.set(printer); }

    /** Prints an error condition on the context printer. */
    public static void
    error(String message) { Printers.get().error(message); }

    /** Prints an error condition on the context printer. */
    public static void
    error(String pattern, Object... arguments) { Printers.get().error(pattern, arguments); }

    /** Prints an error condition on the context printer. */
    public static void
    error(String message, Throwable t) { Printers.get().error(message, t); }

    /** Prints an error condition on the context printer. */
    public static void
    error(String pattern, Throwable t, Object... arguments) { Printers.get().error(pattern, t, arguments); }

    /** Prints a warning condition on the context printer. */
    public static void
    warn(String message) { Printers.get().warn(message); }

    /** Prints a warning condition on the context printer. */
    public static void
    warn(String pattern, Object... arguments) { Printers.get().warn(pattern, arguments); }

    /** Prints an informative ("normal") message on the context printer. */
    public static void
    info(String message) { Printers.get().info(message); }

    /** Prints an informative ("normal") message on the context printer. */
    public static void
    info(String pattern, Object... arguments) { Printers.get().info(pattern, arguments); }

    /** Prints a verbose message on the context printer. */
    public static void
    verbose(String message) { Printers.get().verbose(message); }

    /** Prints a verbose message on the context printer. */
    public static void
    verbose(String pattern, Object... arguments) { Printers.get().verbose(pattern, arguments); }

    /** Prints a debug message on the context printer. */
    public static void
    debug(String message) { Printers.get().debug(message); }

    /** Prints a debug message on the context printer. */
    public static void
    debug(String pattern, Object... arguments) { Printers.get().debug(pattern, arguments); }

    /**
     * Replaces the context printer with the given {@code printer} while the {@code runnable} is running.
     */
    public static synchronized void
    withPrinter(Printer printer, Runnable runnable) {

        final Printer oldThreadPrinter = Printers.get();

        Printers.set(printer);
        try {
            runnable.run();
        } finally {
            Printers.set(oldThreadPrinter);
        }
    }

    /**
     * Replaces the context printer with the given {@code printer} while the {@code runnable} is running.
     */
    public static synchronized <EX extends Exception> void
    withPrinter(Printer printer, RunnableWhichThrows<EX> runnable) throws EX {

        final Printer oldThreadPrinter = Printers.get();

        Printers.set(printer);
        try {
            runnable.run();
        } finally {
            Printers.set(oldThreadPrinter);
        }
    }
}
