
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2012, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.net.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;

import de.unkrig.commons.lang.protocol.RunnableWhichThrows;
import de.unkrig.commons.lang.protocol.Stoppable;
import de.unkrig.commons.net.TcpServer;
import de.unkrig.commons.net.http.servlett.Servlett;

/**
 * Implementation of an HTTP server.
 *
 * @see #HttpServer(InetSocketAddress, HttpClientConnectionHandler)
 */
public final
class HttpServer implements RunnableWhichThrows<IOException>, Stoppable {

    private final TcpServer tcpServer;

    /**
     * Accepts connections on the {@code endpoint}; obtains a new {@link Servlett} from the
     * {@code httpClientConnectionHandlerFactory} for each accepted connection, and then calls
     * {@link Servlett#handleRequest(HttpRequest)} for each received request.
     */
    public
    HttpServer(
        InetSocketAddress                 endpoint,
        final HttpClientConnectionHandler httpClientConnectionHandler
    ) throws IOException {
        this.tcpServer = new TcpServer(endpoint, 0, new TcpServer.ConnectionHandler() {

            @Override public void
            handleConnection(
                InputStream       in,
                OutputStream      out,
                InetSocketAddress localSocketAddess,
                InetSocketAddress remoteSocketAddress,
                Stoppable         stoppable
            ) throws Exception {

                httpClientConnectionHandler.handleConnection(
                    in,
                    out,
                    localSocketAddess,
                    remoteSocketAddress,
                    stoppable
                );
            }
        });
    }

    @Override public void
    run() throws IOException {
        this.tcpServer.run();
    }

    @Override public void
    stop() {
        this.tcpServer.stop();
    }

    /**
     * @return The endpoint address of this HTTP server
     * @see    TcpServer#getEndpointAddress()
     */
    public InetSocketAddress
    getEndpointAddress() {
        return this.tcpServer.getEndpointAddress();
    }
}
