
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2011, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.lang.protocol;

import java.util.Collection;
import java.util.Iterator;
import java.util.Random;

import de.unkrig.commons.nullanalysis.Nullable;

/**
 * Various {@link Producer}-related utility methods.
 */
public final
class ProducerUtil {

    private
    ProducerUtil() {}

    /**
     * The returned {@link Producer} calls the {@code delegate} iff the {@code condition} returns {@code true},
     * otherwise it returns the previous product of the {@code delegate}, or {@code null} iff the delegate has
     * not yet been called.
     *
     * @param subject The {@code subject} argument for the {@code condition}
     */
    public static <T, ST> Producer<T>
    sparingProducer(final Producer<T> delegate, final Predicate<ST> condition, final ST subject) {
        return new Producer<T>() {

            @Nullable private T product;

            @Override @Nullable public T
            produce() {
                if (condition.evaluate(subject)) this.product = delegate.produce();
                return this.product;
            }
        };
    }

    /**
     * Returns a {@code Producer<Boolean>} who's first evaluation result is {@code true}, and each following result is
     * {@code true} iff the last {@code true} result was returned at least the given {@code interval} milliseconds ago.
     * In other words, the interval between two returned {@code true} values is never shorter than {@code interval}
     * milliseconds.
     */
    public static Producer<Boolean>
    every(final long interval) {
        return new Producer<Boolean>() {

            private long expirationTime;

            @Override @Nullable public synchronized Boolean
            produce() {
                long now = System.currentTimeMillis();
                if (now >= this.expirationTime) {
                    this.expirationTime = now + interval;
                    return true;
                } else {
                    return false;
                }
            }
        };
    }

    /**
     * Converts a {@link Producer Producer&lt;? extends TT>} (the "source") into a {@link ProducerWhichThrows
     * ProducerWhichThrows&lt;TT, TE>} (the "target"), which is always possible.
     *
     * @param <T>  The element type
     * @param <EX> The target producer's exception
     */
    public static <T, EX extends Throwable> ProducerWhichThrows<T, EX>
    asProducerWhichThrows(final Producer<? extends T> source) {

        return new ProducerWhichThrows<T, EX>() {

            @Override @Nullable public T
            produce() { return source.produce(); }
        };
    }

    /**
     * Converts a {@link ProducerWhichThrows ProducerWhichThrows&lt;? extends T, ? extends RuntimeException>} (the
     * "source") into a {@link Producer Producer&lt;T>} (the "target"), which is always possible.
     *
     * @param <T>  The element type
     */
    public static <T> Producer<T>
    asProducer(final ProducerWhichThrows<? extends T, ? extends RuntimeException> source) {

        return new Producer<T>() {

            @Override @Nullable public T
            produce() { return source.produce(); }
        };
    }

    /**
     * @return A {@link Producer} that produced the given {@code elements}
     */
    public static <T> Producer<T>
    fromElements(final T... elements) {
        return new Producer<T>() {

            int idx;

            @Override @Nullable public T
            produce() {
                return this.idx == elements.length ? null : elements[this.idx++];
            }
        };
    }

    /**
     * Produces the elements of the {@code delegate} collection, in its iteration order, or {@code null} iff the
     * collection is empty. The elements are <i>removed</i> from the collection as they are produced.
     */
    public static <T> Producer<T>
    fromCollection(final Collection<T> delegate) {

        return new Producer<T>() {

            @Override @Nullable public T
            produce() {

                Iterator<T> it = delegate.iterator();

                if (!it.hasNext()) return null;

                T product = it.next();
                it.remove();
                return product;
            }
        };
    }

    /**
     * Produces the products of the {@code iterator}, or {@code null} iff the {@code iterator} has no more elements.
     */
    public static <T> Producer<T>
    fromIterator(final Iterator<T> iterator) {

        return new Producer<T>() {

            @Override @Nullable public T produce() { return iterator.hasNext() ? iterator.next() : null; }
        };
    }

    /**
     * Produces objects based on the number of preceding invocations, i.e. the {@code indexTransformer} is invoked
     * with subjects '0', '1', '2', ...
     */
    public static <T> Producer<T>
    fromIndexTransformer(final Transformer<Integer, T> indexTransformer) {
        return new Producer<T>() {

            private int index;

            @Override @Nullable public T
            produce() { return indexTransformer.transform(this.index++); }
        };
    }

    /**
     * Produces objects based on the number of preceding invocations, i.e. the {@code indexTransformer} is invoked
     * with subjects '0', '1', '2', ...
     */
    public static <T, EX extends Exception> ProducerWhichThrows<T, EX>
    fromIndexTransformer(final TransformerWhichThrows<Integer, T, EX> indexTransformer) {
        return new ProducerWhichThrows<T, EX>() {

            private int index;

            @Override @Nullable public T
            produce() throws EX { return indexTransformer.transform(this.index++); }
        };
    }

    /**
     * @return A producer which produces bytes through {@code new java.util.Random(seed).nextInt(0x100)}
     */
    public static Producer<Byte>
    randomByteProducer(final long seed) {

        return new Producer<Byte>() {

            final Random r = new Random(seed);

            @Override public Byte
            produce() { return (byte) this.r.nextInt(0x100); }
        };
    }

    /**
     * @return A producer which always produces the {@code constant}
     */
    public static <T> Producer<T>
    constantProducer(final T constant) {

        return new Producer<T>() {
            @Override public T produce() { return constant; }
        };
    }
}
