
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2011, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.lang.protocol;

import java.util.Collection;
import java.util.Map;

/**
 * Various {@link Predicate}-related utility methods.
 */
public final
class PredicateUtil {

    private
    PredicateUtil() {}

    /**
     * @return A {@link Predicate} who's {@link Predicate#evaluate(Object)} method always returns {@code true}.
     */
    @SuppressWarnings("unchecked") public static <T> Predicate<T>
    always() {
        return PredicateUtil.ALWAYS;
    }

    /**
     * @return A {@link Predicate} who's {@link Predicate#evaluate(Object)} method always returns {@code false}.
     */
    @SuppressWarnings("unchecked") public static <T> Predicate<T>
    never() {
        return PredicateUtil.NEVER;
    }

    @SuppressWarnings("rawtypes") private static final Predicate
    ALWAYS = new Predicate() {

        @Override public boolean
        evaluate(Object subject) { return true; }

        @Override public String
        toString() { return "ALWAYS"; }
    };

    @SuppressWarnings("rawtypes") private static final Predicate
    NEVER = new Predicate() {

        @Override public boolean
        evaluate(Object subject) { return false; }

        @Override public String
        toString() { return "NEVER"; }
    };

    /**
     * Returns a {@link Predicate} which returns {@code true} iff both {@code p1} and {@code p2} return {@code true}
     * for any given {@code subject}.
     * <p>
     * <b>Notice:</b> A new predicate object needs to be allocated only if neither of the arguments is {@link
     * PredicateUtil#never()} or {@link PredicateUtil#always()}.
     */
    @SuppressWarnings("unchecked") public static <T> Predicate<T>
    and(final Predicate<? super T> p1, final Predicate<? super T> p2) {
        if (p1 == PredicateUtil.NEVER || p2 == PredicateUtil.NEVER) return PredicateUtil.NEVER;
        if (p1 == PredicateUtil.ALWAYS) return (Predicate<T>) p2;
        if (p2 == PredicateUtil.ALWAYS) return (Predicate<T>) p1;

        return new Predicate<T>() {

            @Override public boolean
            evaluate(T subject) { return p1.evaluate(subject) && p2.evaluate(subject); }

            @Override public String
            toString() { return p1 + " && " + p2; }
        };
    }

    /**
     * @return A {@link Predicate} which returns {@code true} if {@code p1} or {@code p2} returns {@code true}
     *         for a given {@code subject}.
     */
    @SuppressWarnings("unchecked") public static <T> Predicate<T>
    or(final Predicate<? super T> p1, final Predicate<? super T> p2) {
        if (p1 == PredicateUtil.ALWAYS || p2 == PredicateUtil.ALWAYS) return PredicateUtil.ALWAYS;
        if (p1 == PredicateUtil.NEVER) return (Predicate<T>) p2;
        if (p2 == PredicateUtil.NEVER) return (Predicate<T>) p1;

        return new Predicate<T>() {

            @Override public boolean
            evaluate(T subject) { return p1.evaluate(subject) || p2.evaluate(subject); }

            @Override public String
            toString() { return p1 + " || " + p2; }
        };
    }

    /**
     * @return A {@link Predicate} which returns {@code true} iff the {@code delegate} returns {@code false} for any
     *         given {@code subject}.
     */
    public static <T> Predicate<T>
    not(final Predicate<T> delegate) {
        return new Predicate<T>() {

            @Override public boolean
            evaluate(T subject) { return !delegate.evaluate(subject); }

            @Override public String
            toString() { return "!" + delegate; }
        };
    }

    /**
     * @return A {@link Predicate} which returns {@code true} iff a given {@code subject} is less than the
     *         {@code other}.
     */
    public static <C extends Comparable<C>> Predicate<C>
    less(final C other) {
        return new Predicate<C>() {

            @Override public boolean
            evaluate(C subject) { return other.compareTo(subject) > 0; }

            @Override public String
            toString() { return "< " + other; }
        };
    }

    /**
     * @return A {@link Predicate} which returns {@code true} iff a given {@code subject} is less than or equal to the
     *         {@code other}.
     */
    public static <C extends Comparable<C>> Predicate<C>
    lessEqual(final C other) {
        return new Predicate<C>() {

            @Override public boolean
            evaluate(C subject) { return other.compareTo(subject) >= 0; }

            @Override public String
            toString() { return "<= " + other; }
        };
    }

    /**
     * @return A {@link Predicate} which returns {@code true} iff a given {@code subject} is greater than the
     *         {@code other}.
     */
    public static <C extends Comparable<C>> Predicate<C>
    greater(final C other) {
        return new Predicate<C>() {

            @Override public boolean
            evaluate(C subject) { return other.compareTo(subject) < 0; }

            @Override public String
            toString() { return "> " + other; }
        };
    }

    /**
     * @return A {@link Predicate} which returns {@code true} iff a given {@code subject} is greater than or equal to
     *         the {@code other}.
     */
    public static <C extends Comparable<C>> Predicate<C>
    greaterEqual(final C other) {
        return new Predicate<C>() {

            @Override public boolean
            evaluate(C subject) { return other.compareTo(subject) <= 0; }

            @Override public String
            toString() { return ">= " + other; }
        };
    }

    /**
     * @return A {@link Predicate} which returns {@code true} iff a given {@code subject} is (arithmetically) equal
     *         to the {@code other}.
     */
    public static <C extends Comparable<C>> Predicate<C>
    equal(final C other) {
        return new Predicate<C>() {

            @Override public boolean
            evaluate(C subject) { return other.compareTo(subject) == 0; }

            @Override public String
            toString() { return "== " + other; }
        };
    }

    /**
     * @return A {@link Predicate} which returns {@code true} iff a given {@code subject} is (arithmetically) not equal
     *         to the {@code other}.
     */
    public static <C extends Comparable<C>> Predicate<C>
    notEqual(final C other) {
        return new Predicate<C>() {

            @Override public boolean
            evaluate(C subject) { return other.compareTo(subject) != 0; }

            @Override public String
            toString() { return "!= " + other; }
        };
    }

    /**
     * @return {@code min <= subject <= max}, or, if {@code min > max}: {@code subject >= min || subject <= max}
     */
    public static <C extends Comparable<C>> Predicate<C>
    between(final C min, final C max) {
        return (
            max.compareTo(min) >= 0
            ? new Predicate<C>() {

                @Override public boolean
                evaluate(C subject) { return min.compareTo(subject) <= 0 && max.compareTo(subject) >= 0; }

                @Override public String
                toString() { return min + "-" + max; }
            }
            : new Predicate<C>() {

                @Override public boolean
                evaluate(C subject) { return min.compareTo(subject) >= 0 || max.compareTo(subject) <= 0; }

                @Override public String
                toString() { return min + "-" + max; }
            }
        );
    }

    /** Value equality */
    public static <T> Predicate<T>
    equals(final T other) {
        return new Predicate<T>() {

            @Override public boolean
            evaluate(T subject) { return other.equals(subject); }

            @Override public String
            toString() { return "eq " + other; }
        };
    }

    /** Value equality */
    public static <T> Predicate<T>
    notEquals(final T other) {
        return new Predicate<T>() {

            @Override public boolean
            evaluate(T subject) { return !other.equals(subject); }

            @Override public String
            toString() { return "nq " + other; }
        };
    }

    /**
     * Returns a {@code Predicate<Object>} that evaluates to {@code true} iff the current time is after the given
     * expiration time.
     * <p>
     * (The returned predicate ignores its {@code subject} argument.)
     */
    public static Predicate<Object>
    after(final long expirationTime) {
        return new Predicate<Object>() {

            @Override public boolean
            evaluate(Object subject) { return System.currentTimeMillis() > expirationTime; }
        };
    }

    /**
     * Returns a predicate that evaluates the {@code format} against the {@code delegate} after each "*" in the {@code
     * format} has been replaced with the subject.
     * with
     */
    public static <T extends Predicate<String>> T
    forString(final String format, final T delegate) {

        @SuppressWarnings("unchecked") T result = (T) new Predicate<String>() {

            @Override public boolean
            evaluate(String subject) { return delegate.evaluate(format.replace("*", subject)); }
        };

        return result;
    }

    /**
     * Converts a {@link Predicate} into a {@link PredicateWhichThrows}, which is possible iff the source's element
     * type is a subclass of the target's element type.
     *
     * @param <T>  The subject type
     * @param <EX> The target predicate's exception
     */
    public static <T, EX extends Throwable> PredicateWhichThrows<T, EX>
    asPredicateWhichThrows(final Predicate<? super T> source) {

        return new PredicateWhichThrows<T, EX>() {

            @Override public boolean
            evaluate(T subject) { return source.evaluate(subject); }
        };
    }

    /**
     * Converts a {@link PredicateWhichThrows} into a {@link Predicate}, which is possible iff the source's exception
     * is a subclass of {@link RuntimeException} and the source's element type is a subclass of the target's element
     * type.
     *
     * @param <T>  The predicate subject type
     * @param <EX> The source predicate's exception
     */
    public static <T, EX extends RuntimeException> Predicate<T>
    asPredicate(final PredicateWhichThrows<T, EX> source) {
        return new Predicate<T>() {

            @Override public boolean
            evaluate(T subject) { return source.evaluate(subject); }
        };
    }

    /**
     * @return A predicate which checks whether the given {@code collection} contains the {@code subject}
     */
    public static <T> Predicate<T>
    contains(final Collection<? extends T> collection) {
        return new Predicate<T>() {
            @Override public boolean evaluate(T subject) { return collection.contains(subject); }
        };
    }

    /**
     * @return A {@link Predicate} that evaluates to {@code true} iff the {@code map} contains a key equal to the
     *         predicate subject
     */
    public static <K> Predicate<K>
    containsKey(final Map<K, ?> map) {

        return new Predicate<K>() {
            @Override public boolean evaluate(K subject) { return map.containsKey(subject); }
        };
    }

    /**
     * @return A {@link Predicate} that evaluates to {@code true} iff the {@code mapping} contains a key equal to the
     *         predicate subject
     */
    public static <K> Predicate<K>
    containsKey(final Mapping<K, ?> mapping) {

        return new Predicate<K>() {
            @Override public boolean evaluate(K subject) { return mapping.containsKey(subject); }
        };
    }
}
