
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2011, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.io;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PipedReader;
import java.io.PipedWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.CRC32;
import java.util.zip.Checksum;

import de.unkrig.commons.lang.AssertionUtil;
import de.unkrig.commons.lang.ExceptionUtil;
import de.unkrig.commons.lang.protocol.ConsumerWhichThrows;
import de.unkrig.commons.lang.protocol.Producer;
import de.unkrig.commons.lang.protocol.ProducerUtil;
import de.unkrig.commons.lang.protocol.ProducerWhichThrows;
import de.unkrig.commons.lang.protocol.RunnableWhichThrows;
import de.unkrig.commons.nullanalysis.NotNullByDefault;
import de.unkrig.commons.nullanalysis.Nullable;

/**
 * Various {@code java.io}-related utility methods.
 */
public final
class IoUtil {

    static { AssertionUtil.enableAssertionsForThisClass(); }

    private static final Logger LOGGER = Logger.getLogger(IoUtil.class.getName());

    private
    IoUtil() {}

    /**
     * Reads the input stream until end-of-input and writes all data to the output stream. Closes neither of the two
     * streams.
     *
     * @return The number of bytes copied
     */
    public static long
    copy(InputStream inputStream, OutputStream outputStream) throws IOException {
        return IoUtil.copy(inputStream, false, outputStream, false);
    }

    /**
     * Reads at most {@code n} bytes from the input stream and writes all data to the output stream. Closes neither of
     * the two streams.
     *
     * @return The number of bytes copied
     */
    public static long
    copy(InputStream inputStream, OutputStream outputStream, long n) throws IOException {
        byte[] buffer = new byte[4096];
        long   count  = 0L;
        while (n > 0) {
            try {
                IoUtil.LOGGER.log(Level.FINEST, "About to ''read(byte[{0}])''", buffer.length);
                int m = inputStream.read(buffer, 0, (int) Math.min(n, buffer.length));
                IoUtil.LOGGER.log(Level.FINEST, "''read()'' returned {0}", m);
                if (m == -1) break;
                IoUtil.LOGGER.log(Level.FINEST, "About to ''write(byte[{0}])''", m);
                outputStream.write(buffer, 0, m);
                IoUtil.LOGGER.log(Level.FINEST, "'write()' returned");
                count += m;
                n     -= m;
            } catch (IOException ioe) {
                throw ExceptionUtil.wrap(count + " bytes copied so far", ioe);
            }
        }

        outputStream.flush();

        IoUtil.LOGGER.log(Level.FINEST, "{0} bytes copied", count);
        return count;
    }

    /**
     * Copies the contents of the {@code inputStream} to the {@code outputStream}.
     *
     * @param closeInputStream  Whether to close the {@code inputStream} (also if an {@link IOException} is thrown)
     * @param closeOutputStream Whether to close the {@code outputStream} (also if an {@link IOException} is thrown)
     * @return The number of bytes copied
     */
    public static long
    copy(InputStream inputStream, boolean closeInputStream, OutputStream outputStream, boolean closeOutputStream)
    throws IOException {

        try {
            final long count = IoUtil.copy(inputStream, outputStream, Long.MAX_VALUE);

            if (closeInputStream) inputStream.close();
            if (closeOutputStream) outputStream.close();

            return count;
        } catch (IOException ioe) {
            if (closeInputStream) try { inputStream.close(); } catch (Exception e) {}
            if (closeOutputStream) try { outputStream.close(); } catch (Exception e) {}
            throw ioe;
        }
    }

    /**
     * Creates and returns a {@link RunnableWhichThrows} that copies bytes from {@code in} to {@code out} until
     * end-of-input.
     */
    public static RunnableWhichThrows<IOException>
    copyRunnable(final InputStream in, final OutputStream out) {
        return new RunnableWhichThrows<IOException>() {

            @Override public void
            run() throws IOException {
                IoUtil.copy(in, out);
            }
        };
    }

    /**
     * Reads the reader until end-of-input and writes all data to the writer. Closes neither the reader nor the writer.
     *
     * @return The number of characters copied
     */
    public static long
    copy(Reader reader, Writer writer) throws IOException {
        return IoUtil.copy(reader, false, writer, false);
    }

    /**
     * Copies the contents of the {@code reader} to the {@code writer}.
     *
     * @return The number of characters copied
     */
    public static long
    copy(Reader reader, boolean closeReader, Writer writer, boolean closeWriter) throws IOException {

        char[] buffer = new char[4096];
        long   count  = 0L;
        try {
            for (;;) {
                IoUtil.LOGGER.log(Level.FINEST, "About to ''read(char[{0}])''", buffer.length);
                int n = reader.read(buffer);
                IoUtil.LOGGER.log(Level.FINEST, "''read()'' returned {0}", n);
                if (n == -1) break;
                IoUtil.LOGGER.log(Level.FINEST, "About to ''write(char[{0}])''", n);
                writer.write(buffer, 0, n);
                IoUtil.LOGGER.log(Level.FINEST, "'write()' returned");
            }
            writer.flush();
            if (closeReader) reader.close();
            if (closeWriter) writer.close();
            IoUtil.LOGGER.log(Level.FINEST, "{0} bytes copied", count);
            return count;
        } catch (IOException ioe) {
            if (closeReader) try { reader.close(); } catch (Exception e) {}
            if (closeWriter) try { writer.close(); } catch (Exception e) {}
            throw ExceptionUtil.wrap(count + " characters copied so far", ioe);
        }
    }

    /**
     * Reads the reader until end-of-input and writes all data to the output stream. Closes neither the reader nor the
     * output stream.
     *
     * @return The number of characters copied
     */
    public static long
    copy(Reader reader, OutputStream outputStream, Charset charset) throws IOException {
        return IoUtil.copy(reader, new OutputStreamWriter(outputStream, charset));
    }

    /**
     * Reads the {@link Readable} until end-of-input and writes all data to the {@link Appendable}.
     *
     * @return The number of characters copied
     */
    public static long
    copy(Readable r, Appendable a) throws IOException {

        CharBuffer cb    = CharBuffer.allocate(4096);
        long       count = 0;
        for (;;) {
            int n = r.read(cb);
            if (n == -1) break;
            cb.flip();
            a.append(cb);
            count += n;
            cb.clear();
        }
        return count;
    }

    /**
     * Copies the contents of the {@code inputStream} to the {@code outputFile}. Attempts to delete a partially written
     * output file if the operation fails.
     *
     * @return The number of bytes copied
     */
    public static long
    copy(InputStream inputStream, boolean closeInputStream, File outputFile, boolean append) throws IOException {

        try {
            return IoUtil.copy(inputStream, closeInputStream, new FileOutputStream(outputFile, append), true);
        } catch (IOException ioe) {
            outputFile.delete();
            throw ioe;
        } catch (RuntimeException re) {
            outputFile.delete();
            throw re;
        }
    }

    /**
     * Copies the contents of the {@code reader} to the {@code outputFile}, encoded with the given {@code
     * outputCharset}. Attempts to delete a partially written output file if the operation fails.
     *
     * @return The number of characters copied
     */
    public static long
    copy(Reader reader, boolean closeReader, File outputFile, boolean append, Charset outputCharset)
    throws IOException {
        try {
            long count = IoUtil.copy(
                reader,
                closeReader,
                new OutputStreamWriter(new FileOutputStream(outputFile, append), outputCharset),
                true
            );
            IoUtil.LOGGER.log(Level.FINEST, "{0} bytes copied", count);
            return count;
        } catch (IOException ioe) {
            outputFile.delete();
            throw ioe;
        } catch (RuntimeException re) {
            outputFile.delete();
            throw re;
        }
    }

    /**
     * Copies the contents of the {@code inputFile} to the {@code outputStream}.
     *
     * @return The number of bytes copied
     */
    public static long
    copy(File inputFile, OutputStream outputStream, boolean closeOutputStream) throws IOException {

        FileInputStream is;
        try {
            is = new FileInputStream(inputFile);
        } catch (IOException ioe) {
            if (closeOutputStream) try { outputStream.close(); } catch (Exception e) {}
            throw ioe;
        }

        return IoUtil.copy(is, true, outputStream, closeOutputStream);
    }

    /**
     * Copies the contents of the {@code inputStream} to the {@code outputFile}.
     *
     * @return The number of bytes copied
     */
    public static long
    copy(InputStream inputStream, boolean closeInputStream, File outputFile) throws IOException {

        OutputStream os = new FileOutputStream(outputFile);
        try {
            return IoUtil.copy(inputStream, closeInputStream, os, true);
        } catch (IOException ioe) {
            if (!outputFile.delete()) {
                throw new IOException("Cannot delete '" + outputFile + "'"); // SUPPRESS CHECKSTYLE AvoidHidingCause
            }
            throw ioe;
        }
    }

    /**
     * Copies the contents of the {@code inputFile} to the {@code outputFile}.
     *
     * @return The number of bytes copied
     */
    public static long
    copy(File inputFile, File outFile) throws IOException {

        return IoUtil.copy(new FileInputStream(inputFile), true, outFile);
    }

    /**
     * @return A {@code Comsumer<OutputStream>} which copies {@code is} to its subject
     */
    public static ConsumerWhichThrows<OutputStream, IOException>
    copyFrom(final InputStream is) {

        return new ConsumerWhichThrows<OutputStream, IOException>() {

            @Override public void
            consume(OutputStream os) throws IOException { IoUtil.copy(is, os); }
        };
    }

    /**
     * Creates and returns an {@link OutputStream} that delegates all work to the given {@code delegates}:
     * <ul>
     *   <li>
     *     The {@link OutputStream#write(byte[], int, int) write()} methods write the given data to all the delegates;
     *     if any of these throw an {@link IOException}, it is rethrown, and it is undefined whether all the data was
     *     written to all the delegates.
     *   </li>
     *   <li>
     *     {@link OutputStream#flush() flush()} flushes the delegates; throws the first {@link IOException} that any
     *     of the delegates throws.
     *   </li>
     *   <li>
     *     {@link OutputStream#close() close()} attempts to close <i>all</i> the {@code delegates}; if any of these
     *     throw {@link IOException}s, one of them is rethrown.
     *   </li>
     * </ul>
     */
    @NotNullByDefault(false) public static OutputStream
    tee(final OutputStream... delegates) {
        return new OutputStream() {

            @Override public void
            close() throws IOException {
                IOException caughtIOException = null;
                for (OutputStream delegate : delegates) {
                    try {
                        delegate.close();
                    } catch (IOException ioe) {
                        caughtIOException = ioe;
                    }
                }
                if (caughtIOException != null) throw caughtIOException;
            }

            @Override public void
            flush() throws IOException {
                for (OutputStream delegate : delegates) delegate.flush();
            }

            @Override public void
            write(byte[] b, int off, int len) throws IOException {
                // Overriding this method is not strictly necessary, because "OutputStream.write(byte[], int, int)"
                // calls "OutputStream.write(int)", but "delegate.write(byte[], int, int)" is probably more
                // efficient. However, the behavior is different when one of the delegates throws an exception
                // while being written to.
                for (OutputStream delegate : delegates) delegate.write(b, off, len);
            }

            @Override public void
            write(int b) throws IOException {
                for (OutputStream delegate : delegates) delegate.write(b);
            }
        };
    }

    /** @return The number of bytes that {@code writeContents} had written to its subject */
    public static long
    writeAndCount(ConsumerWhichThrows<OutputStream, IOException> writeContents, OutputStream os) throws IOException {

        CountingOutputStream cos  = new CountingOutputStream();

        writeContents.consume(IoUtil.tee(cos, os));

        return cos.getCount();
    }

    /**
     * An entity which writes characters to a {@link Writer}.
     */
    public
    interface WritingRunnable {

        /**
         * @see WritingRunnable
         */
        void
        run(Writer w) throws Exception;
    }

    private static final ExecutorService EXECUTOR_SERVICE = new ScheduledThreadPoolExecutor(
        3 * Runtime.getRuntime().availableProcessors()
    );

    /**
     * Executes the {@code writingRunnables} in parallel, <i>concatenates</i> their output, and writes it to the {@code
     * writer}, i.e. the output of the runnables does not mix, but the <i>complete</i> output of the first runnable
     * appears before that of the second runnable, and so on.
     * <p>
     * Since the character buffer for each {@link WritingRunnable} has a limited size, the runnables with higher
     * indexes tend to block if the runnables with lower indexes do not complete quickly enough.
     */
    public static void
    parallel(WritingRunnable[] writingRunnables, final Writer writer) {
        List<Callable<Void>> callables = IoUtil.toCallables(writingRunnables, writer);

        try {
            IoUtil.EXECUTOR_SERVICE.invokeAll(callables);
        } catch (InterruptedException ie) {
            Thread.currentThread().interrupt(); // Preserve interrupt status.
        }
    }

    /**
     * Creates and returns a list of callables; when all of these have been called, then all the given {@code
     * writingRunnables} have been run, and their output is written strictly sequentially to the given {@code writer},
     * even if the callables were called out-of-sequence or in parallel.
     * <p>
     * Deadlocks may occur if lower-index {@code writingRunnables} "depend" on higher-index {@code writingRunnables},
     * i.e. the former do not complete because they wait for a certain state of completion of the latter.
     */
    private static List<Callable<Void>>
    toCallables(WritingRunnable[] writingRunnables, final Writer writer) {
        List<Callable<Void>> callables = new ArrayList<Callable<Void>>(writingRunnables.length + 1);
        final List<Reader>   readers   = new ArrayList<Reader>(writingRunnables.length);

        // Create the 'collector' that concatenates the runnnables' outputs.
        callables.add(new Callable<Void>() {

            @Override @Nullable public Void
            call() throws Exception {
                for (Reader reader : readers) {
                    IoUtil.copy(reader, writer);
                }
                return null;
            }
        });

        for (final WritingRunnable wr : writingRunnables) {

            // Create a PipedReader/PipedWriter pair for communication between the runnable and the 'collector'.
            final PipedWriter pw = new PipedWriter();
            try {
                readers.add(new PipedReader(pw));
            } catch (IOException ioe) {
                throw ExceptionUtil.wrap(
                    "Should never throw an IOException if the argument is a 'fresh' PipedWriter",
                    ioe,
                    AssertionError.class
                );
            }

            // Create a callable that will run the runnable.
            callables.add(new Callable<Void>() {

                @Override @Nullable public Void
                call() throws Exception {
                    try {
                        wr.run(pw);
                        return null;
                    } catch (Exception e) {
                        IoUtil.LOGGER.log(Level.WARNING, null, e);
                        throw e;
                    } catch (Error e) { // SUPPRESS CHECKSTYLE Illegal Catch
                        IoUtil.LOGGER.log(Level.SEVERE, null, e);
                        throw e;
                    } finally {
                        try { pw.close(); } catch (Exception e) {}
                    }
                }
            });
        }
        return callables;
    }

    /**
     * @return All bytes that the given {@link InputStream} produces
     */
    public static byte[]
    readAll(InputStream is) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        IoUtil.copy(is, baos);
        return baos.toByteArray();
    }

    /**
     * @return All bytes that the given {@link InputStream} produces, decoded into a string
     */
    public static String
    readAll(InputStream inputStream, Charset charset, boolean closeInputStream) throws IOException {

        StringWriter sw = new StringWriter();
        IoUtil.copy(
            new InputStreamReader(inputStream, charset),
            closeInputStream,
            sw,
            false
        );
        return sw.toString();
    }

    /**
     * Creates and returns an {@link OutputStream} which writes at most {@code byteCountLimits.produce()} bytes to
     * {@code delegates.produce()} before closing it and writing the next {@code byteCountLimits.produce()} bytes to
     * {@code delegates.produce()}, and so on.
     *
     * @param delegates       Must produce a non-{@code null} series of {@link OutputStream}s
     * @param byteCountLimits Must produce a non-{@code null} series of {@link Long}s
     */
    public static OutputStream
    split(final ProducerWhichThrows<OutputStream, IOException> delegates, final Producer<Long> byteCountLimits)
    throws IOException {

        return new OutputStream() {

            /** Current delegate to write to. */
            private OutputStream delegate = AssertionUtil.notNull(delegates.produce(), "'delegates' produced <null>");

            /** Number of remaining bytes to be written. */
            private long delegateByteCount = AssertionUtil.notNull(
                byteCountLimits.produce(),
                "'byteCountLimits' produced <null>"
            );

            @Override public void
            write(int b) throws IOException { this.write(new byte[] { (byte) b }, 0, 1); }

            @Override public synchronized void
            write(@Nullable byte[] b, int off, int len) throws IOException {

                while (len > this.delegateByteCount) {
                    this.delegate.write(b, off, (int) this.delegateByteCount);
                    this.delegate.close();
                    off += this.delegateByteCount;
                    len -= this.delegateByteCount;

                    this.delegate = AssertionUtil.notNull(
                        delegates.produce(),
                        "'delegates' produced <null>"
                    );
                    this.delegateByteCount = AssertionUtil.notNull(
                        byteCountLimits.produce(),
                        "'byteCountLimits' produced <null>"
                    );
                }

                this.delegate.write(b, off, len);
                this.delegateByteCount -= len;
            }

            @Override public void
            flush() throws IOException { this.delegate.flush(); }

            @Override public void
            close() throws IOException { this.delegate.close(); }
        };
    }

    /** An {@link InputStream} that produces exactly 0 bytes. */
    public static final InputStream
    EMPTY_INPUT_STREAM = new InputStream() {
        @Override public int read()                                       { return -1; }
        @Override public int read(@Nullable byte[] buf, int off, int len) { return -1; }
    };

    /** An {@link OutputStream} that discards all bytes written to it. */
    public static final OutputStream
    NULL_OUTPUT_STREAM = new OutputStream() {
        @Override public void write(@Nullable byte[] b, int off, int len) {}
        @Override public void write(int b)                                {}
    };

    /**
     * @return An input stream that reads an endless stream bytes of value {@code b}.
     */
    public static InputStream
    constantInputStream(final byte b) {
        return new InputStream() {

            @Override public int
            read() { return 0; }

            @Override public int
            read(@Nullable byte[] buf, int off, int len) { Arrays.fill(buf,  off, len, b); return len; }
        };
    }

    /**
     * An input stream that reads an endless stream of zeros.
     */
    public static final InputStream
    ZERO_INPUT_STREAM = IoUtil.constantInputStream((byte) 0);

    /**
     * Writes {@code count} bytes of value {@code b} to the given output stream.
     */
    public static void
    fill(OutputStream outputStream, byte b, long count) throws IOException {

        if (count > 8192) {
            byte[] ba = new byte[8192];
            if (b != 0) Arrays.fill(ba, b);
            do {
                outputStream.write(ba);
                count -= 8192;
            } while (count > 8192);
        }

        byte[] ba = new byte[(int) count];
        Arrays.fill(ba, b);
        outputStream.write(ba);
    }

    /**
     * @return An input stream which reads the data produced by the {@code delegate} byte producer; {@code null}
     *         products are returned as 'end-of-input'
     */
    public static InputStream
    byteProducerInputStream(final ProducerWhichThrows<Byte, ? extends IOException> delegate) {

        return new InputStream() {

            @Override public int
            read() throws IOException {
                Byte b = delegate.produce();
                return b != null ? 0xff & b : -1;
            }
        };
    }

    /**
     * @return An input stream which reads the data produced by the {@code delegate} byte producer; {@code null}
     *         products are returned as 'end-of-input'
     */
    public static InputStream
    byteProducerInputStream(final Producer<Byte> delegate) {

        return IoUtil.byteProducerInputStream(ProducerUtil.<Byte, IOException>asProducerWhichThrows(delegate));
    }

    /**
     * @return An input stream which reads the data produced by the {@code delegate} byte producer; {@code null}
     *         products are returned as 'end-of-input'
     */
    public static InputStream
    randomInputStream(final long seed) {

        return IoUtil.byteProducerInputStream(ProducerUtil.randomByteProducer(seed));
    }

    /**
     * @return An output stream which feeds the data to the {@code delegate} byte consumer
     */
    public static OutputStream
    byteConsumerOutputStream(final ConsumerWhichThrows<Byte, IOException> delegate) {

        return new OutputStream() {

            @Override public void
            write(int b) throws IOException { delegate.consume((byte) b); }
        };
    }

    /**
     * @return All characters that the given {@link Reader} produces
     */
    public static String
    readAll(Reader r) throws IOException {
        char[]        chars = new char[4096];
        StringBuilder sb    = new StringBuilder();
        for (;;) {
            int n = r.read(chars);
            if (n == -1) break;
            sb.append(chars, 0, n);
        }
        return sb.toString();
    }

    /**
     * @return An {@link InputStream} which first closes the {@code delegate}, and then attempts to delete the {@code
     *         file}
     */
    protected static InputStream
    deleteOnClose(InputStream delegate, final File file) {

        return new FilterInputStream(delegate) {

            @Override public void
            close() throws IOException {
                super.close();
                file.delete();
            }
        };
    }

    /**
     * Creates and returns an array of {@code n} {@link OutputStream}s.
     * <p>
     * Iff exactly the same bytes are written to
     * all of these streams, and then all the streams are closed, then {@code whenIdentical} will be run (exactly once).
     * <p>
     * Otherwise, when the first non-identical byte is written to one of the streams, or at the latest when that stream
     * is closed, {@code whenNotIdentical} will be run (possibly more than once).
     */
    public static OutputStream[]
    compareOutput(final int n, final Runnable whenIdentical, final Runnable whenNotIdentical) {

        /**
         * Logs checksums of the first n1, n2, n3, ... bytes written.
         * <p>
         * This class is used to compare the date written to multiple output streams without storing the entire data
         * in memory.
         * <p>
         * n1, n2, n3, ... is an exponentially growing series, starting with a very small value.
         */
        abstract
        class ChecksumOutputStream extends OutputStream {

            /** The checksum of the bytes written to this stream so far. */
            private final Checksum checksum  = new CRC32();

            /** The number of bytes written to this stream so far. */
            private long count;

            /**
             * {@code checksums[i]} is the checksum of the first {@code THRESHOLD[i]} that were written to this stream.
             * <p>
             * After this stream was closed, {@code checksums[idx - 1]} is the checksum of <b>all</b> bytes that were
             * written to this stream.
             */
            protected final long[] checksums = new long[IoUtil.THRESHOLDS.length];

            /** The number of checksums in the {@link #checksums} array. */
            protected int idx;

            /**
             * Indicates that this stream is closed and that {@code checksums[idx - 1]} is the checksum of <b>all</b>
             * bytes that were written to this stream.
             */
            private boolean closed;

            @Override public void
            write(int b) throws IOException {
                if (this.closed) throw new IOException("Stream is closed");

                if (this.count == IoUtil.THRESHOLDS[this.idx]) this.pushChecksum();
                this.checksum.update(b);
                this.count++;
            }

            @Override public void
            write(@Nullable byte[] b, int off, int len) throws IOException {
                assert b != null;
                if (this.closed) throw new IOException("Stream is closed");

                while (this.count + len > IoUtil.THRESHOLDS[this.idx]) {
                    int part = (int) Math.min(Integer.MAX_VALUE, IoUtil.THRESHOLDS[this.idx] - this.count);
                    this.checksum.update(b, off, part);
                    this.count = IoUtil.THRESHOLDS[this.idx];
                    this.pushChecksum();
                    off += part;
                    len -= part;
                }

                this.checksum.update(b, off, len);
                this.count += len;
            }

            private void
            pushChecksum() {
                this.checksums[this.idx] = this.checksum.getValue();
                this.checksumWasPushed(this.idx);
                this.idx++;
            }

            /**
             * Is called when another checksum is entered in {@link #checksums}.
             *
             * @param idx The index in {@link #checksums} where the checksum was stored
             */
            abstract void checksumWasPushed(int idx);

            @Override public void
            close() {
                if (this.closed) return;
                this.pushChecksum();
                this.closed = true;
                this.wasClosed();
            }

            /**
             * Is called after this stream has been closed (for the first time).
             */
            abstract void wasClosed();
        }

        final ChecksumOutputStream[] result = new ChecksumOutputStream[n];
        for (int i = 0; i < n; i++) {
            result[i] = new ChecksumOutputStream() {

                @Override void
                checksumWasPushed(int idx) {
                    for (int i = 0; i < n; i++) {
                        if (result[i].idx == idx + 1 && result[i].checksums[idx] != this.checksums[idx]) {
                            whenNotIdentical.run();
                            return;
                        }
                    }
                }

                @Override void
                wasClosed() {
                    for (int i = 0; i < n; i++) {
                        if (!result[i].closed) return;
                        if (
                            result[i].idx != this.idx
                            || result[i].checksums[this.idx - 1] != this.checksums[this.idx - 1]
                        ) {
                            whenNotIdentical.run();
                            return;
                        }
                    }
                    whenIdentical.run();
                }
            };
        }

        return result;
    }
    private static final long[] THRESHOLDS;

    static {
        THRESHOLDS = new long[126];
        long x = 2;
        for (int i = 0; i < IoUtil.THRESHOLDS.length; x <<= 1) {
            IoUtil.THRESHOLDS[i++] = x;
            IoUtil.THRESHOLDS[i++] = x + (x >> 1);
        }
    }
}
