
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2011, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.file.filetransformation;

import java.io.File;
import java.io.IOException;

/**
 * @see #transform(String, File, File, Mode)
 */
public
interface FileTransformer {

    /**
     * @see FileTransformer#transform(String, File, File, Mode)
     * @see #TRANSFORM
     * @see #CHECK
     * @see #CHECK_AND_TRANSFORM
     */
    enum Mode {

        /** Execute the operation without previously checking if it actually changes any files. */
        TRANSFORM,

        /**
         * Execute the operation, but do not create or modify any files, and throw {@link #NOT_IDENTICAL} iff the
         * operation does not produce an identical result. Since {@link FileTransformer#transform(String, File, File,
         * Mode)} is typically much cheaper in this mode than in mode {@link #TRANSFORM}, it may be efficient to
         * execute the {@link FileTransformer} in this mode first to check whether the transformation would modify and
         * files, before executing it in {@link #TRANSFORM} mode, particularly if you do not expect any modifications.
         */
        CHECK,

        /**
         * Before executing the actual transformation, verify that it will actually modify any files.
         * <p>
         * Since checking whether a transformation would actually change any files is typically much cheaper than
         * the executing the actual transformation, this mode may be more efficient than {@link #TRANSFORM},
         * particularly if you expect few or no modifications.
         */
        CHECK_AND_TRANSFORM,
    }

    /**
     * Thrown by {@link #transform(String, File, File, Mode) transform}{@code (String, File, File, }{@link
     * Mode#CHECK}{@code )}; indicates that the output created by {@link #transform(String, File, File, Mode)
     * transform}{@code (String, File, File, }{@link Mode#TRANSFORM}{@code )} would be identical with the input.
     */
    RuntimeException
    NOT_IDENTICAL = new RuntimeException() {

        private static final long serialVersionUID = 1L;

        @Override public Throwable fillInStackTrace() { return this; }
    };

    /**
     * A {@link Runnable} that simply throws {@link #NOT_IDENTICAL}.
     */
    Runnable
    THROW_NOT_IDENTICAL = new Runnable() { @Override public void run() { throw FileTransformer.NOT_IDENTICAL; } };

    /**
     * Creates the file {@code out}, based on the file {@code in}. Iff {@code in.equals(out)} ('in-place
     * transformation'), then the original file remains unchanged, is modified, or replaced with a new file.
     * <p>
     * The precise contract is as follows:
     * <ul>
     *   <li>
     *     <b>If {@code mode == Mode.CHECK}</b>, then this method merely <i>checks</i> whether the file would be
     *     changed, and, if so, completes normally, otherwise it throws {@link #NOT_IDENTICAL}. (Parameter {@code out}
     *     is ignored.)
     *   </li>
     *   <li>
     *     <b>Otherwise, if {@code !in.equals(out)} ('out-of-place transformation')</b>, {@code out} is created, based
     *     on {@code in}. (Parameter {@code mode} is ignored.) (If this method throws an exception, then it must not
     *     leave a file {@code out} behind.)
     *   </li>
     *   <li>
     *     <b>Otherwise</b>, the file is left unchanged, is modified, or is replaced with a new file. If {@code mode ==
     *     Mode.CHECK_AND_TRANSFORM}, then the method attempts to avoid unnecessary i/o and processing by first checking
     *     whether the file requires any modifications before applying them. (If this method throws an exception, then
     *     it must revert the file to its original state as far as is reasonably possible.)
     *   /li>
     * </ul>
     *
     * @param  path             A text designating the input file; typically, but not necessarily identical with {@link
     *                          File#getPath() in.getPath()}
     * @throws RuntimeException {@link #NOT_IDENTICAL} iff {@code mode == }{@link Mode#CHECK} and the output produced
     *                          by {@link #transform(String, File, File, Mode)} would not be identical with the input
     * @see #NOT_IDENTICAL
     * @see Mode#TRANSFORM
     * @see Mode#CHECK
     * @see Mode#CHECK_AND_TRANSFORM
     */
    void
    transform(String path, File in, File out, Mode mode) throws IOException;
}
