
/*
 * de.unkrig.commons - A general-purpose Java class library
 *
 * Copyright (c) 2011, Arno Unkrig
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *       following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *       following disclaimer in the documentation and/or other materials provided with the distribution.
 *    3. The name of the author may not be used to endorse or promote products derived from this software without
 *       specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package de.unkrig.commons.file.filetransformation;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import de.unkrig.commons.file.FileUtil;
import de.unkrig.commons.file.contentstransformation.ContentsTransformer;
import de.unkrig.commons.io.IoUtil;
import de.unkrig.commons.io.MarkableFileInputStream;
import de.unkrig.commons.io.WyeInputStream;
import de.unkrig.commons.lang.AssertionUtil;
import de.unkrig.commons.lang.ExceptionUtil;
import de.unkrig.commons.lang.protocol.RunnableUtil;

/**
 * A {@link FileTransformer} that transforms a file be feeding its contents through a {@link ContentsTransformer}.
 */
public
class FileContentsTransformer implements FileTransformer {

    private final ContentsTransformer contentsTransformer;
    private final boolean             keepOriginals;

    public
    FileContentsTransformer(ContentsTransformer contentsTransformer, boolean keepOriginals) {
        this.contentsTransformer = contentsTransformer;
        this.keepOriginals       = keepOriginals;
    }

    /**
     * Opens the {@code in} file for reading, opens the {@code out} file for writing, reads the contents, feeds it
     * through the given {@link ContentsTransformer}, and writes it to the {@code out} file.
     */
    @Override public void
    transform(String path, File in, File out, Mode mode) throws IOException {

        boolean inPlace = in.equals(out);

        switch (mode) {

        case CHECK:
            {
                InputStream is = new MarkableFileInputStream(in);
                try {
                    FileContentsTransformer.checkIdentity(path, is, this.contentsTransformer);
                    is.close();
                    return;
                } finally {
                    try { is.close(); } catch (IOException ioe) {}
                }
            } // SUPPRESS CHECKSTYLE:FallThrough

        case CHECK_AND_TRANSFORM:
            {
                try {
                    this.transform(path, in, out, Mode.CHECK);
                } catch (RuntimeException re) {
                    if (re != FileTransformer.NOT_IDENTICAL) throw re;

                    // Changed contents.
                    this.transform(path, in, out, Mode.TRANSFORM);
                    return;
                }
                // Transformation produces identical file.

                if (inPlace) {

                    if (this.keepOriginals) {

                        // Output is identical with input, so no need to create a '.orig' copy.
                        ;
                    }
                } else {
                    IoUtil.copy(in, out);
                }
                return;
            }

        case TRANSFORM:
            {
                File newFile = FileTransformations.newFile(out);

                InputStream is = new MarkableFileInputStream(in);
                try {

                    OutputStream os = new FileOutputStream(newFile);
                    try {

                        this.contentsTransformer.transform(in.getPath(), is, os);
                        os.close();
                    } catch (IOException ioe) {
                        try { os.close(); } catch (Exception e) {}
                        newFile.delete();
                        throw ExceptionUtil.wrap("Transforming file '" + in + "' into '" + newFile + "'", ioe);
                    } catch (RuntimeException re) {
                        try { os.close(); } catch (Exception e) {}
                        newFile.delete();
                        throw ExceptionUtil.wrap("Transforming file '" + in + "' into '" + newFile + "'", re);
                    }
                    is.close();
                } finally {
                    try { is.close(); } catch (IOException ioe) {}
                }

                if (this.keepOriginals) {
                    File origFile = FileTransformations.origFile(in);
                    if (origFile.exists()) FileUtil.deleteRecursively(origFile);
                    try {
                        FileUtil.rename(in, origFile);
                    } catch (IOException ioe) {
                        FileUtil.attemptToDeleteRecursively(newFile);
                        throw ioe;
                    } catch (RuntimeException re) {
                        FileUtil.attemptToDeleteRecursively(newFile);
                        throw re;
                    }
                } else {
                    FileUtil.deleteRecursively(in);
                }

                FileUtil.rename(newFile, out);

                if (!out.setLastModified(in.lastModified())) {
                    throw new IOException("Could not set modification time of '" + out + "'");
                }
                return;
            }

        default:
            throw AssertionUtil.<Error>fail("Unexpected mode '" + mode + "'");
        }
    }

    /**
     * Consumes the {@code inputStream} and feeds it through the {@code contentsTransformer}.
     *
     * @throws RuntimeException {@link FileTransformer#NOT_IDENTICAL} iff the transformer contents differs from the
     *                          original contents
     */
    public static void
    checkIdentity(String path, InputStream inputStream, ContentsTransformer contentsTransformer) throws IOException {

        OutputStream[] oss = IoUtil.compareOutput(2, RunnableUtil.NOP, FileTransformer.THROW_NOT_IDENTICAL);

        contentsTransformer.transform(path, new WyeInputStream(inputStream, oss[0]), oss[1]);

        oss[0].close();
        oss[1].close();
        inputStream.close();
    }
}
