package de.uniks.networkparser.graph;

import de.uniks.networkparser.buffer.CharacterBuffer;
import de.uniks.networkparser.graph.ClazzType;
import de.uniks.networkparser.list.SimpleList;
import de.uniks.networkparser.list.SimpleSet;

/*
NetworkParser
Copyright (c) 2011 - 2015, Stefan Lindel
All rights reserved.

Licensed under the EUPL, Version 1.1 or (as soon they
will be approved by the European Commission) subsequent
versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at:

http://ec.europa.eu/idabc/eupl5

Unless required by applicable law or agreed to in
writing, software distributed under the Licence is
distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
express or implied.
See the Licence for the specific language governing
permissions and limitations under the Licence.
*/
/**
 * Special Util for package Method
 * @author Stefan Lindel
 */
public class GraphUtil {
	public static final Clazz getByObject(GraphEntity item, String clazz, boolean fullName) {
		if(clazz == null) {
			return null;
		}
		return (Clazz) item.getByObject(clazz, fullName);
	}
	public static final SimpleSet<Annotation> getAnnotations(GraphMember item) {
		if(item == null) {
			return null;
		}
		SimpleSet<Annotation> collection = new SimpleSet<Annotation>();
		Annotation annotation = null;
		if(item instanceof Clazz) {
			annotation = ((Clazz)item).getAnnotation();
		}
		if(item instanceof Attribute) {
			annotation = ((Attribute)item).getAnnotation();
		}
		if(item instanceof Annotation) {
			annotation = (Annotation) item;
		}
		if(annotation != null) {
			collection.add(annotation);
			while(annotation.hasNext()) {
				annotation = annotation.next();
				collection.add(annotation);
			}
		}
		return collection;
	}

	public static final boolean isWithNoObjects(Clazz clazz) {
		if(clazz == null) {
			return false;
		}
		return (clazz.getModifier().has(Modifier.ABSTRACT) || clazz.getType() == ClazzType.INTERFACE);
	}
	public static final boolean isInterface(Clazz clazz) {
		if(clazz == null) {
			return false;
		}
		return clazz.getType() == ClazzType.INTERFACE;
	}
	public static final boolean isEnumeration(Clazz clazz) {
		if(clazz == null) {
			return false;
		}
		return clazz.getType() == ClazzType.ENUMERATION;
	}
	public static final boolean isUndirectional(Association assoc) {
		if(assoc == null) {
			return false;
		}
		if((assoc.getType()==AssociationTypes.ASSOCIATION || assoc.getType()==AssociationTypes.UNDIRECTIONAL ) && assoc.getOtherType()==AssociationTypes.EDGE) {
			return true;
		}
		return (assoc.getOtherType()==AssociationTypes.ASSOCIATION || assoc.getOtherType()==AssociationTypes.UNDIRECTIONAL) && assoc.getType()==AssociationTypes.EDGE;
	}

	public static final boolean isInterfaceAssociation(Association assoc) {
		if(assoc == null) {
			return false;
		}
		if(assoc.getType()==AssociationTypes.IMPLEMENTS && assoc.getOtherType()==AssociationTypes.EDGE) {
			return true;
		}
		return assoc.getOtherType()==AssociationTypes.IMPLEMENTS && assoc.getType()==AssociationTypes.EDGE;
	}

	public static final CharacterBuffer getMethodParameters(Method method, boolean shortName) {
		return method.getParameterString(shortName, false);
	}

	public static final SimpleSet<Association> getOtherAssociations(Clazz clazz) {
		SimpleSet<Association> collection = new SimpleSet<Association>();
		for(Association assoc : clazz.getAssociations()) {
			collection.add(assoc.getOther());
		}
		return collection;
	}

	public static final GraphSimpleSet getChildren(GraphMember item) {
		return item.getChildren();
	}
	public static final String getSeperator(Association item) {
		return item.getSeperator();
	}
	public static final SimpleSet<GraphEntity> getNodes(GraphMember item) {
		return item.getNodes();
	}
	public static final GraphDiff getDifference(GraphMember item) {
		return item.getDiff();
	}

	public static final void removeYou(GraphMember value) {
		if(value == null) {
			return;
		}
		value.setParentNode(null);
		if(value instanceof Attribute) {
			Attribute attribute = (Attribute) value;
			Annotation annotation = attribute.getAnnotation();
			value.without(annotation);
		}
		if(value instanceof Association) {
			Association assoc = (Association) value;
			assoc.withOther(null);
			assoc.without(assoc.getClazz());
		}
		if(value instanceof Clazz) {
			Clazz clazz = (Clazz) value;
			GraphSimpleSet collection = clazz.getChildren();
			clazz.without(collection.toArray(new GraphMember[collection.size()]));
		}
	}

	public static final boolean containsClazzAssociation(SimpleList<GraphMember> visited, Association assoc, Association other) {
		boolean foundAssoc = false;
		for(GraphMember checkItem : visited) {
			if(checkItem instanceof Association == false || checkItem.getName() == null) {
				continue;
			}
			Association assocA = (Association) checkItem;
			Association assocB = assocA.getOther();
			if(assocB.getName() == null) {
				continue;
			}
			if(assocA.getName().equals(assoc.getName())) {
				if(assocB.getName().equals(other.getName())) {
					//Found Link ??
					foundAssoc = true;
					if(assocA.getClazz() == assoc.getClazz()) {
						if(assocB.getClazz() == other.getClazz()) {
							// May be n-m
							assocA.with(Cardinality.MANY);
							assocB.with(Cardinality.MANY);
						} else {
							assocA.with(Cardinality.MANY);
						}
					} else {
						assocB.with(Cardinality.MANY);
					}
					break;
				}
			}
		}
		return foundAssoc;
	}
	public static final String getShortAssoc(Association assoc) {
		if( assoc == null) {
			return "";
		}
		CharacterBuffer sb=new CharacterBuffer();
		Clazz clazz = assoc.getClazz();
		if(clazz != null) {
			sb.with(clazz.getName(true));
		}
		sb.with(':');
		sb.with(assoc.getName());
		sb.with("_");
		sb.with(assoc.getCardinality().getValue());
		sb.with(assoc.getSeperator());
		assoc = assoc.getOther();
		if(assoc != null) {
			clazz = assoc.getClazz();
			if(clazz != null) {
				sb.with(clazz.getName(true));
			}
			sb.with(':');
			sb.with(assoc.getName());
			sb.with("_");
			sb.with(assoc.getCardinality().getValue());
		}
		return sb.toString();
	}
	
	public static final GraphModel getGraphModel(GraphMember member) {
		if(member instanceof GraphModel) {
			return (GraphModel) member;
		}
		Object parent = member.getParent();
		if(parent instanceof GraphMember) {
			return getGraphModel((GraphMember)parent);
		} else if(parent instanceof GraphSimpleSet) {
			GraphSimpleSet list = (GraphSimpleSet) parent;
			if(list.size()>0) {
				return getGraphModel(list.first());
			}
		}
		return null;
	}
	public static final Clazz getParentClazz(GraphMember member) {
		if(member instanceof Clazz) {
			return (Clazz) member;
		}
		Object parent = member.getParent();
		if(parent instanceof GraphMember) {
			return getParentClazz((GraphMember)parent);
		} else if(parent instanceof GraphSimpleSet) {
			GraphSimpleSet list = (GraphSimpleSet) parent;
			if(list.size()>0) {
				return getParentClazz(list.first());
			}
		}
		return null;
	}
}
