package de.uni_trier.wi2.procake.gui.similarity.cache;

import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.gui.DataObjectLabelProvider;
import de.uni_trier.wi2.procake.similarity.impl.SimCacheListener;
import de.uni_trier.wi2.procake.similarity.impl.SimilarityCacheKey;
import de.uni_trier.wi2.procake.similarity.impl.SimilarityCacheValue;
import de.uni_trier.wi2.procake.similarity.wf.AbstractSimilarityCache;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SimilarityCacheMonitor extends DataObjectLabelProvider implements SimCacheListener {


  private static final Logger logger = LoggerFactory.getLogger(SimilarityCacheMonitor.class);
  public MonitorFrame monitor; // GUI component to display the similarity cache

  /**
   * Mutex for closing window, used in assigning the {@link WindowAdapter}
   */
  private static final Object windowCloseLock = new Object();

  public SimilarityCacheMonitor(AbstractSimilarityCache similarityCache) {
    if (similarityCache == null) {
      throw new IllegalArgumentException("SimilarityCache must not be null");
    }
    monitor = new MonitorFrame(similarityCache);
    similarityCache.addListener(this);

    monitor.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        synchronized (windowCloseLock) {
          JFrame frame = ((JFrame) e.getComponent());
          frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
          monitor.setVisible(false);
          monitor.dispose();
          windowCloseLock.notify();
        }
      }
    });
  }

  @Override
  public void update(SimilarityCacheKey key, SimilarityCacheValue value,
      NotificationType notificationType) {

    switch (notificationType) {
      case ADD -> monitor.insertRow(key, value);
      case UPDATE, READ -> monitor.updateRow(key, value, notificationType);
      case CLEAR -> monitor.clearTable();
      case REMOVE -> monitor.removeRow(key);
    }
  }

  @Override
  public String getLabel(Object object) {
    if (object instanceof DataObject) {
      return ((DataObject) object).getDataClass().getName() + ": " + super.getLabel(object);
    }

    return super.getLabel(object);
  }

  @SuppressWarnings({"BusyWait"})
  public void waitUntilWindowClosed() {
    try {
      while (!monitor.isVisible()) {
        Thread.sleep(100);
      }
    } catch (InterruptedException e) {
      logger.error("Exception: ", e);
    }
    synchronized (windowCloseLock) {
      while (monitor.isVisible()) {
        try {
          windowCloseLock.wait();
        } catch (InterruptedException e) {
          logger.error("Exception: ", e);
        }
      }
    }
  }
}