package de.uni_trier.wi2.procake.gui.similarity.cache;

import static javax.swing.ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER;
import static javax.swing.ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS;

import de.uni_trier.wi2.procake.similarity.Similarity;
import de.uni_trier.wi2.procake.similarity.SimilarityCache;
import de.uni_trier.wi2.procake.similarity.impl.SimCacheListener.NotificationType;
import de.uni_trier.wi2.procake.similarity.impl.SimilarityCacheKey;
import de.uni_trier.wi2.procake.similarity.impl.SimilarityCacheValue;
import de.uni_trier.wi2.procake.similarity.wf.AbstractSimilarityCache;
import de.uni_trier.wi2.procake.utils.Formatter;
import java.awt.*;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.List;
import java.util.Vector;
import javax.swing.*;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import org.apache.commons.collections4.map.ListOrderedMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MonitorFrame extends JFrame {

  private final SimilarityCache similarityCache;//used to access similarityCache.getLastTimeNanosForGetSimilarity()
  private static final Logger logger = LoggerFactory.getLogger(MonitorFrame.class);
  private final JTable monitorTable;
  private final JLabel showSizeLabel;//shows the current Cache Size, default set to "0"
  private final JLabel showLastTimeReadSimilarityValueLabel;//Value Label when READ action performed in cache, default: "none"
  public final ListOrderedMap<SimilarityCacheKey, Integer> indexMap; //link RowIndex to DataObjectPair
  private final DefaultTableModel model;
  private final DateFormat df = new SimpleDateFormat("HH:mm:ss:SSS yyyy-MM-dd");

  //constructor for MonitorFrame class to initialize the frame and table with the cache entries
  public MonitorFrame(AbstractSimilarityCache similarityCache) throws HeadlessException {
    this.similarityCache = similarityCache;
    this.indexMap = new ListOrderedMap<>();
    //initialize the table model with column names and types
    this.model = new DefaultTableModel(0, 0) {
      final Class<?>[] columnClass = {
          Integer.class, // index
          String.class, // query
          String.class, // case
          Similarity.class, // similarity
          Long.class, // comp. time
          String.class, // added
          Integer.class, // hits
          String.class // last time read
      };

      //get the column class for the given column index
      @Override
      public Class<?> getColumnClass(int columnIndex) {
        return this.columnClass[columnIndex];
      }

    };
    //add the column headers to the table model
    List<String> columnHeaderNames = List.of("index", "query", "case", "similarity", "comp. time",
        "added", "hits", "last time read");
    for (String s : columnHeaderNames) {
      this.model.addColumn(s);
    }
    //initialize the table with the table model
    this.monitorTable = new JTable(this.model);
    //set the column widths for the table
    columnHeaderNames.forEach(s -> this.monitorTable.getColumn(s).setPreferredWidth(110));
    //set the cell renderer for the similarity column to show the similarity object as a string in the table cell
    this.monitorTable.getColumnModel().getColumn(4).setCellRenderer(
        new DefaultTableCellRenderer() {
          @Override
          public Component getTableCellRendererComponent(JTable table, Object value,
              boolean isSelected, boolean hasFocus, int row, int column) {
            final Component result = super.getTableCellRendererComponent(table, value,
                isSelected,
                hasFocus, row,
                column);
            if (value instanceof Long) {
              setText(Formatter.nsToDetailedString((Long) value));
            }
            return result;
          }
        }
    );

    this.monitorTable.setAutoCreateRowSorter(true);
    //set the title and icon for the frame window
    this.setTitle("SimilarityCacheMonitor");
    this.setIconImage(
        Toolkit.getDefaultToolkit().getImage(getClass().getResource("/images/ProCAKE_Icon.png")));

    JScrollPane monitorScrollPane = new JScrollPane(this.monitorTable);
    monitorScrollPane.setVerticalScrollBarPolicy(VERTICAL_SCROLLBAR_ALWAYS);
    monitorScrollPane.setHorizontalScrollBarPolicy(HORIZONTAL_SCROLLBAR_NEVER);
    monitorScrollPane.setViewportView(monitorTable);
    this.getContentPane().add(monitorScrollPane, BorderLayout.CENTER);

    //infoPanel for SizeLabel and LastTimeReadSimilarityValueLabel
    JPanel infoPanel = new JPanel();
    infoPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
    showSizeLabel = new JLabel("0");//label
    infoPanel.add(new JLabel("Size: "));//label
    infoPanel.add(showSizeLabel);
    showLastTimeReadSimilarityValueLabel = new JLabel("-");
    infoPanel.add(new JLabel("Last time read similarity:"));
    infoPanel.add(showLastTimeReadSimilarityValueLabel);
    this.getContentPane().add(infoPanel, BorderLayout.SOUTH);
    //set up the frame properties and make it visible
    this.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    this.setMinimumSize(
        new Dimension(770,
            48));// minimum width for all column titles to fully show without "..."
    this.setLocationRelativeTo(null);
    this.pack();
    this.setVisible(true);
    //add MouseListener to show detailed similarity on double click
    {
      this.monitorTable.addMouseListener(new MouseAdapter() {
        @Override
        public void mousePressed(MouseEvent e) {
          if (e.getClickCount() != 2) {
            return;
          }

          Point clickedPoint = e.getPoint();
          int column = monitorTable.columnAtPoint(clickedPoint);
          if (column != 3) {
            return;
          }
          int row = monitorTable.rowAtPoint(clickedPoint);

          JFrame dialog = new JFrame();
          dialog.setIconImage(Toolkit.getDefaultToolkit()
              .getImage(getClass().getResource("/images/ProCAKE_Icon.png")));
          dialog.setTitle("Similarity");
          dialog.setLocationRelativeTo(null);
          dialog.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
          dialog.getContentPane().add(
              new JScrollPane(new JTextArea(
                  ((Similarity) monitorTable.getValueAt(row, column)).toDetailedString())));
          dialog.setMinimumSize(new Dimension(600, 600));
          dialog.pack();
          dialog.setVisible(true);
        }
      });

    }

    //initialize table with cache entries
    tableInit(similarityCache);

  }

  /**
   * updates the SizeLabel with the current Cache Size
   */
  private void updateVisualSize() {

    this.showSizeLabel.setText(this.model.getRowCount() + "  ");
    this.repaint();

  }

  //initialize the table with the cache entries
  private void tableInit(AbstractSimilarityCache similarityCache) {

    synchronized (this) {
      SwingUtilities.invokeLater(() -> {
        Collection<SimilarityCacheKey> keys = similarityCache.getKeys();
        for (SimilarityCacheKey key : keys) {
          SimilarityCacheValue value = similarityCache.get(key);

          this.model.insertRow(0, buildRow(key, value));
          indexMap.put(key, -1);
          for (int i = indexMap.size() - 1; i >= 0; i--) {
            indexMap.setValue(i, indexMap.size() - i - 1);
            this.model.setValueAt(i, i, 0);
          }
        }

        updateVisualSize();
      });
    }
  }


  //insert the row in the table at the top of the table
  public void insertRow(SimilarityCacheKey key, SimilarityCacheValue value) {
    synchronized (this) {
      SwingUtilities.invokeLater(() -> {
        this.model.insertRow(0, buildRow(key, value));
        indexMap.put(key, -1);
        for (int i = indexMap.size() - 1; i >= 0; i--) {
          indexMap.setValue(i, indexMap.size() - i - 1);
          this.model.setValueAt(i, i, 0);
        }
        updateVisualSize();
      });
    }
  }

  //constructing the row for the table as a Vector
  private Vector<Object> buildRow(SimilarityCacheKey key, SimilarityCacheValue value) {
    Vector<Object> row = new Vector<>(8);
    row.add(0);//col 0
    row.add(key.getQueryId());//col 1
    row.add(key.getCaseId());
    row.add(value.getSimilarity());
    row.add(value.getSimilarity() != null ? value.getSimilarity().getComputationTimeNanos() : null);
    row.add(value.getAdded() != null ? df.format(value.getAdded()) : "");
    row.add(value.getHits());
    row.add(value.getLastTimeRead() != null ? df.format(value.getLastTimeRead()) : "");//col 7
    //total fields per row = 8
    return row;
  }

  //locate the row in the table
  private int locateRow(SimilarityCacheKey key) {
    return indexMap.get(key) != null ? indexMap.get(key) : -1;

  }

  //remove the row from the table
  public void removeRow(SimilarityCacheKey key) {
    synchronized (this) {
      SwingUtilities.invokeLater(() -> {
        if (!indexMap.containsKey(key)) {
          logger.warn("key not found in indexMap");
          return;
        }

        int targetRowIndex = indexMap.remove(key);
        this.model.removeRow(targetRowIndex);
        for (int i = indexMap.size() - 1; i >= 0; i--) {
          indexMap.setValue(i, indexMap.size() - i - 1);
          this.model.setValueAt(i, i, 0);
        }
        updateVisualSize();
      });
    }
  }

  //update the row in the table
  public void updateRow(SimilarityCacheKey key, SimilarityCacheValue value,
      NotificationType notificationType) {
    synchronized (this) {
      SwingUtilities.invokeLater(() -> {
        if (notificationType == NotificationType.READ) {
          //update Label to show last time For get Similarity
          this.showLastTimeReadSimilarityValueLabel.setText(
              Formatter.nsToDetailedString(
                  this.similarityCache.getLastTimeNanosForGetSimilarity()));
        }

        logger.trace("UpdateRow(" + notificationType.toString() + ") @ " + locateRow(key) + " \n");
        if (!indexMap.containsKey(key)) {
          logger.warn("No key found in indexMap ");
          return;

        }
        Integer rowLocation = indexMap.remove(key);

        if (rowLocation != null) {
          this.model.removeRow(rowLocation);
          //update index
          for (int i = indexMap.size() - 1; i >= 0; i--) {
      /*Think help: Map fills top to bottom with values large to small
       e.g. Map [0](first/the oldest element) gets value for last Row number in table */
            indexMap.setValue(i, indexMap.size() - i);
            this.model.setValueAt(i + 1, i, 0);

          }
          this.model.insertRow(0, buildRow(key, value));
          indexMap.put(indexMap.size(), key, 0);
          updateVisualSize();
        } else {

          logger.warn(
              "Called updateRow(): Key not found in indexMap.\n locateRow(key)= " + locateRow(key)
                  + "\n indexMap.get(key)= " + indexMap.get(key));
        }
      });
    }
  }

  //clear the table
  public void clearTable() {
    synchronized (this) {
      SwingUtilities.invokeLater(() -> {
        if (this.model.getRowCount() > 0) {

          for (int i = this.model.getRowCount() - 1; i >= 0; i--) {
            this.model.removeRow(i);
            updateVisualSize();
          }
          indexMap.clear();
          updateVisualSize();
          logger.trace("Table cleared");
        }
      });
    }
  }

  //getter methods
  public DefaultTableModel getModel() {
    return model;
  }

  /**
   * @return <code>String</code> initial value= "0"
   * get SizeLabels current value
   */

  public String getSizeLabelValue() {
    return showSizeLabel.getText();
  }

  /**
   * @return <code>String</code> initial value= "-" otherwise a
   * <code>cache.getSimilarity(...)</code> Operation has been executed and Text will be formatted
   * like so:
   * <code>
   * Formatter.nsToDetailedString(this.similarityCache.getLastTimeNanosForGetSimilarity())</code>
   * get LastTimeRead current value
   **/
  public String getLastTimeReadSimilarityValueLabel() {
    return showLastTimeReadSimilarityValueLabel.getText();
  }
}

