package de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.utils;

import de.uni_trier.wi2.procake.data.model.DataClass;
import de.uni_trier.wi2.procake.data.model.ModelFactory;
import de.uni_trier.wi2.procake.data.model.nest.NESTConstraintEdgeClass;
import de.uni_trier.wi2.procake.data.model.nest.NESTControlflowEdgeClass;
import de.uni_trier.wi2.procake.data.model.nest.NESTDataflowEdgeClass;
import de.uni_trier.wi2.procake.data.model.nest.NESTPartOfEdgeClass;
import de.uni_trier.wi2.procake.data.model.nest.NESTWorkflowClass;
import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.object.base.AggregateObject;
import de.uni_trier.wi2.procake.data.object.base.AtomicObject;
import de.uni_trier.wi2.procake.data.object.base.CollectionObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTEdgeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTGraphItemObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTWorkflowObject;
import de.uni_trier.wi2.procake.data.object.nest.controlflowNode.NESTControlflowNodeObject;
import de.uni_trier.wi2.procake.data.objectpool.ReadableObjectPool;
import java.awt.*;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import javax.swing.*;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;

public class Utils {

  public static String aggregateToPaddedTable(
      AggregateObject aggregateObject, boolean includeNullAttributes) {
    return Utils.aggregateToPaddedTable(aggregateObject, includeNullAttributes, 0);
  }

  public static String aggregateToPaddedTable(
      AggregateObject aggregateObject, boolean includeNullAttributes, int offset) {
    List<String> attributeNames =
        aggregateObject.getAggregateClass().getAttributeNames().stream()
            .filter(
                attributeName ->
                    includeNullAttributes
                        || aggregateObject.getAttributeValue(attributeName) != null)
            .collect(Collectors.toList());
    int maxAttributeNameLength =
        attributeNames.stream().max(Comparator.comparing(String::length)).orElse("").length();
    int paddingLength = maxAttributeNameLength + offset;
    List<String> lines =
        IntStream.range(0, attributeNames.size())
            .mapToObj(
                i -> {
                  String attributeName = attributeNames.get(i);
                  DataObject attributeValue = aggregateObject.getAttributeValue(attributeName);
                  String line =
                      StringUtils.leftPad(
                          attributeName, i == 0 ? maxAttributeNameLength : paddingLength)
                          + ": ";
                  if (attributeValue == null) {
                    line += "null";
                  } else if (attributeValue.isAggregate()) {
                    line +=
                        Utils.aggregateToPaddedTable(
                            (AggregateObject) attributeValue,
                            includeNullAttributes,
                            paddingLength + 2);
                  } else if (attributeValue.isAtomic()) {
                    line += ((AtomicObject) attributeValue).getNativeObject().toString();
                  } else {
                    line += Objects.toString(attributeValue);
                  }
                  return line;
                })
            .collect(Collectors.toList());
    return lines.stream().collect(Collectors.joining("\n"));
  }

  // TODO: use ClassType instead of predicates?
  public static String getEdgeClassForNodeConnection(
      NESTNodeObject startNode, NESTNodeObject endNode) {
    List<ImmutablePair<BiPredicate<NESTNodeObject, NESTNodeObject>, String>> rules =
        Arrays.asList(
            // subworkflow -> x
            new ImmutablePair<>(
                (a, b) -> a.isNESTSubWorkflowNode() && b.isNESTWorkflowNode(),
                NESTPartOfEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTSubWorkflowNode() && b.isNESTSubWorkflowNode(),
                NESTPartOfEdgeClass.CLASS_NAME),

            // task -> x
            new ImmutablePair<>(
                (a, b) -> a.isNESTTaskNode() && b.isNESTWorkflowNode(),
                NESTPartOfEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTTaskNode() && b.isNESTSubWorkflowNode(),
                NESTPartOfEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTTaskNode() && b.isNESTTaskNode(),
                NESTControlflowEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTTaskNode() && b.isNESTControlflowNode(),
                NESTControlflowEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTTaskNode() && b.isNESTDataNode(),
                NESTDataflowEdgeClass.CLASS_NAME),

            // data -> x
            new ImmutablePair<>(
                (a, b) -> a.isNESTDataNode() && b.isNESTWorkflowNode(),
                NESTPartOfEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTDataNode() && b.isNESTSubWorkflowNode(),
                NESTPartOfEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTDataNode() && b.isNESTTaskNode(),
                NESTDataflowEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTDataNode() && b.isNESTDataNode(),
                NESTConstraintEdgeClass.CLASS_NAME),

            // controlflow -> x
            new ImmutablePair<>(
                (a, b) -> a.isNESTControlflowNode() && b.isNESTWorkflowNode(),
                NESTPartOfEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTControlflowNode() && b.isNESTSubWorkflowNode(),
                NESTPartOfEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTControlflowNode() && b.isNESTTaskNode(),
                NESTControlflowEdgeClass.CLASS_NAME),
            new ImmutablePair<>(
                (a, b) -> a.isNESTControlflowNode() && b.isNESTControlflowNode(),
                NESTControlflowEdgeClass.CLASS_NAME));

    var result = rules.stream().filter(rule -> rule.getLeft().test(startNode, endNode)).findAny();
    return result.map(ImmutablePair::getRight).orElse(null);
  }

  public static boolean isEdgeLoopReturnEdge(NESTEdgeObject edge) {
    if (edge.isNESTControlflowEdge()
        && edge.getPre() != null
        && edge.getPre().isNESTControlflowNode()) {
      NESTControlflowNodeObject pre = (NESTControlflowNodeObject) edge.getPre();
      return pre.isLoopNode()
          && pre.isEndControlflowNode()
          && edge.getPost() == pre.getMatchingBlockControlflowNode();
    }
    return false;
  }

  public static List<DataClass> getSubClassesDeep(DataClass dataClass) {
    List<DataClass> subClasses = new LinkedList<>(dataClass.getSubClasses());
    List<DataClass> subSubClasses = new LinkedList<>();
    subClasses.forEach(subClass -> subSubClasses.addAll(getSubClassesDeep(subClass)));
    subClasses.addAll(subSubClasses);
    return subClasses;
  }

  public static List<NESTWorkflowClass> getNESTWorkflowClasses() {
    return ModelFactory.getDefaultModel().getClasses().stream()
        .filter(DataClass::isNESTWorkflow)
        .map(NESTWorkflowClass.class::cast)
        .collect(Collectors.toList());
  }

  public static Set<NESTWorkflowObject> extractNESTWorkflowObjectsFromPool(
      ReadableObjectPool<? extends DataObject> pool) {
    Set<NESTWorkflowObject> result = new HashSet<>();
    pool.forEach(dataObject -> result.addAll(extractNESTWorkflowObjectsFromPool(dataObject)));
    return result;
  }

  private static Set<NESTWorkflowObject> extractNESTWorkflowObjectsFromPool(DataObject dataObject) {
    Set<NESTWorkflowObject> result = new HashSet<>();
    if (dataObject == null) {
      return result;
    }
    if (dataObject.isNESTWorkflow()) {
      result.add((NESTWorkflowObject) dataObject);
    } else if (dataObject.isCollection()) {
      ((CollectionObject) dataObject)
          .iterator()
          .forEachRemaining(
              collectionDataObject ->
                  result.addAll(
                      extractNESTWorkflowObjectsFromPool((DataObject) collectionDataObject)));
    } else if (dataObject.isAggregate()) {
      ((AggregateObject) dataObject)
          .getAttributeMap()
          .forEach(
              (attributeName, attributeDataObject) ->
                  result.addAll(extractNESTWorkflowObjectsFromPool(attributeDataObject)));
    }
    return result;
  }

  public static boolean containsChild(DataObject parent, DataObject child) {
    if (parent == child) {
      return true;
    } else if (parent.isNESTEdge() || parent.isNESTNode()) {
      return Utils.containsChild(((NESTGraphItemObject) parent).getSemanticDescriptor(), child);
    } else if (parent.isNESTWorkflow()) {
      NESTWorkflowObject nestWorkflow = (NESTWorkflowObject) parent;
      return Stream
          .concat(nestWorkflow.getGraphNodes().stream(), nestWorkflow.getGraphEdges().stream())
          .anyMatch(
              graphElement -> graphElement == child || Utils.containsChild(graphElement, child));
    } else if (parent.isCollection()) {
      boolean contains[] = {false};
      ((CollectionObject) parent)
          .iterator()
          .forEachRemaining(
              collectionDataObject ->
                  contains[0] =
                      Utils.containsChild((DataObject) collectionDataObject, child) || contains[0]);
      return contains[0];
    } else if (parent.isAggregate()) {
      return ((AggregateObject) parent)
          .getAttributeMap().values().stream()
          .anyMatch(attributeValue -> Utils.containsChild(attributeValue, child));
    }
    return false;
  }

  /**
   * Source: https://stackoverflow.com/a/26877737
   */
  public static void setDefaultFontSize(float size) {
    Set<Object> keySet = UIManager.getLookAndFeelDefaults().keySet();
    Object[] keys = keySet.toArray(new Object[keySet.size()]);

    for (Object key : keys) {
      if (key != null && key.toString().toLowerCase().contains("font")) {
        Font font = UIManager.getDefaults().getFont(key);
        if (font != null) {
          font = font.deriveFont(size);
          UIManager.put(key, font);
        }
      }
    }
  }
}
