/*
 * Copyright (c) 2001-2012, JGraph Ltd
 */
package de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.editor;

import com.google.common.collect.ClassToInstanceMap;
import com.google.common.collect.ImmutableClassToInstanceMap;
import com.mxgraph.analysis.mxDistanceCostFunction;
import com.mxgraph.analysis.mxGraphAnalysis;
import com.mxgraph.canvas.mxGraphics2DCanvas;
import com.mxgraph.canvas.mxICanvas;
import com.mxgraph.canvas.mxSvgCanvas;
import com.mxgraph.io.mxCodec;
import com.mxgraph.model.mxGeometry;
import com.mxgraph.model.mxGraphModel;
import com.mxgraph.model.mxICell;
import com.mxgraph.model.mxIGraphModel;
import com.mxgraph.shape.mxStencilShape;
import com.mxgraph.swing.handler.mxConnectionHandler;
import com.mxgraph.swing.mxGraphComponent;
import com.mxgraph.swing.mxGraphOutline;
import com.mxgraph.swing.util.mxGraphActions;
import com.mxgraph.swing.view.mxCellEditor;
import com.mxgraph.util.mxCellRenderer;
import com.mxgraph.util.mxCellRenderer.CanvasFactory;
import com.mxgraph.util.mxConstants;
import com.mxgraph.util.mxDomUtils;
import com.mxgraph.util.mxResources;
import com.mxgraph.util.mxUtils;
import com.mxgraph.util.mxXmlUtils;
import com.mxgraph.view.mxGraph;
import de.uni_trier.wi2.procake.data.model.nest.NESTWorkflowClass;
import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.object.base.AggregateObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTGraphItemObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTWorkflowObject;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.NESTWorkflowEditor;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.NESTWorkflowEditor.CustomGraph;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.NESTWorkflowEditor.CustomGraphComponent;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.NESTWorkflowLayout;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.StylesheetEditor;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.mxgraph.mxSvgCanvasWithStencilSupport;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.utils.Utils;
import de.uni_trier.wi2.procake.utils.conversion.xml.JAXBUtil;
import de.uni_trier.wi2.procake.utils.io.IOUtil;
import jakarta.xml.bind.JAXBException;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlElementWrapper;
import jakarta.xml.bind.annotation.XmlRootElement;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.awt.print.PageFormat;
import java.awt.print.Paper;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Method;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.imageio.ImageIO;
import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import javax.swing.text.html.HTML;
import javax.swing.text.html.HTMLDocument;
import javax.swing.text.html.HTMLEditorKit;
import org.apache.commons.io.FileUtils;
import org.w3c.dom.Document;

/**
 *
 */
public class EditorActions {

  private static final ClassToInstanceMap<Action> actionMap =
      new ImmutableClassToInstanceMap.Builder<Action>()
          .put(ToggleAutoImportExportConfigAction.class, new ToggleAutoImportExportConfigAction())
          .put(ToggleWorkflowNodeVisibilityAction.class, new ToggleWorkflowNodeVisibilityAction())
          .put(ToggleSequenceNodeVisibilityAction.class, new ToggleSequenceNodeVisibilityAction())
          .put(ToggleDataNodeVisibilityAction.class, new ToggleDataNodeVisibilityAction())
          .put(ToggleIdVisibilityAction.class, new ToggleIdVisibilityAction())
          .put(ToggleEdgeLabelVisibilityAction.class, new ToggleEdgeLabelVisibilityAction())
          .put(ToggleCellTooltipVisibilityAction.class, new ToggleCellTooltipVisibilityAction())
          .put(ToggleDataFlowEdgeStyleAction.class, new ToggleDataFlowEdgeStyleAction())
          .put(ToggleReLayoutAction.class, new ToggleReLayoutAction())
          //.put(RefreshNESTWorkflowAction.class, new RefreshNESTWorkflowAction())
          .build();

  /**
   * @param e
   * @return Returns the graph for the given action event.
   */
  public static final BasicGraphEditor getEditor(ActionEvent e) {
    if (e.getSource() instanceof Component) {
      Component component = (Component) e.getSource();

      while (component != null && !(component instanceof BasicGraphEditor)) {
        component =
            component instanceof JPopupMenu
                ? ((JPopupMenu) component).getInvoker()
                : (component instanceof EditorMenuBar
                    ? ((EditorMenuBar) component).getEditor()
                    : component.getParent());
      }

      return (BasicGraphEditor) component;
    }

    return null;
  }

  public static Action getActionFor(Class<? extends Action> actionClasss) {
    return actionMap.getInstance(actionClasss);
  }

  public static class AddSemanticDescriptorNodeAction extends AbstractAction {

    public static final String SEMANTIC_DESCRIPTOR_NODE_ID_PREFIX = "SDN/";
    public static final String SEMANTIC_DESCRIPTOR_NODE_CONNECTOR_ID_PREFIX = "SDNC/";
    private List<mxICell> cells;
    private NESTWorkflowEditor editor;

    public AddSemanticDescriptorNodeAction(List<mxICell> cells) {
      super("Add semantic descriptor node");
      this.cells = cells;
    }

    @Override
    public void actionPerformed(ActionEvent e) {
      SwingUtilities.invokeLater(
          () -> {
            this.editor = (NESTWorkflowEditor) getEditor(e);
            mxGraph graph = editor.getGraphComponent().getGraph();
            graph.getModel().beginUpdate();
            try {
              this.cells.forEach(cell -> this.addSemanticDescriptorNode(cell, graph));
            } finally {
              graph.getModel().endUpdate();
            }
          });
    }

    private void addSemanticDescriptorNode(mxICell cell, mxGraph graph) {
      DataObject semanticDescriptor = ((NESTGraphItemObject) cell.getValue()).getSemanticDescriptor();
      String semanticDescriptorVertexContent = "null";
      if (semanticDescriptor != null) {
        if (semanticDescriptor.isAggregate()) {
          semanticDescriptorVertexContent = Utils.aggregateToPaddedTable(
              (AggregateObject) semanticDescriptor, false);
        } else {
          semanticDescriptorVertexContent = semanticDescriptor.toString();
        }
      }

      double semanticDescriptorVertexX = cell.getGeometry().getX();
      double semanticDescriptorVertexY = cell.getGeometry().getY();
      if (cell.isEdge()) {
        double sourceCenterX = cell.getTerminal(true).getGeometry().getCenterX();
        double sourceCenterY = cell.getTerminal(true).getGeometry().getCenterY();
        double targetCenterX = cell.getTerminal(false).getGeometry().getCenterX();
        double targetCenterY = cell.getTerminal(false).getGeometry().getCenterY();
        semanticDescriptorVertexX = sourceCenterX - (sourceCenterX - targetCenterX) / 2.0;
        semanticDescriptorVertexY = sourceCenterY - (sourceCenterY - targetCenterY) / 2.0;
      }
      String semanticDescriptorCellsID = UUID.randomUUID().toString();
      mxICell semanticDescriptorVertex =
          (mxICell)
              graph.insertVertex(
                  graph.getDefaultParent(),
                  SEMANTIC_DESCRIPTOR_NODE_ID_PREFIX + semanticDescriptorCellsID,
                  // SDN = Semantic Descriptor Node
                  semanticDescriptorVertexContent,
                  semanticDescriptorVertexX,
                  semanticDescriptorVertexY,
                  50,
                  50,
                  "SEMANTIC_DESCRIPTOR_NODE_STYLE");
      editor
          .getGraphComponent()
          .labelChanged(semanticDescriptorVertex, semanticDescriptorVertex.getValue(), null);
      graph.insertEdge(
          graph.getDefaultParent(),
          SEMANTIC_DESCRIPTOR_NODE_CONNECTOR_ID_PREFIX + semanticDescriptorCellsID,
          // SDNC = Semantic Descriptor Node Connector
          null,
          cell,
          semanticDescriptorVertex,
          "SEMANTIC_DESCRIPTOR_NODE_CONNECTOR_STYLE");
    }
  }

  public static class ToggleAutoImportExportConfigAction extends AbstractAction {

    public ToggleAutoImportExportConfigAction() {
      super("Auto import/export layout & style");
      putValue(Action.SELECTED_KEY, true);
    }

    @Override
    public void actionPerformed(ActionEvent e) {

    }
  }

  private static class ToggleNodeVisibilityAction extends AbstractAction {

    private Predicate<? super NESTNodeObject> nodeFilter;

    public ToggleNodeVisibilityAction(String name, Predicate<? super NESTNodeObject> nodeFilter) {
      super(name);
      putValue(Action.SELECTED_KEY, true);
      this.nodeFilter = nodeFilter;
    }

    @Override
    public void actionPerformed(ActionEvent e) {
      SwingUtilities.invokeLater(
          () -> {
            var editor = getEditor(e);
            mxGraph graph = editor.getGraphComponent().getGraph();
            Collection<mxICell> cells =
                ((mxGraphModel) graph.getModel())
                    .getCells().values().stream()
                    .map(mxICell.class::cast)
                    .collect(
                        Collectors
                            .toList()); // only model.getCells includes previously hidden cells.
            // Downcast from interface to implementation necessary
            // to access the method
            List<mxICell> cellsToToggle =
                cells.stream()
                    .filter(
                        nodeCell ->
                            nodeCell.getValue() instanceof NESTNodeObject // only node cells
                                && nodeFilter.test(
                                ((NESTNodeObject)
                                    nodeCell.getValue()))) // whose type match to the specific
                    // ToggleNodeVisibilityAction
                    .collect(Collectors.toList());
            graph.getModel().beginUpdate();
            try {
              cellsToToggle.forEach(
                  cell ->
                      graph.getModel().setVisible(cell, (boolean) getValue(Action.SELECTED_KEY)));
            } finally {
              graph.getModel().endUpdate();
            }
          });
    }
  }

  public static class ToggleWorkflowNodeVisibilityAction extends ToggleNodeVisibilityAction {

    public ToggleWorkflowNodeVisibilityAction() {
      super("(Sub)Workflow", node -> node.isNESTSubWorkflowNode() || node.isNESTWorkflowNode());
    }
  }

  public static class ToggleSequenceNodeVisibilityAction extends ToggleNodeVisibilityAction {

    public ToggleSequenceNodeVisibilityAction() {
      super("Sequence", NESTNodeObject::isNESTSequenceNode);
    }
  }

  public static class ToggleDataNodeVisibilityAction extends ToggleNodeVisibilityAction {

    public ToggleDataNodeVisibilityAction() {
      super("Data", NESTNodeObject::isNESTDataNode);
    }
  }

  public static class ToggleIdVisibilityAction extends AbstractAction {

    public ToggleIdVisibilityAction() {
      super("Show IDs");
      putValue(Action.SELECTED_KEY, false);
    }

    @Override
    public void actionPerformed(ActionEvent e) {
      SwingUtilities.invokeLater(
          () -> {
            var editor = getEditor(e);
            NESTWorkflowEditor.CustomGraph graph =
                (NESTWorkflowEditor.CustomGraph) editor.getGraphComponent().getGraph();
            graph.getCellLabelGenerator().setShowIds((boolean) getValue(Action.SELECTED_KEY));
            ((NESTWorkflowEditor.CustomGraphComponent) editor.getGraphComponent())
                .updateAllCellLabels();
          });
    }
  }

  public static class ToggleEdgeLabelVisibilityAction extends AbstractAction {

    public ToggleEdgeLabelVisibilityAction() {
      super("Show edge labels");
      putValue(Action.SELECTED_KEY, true);
    }

    @Override
    public void actionPerformed(ActionEvent e) {
      SwingUtilities.invokeLater(
          () -> {
            var editor = getEditor(e);
            NESTWorkflowEditor.CustomGraph graph =
                (NESTWorkflowEditor.CustomGraph) editor.getGraphComponent().getGraph();
            graph
                .getCellLabelGenerator()
                .setShowEdgeLabels((boolean) getValue(Action.SELECTED_KEY));
            ((NESTWorkflowEditor.CustomGraphComponent) editor.getGraphComponent())
                .updateAllCellLabels();
          });
    }
  }

  public static class ToggleCellTooltipVisibilityAction extends AbstractAction {

    public ToggleCellTooltipVisibilityAction() {
      super("Show cell tooltip on hover");
      putValue(Action.SELECTED_KEY, false);
    }

    @Override
    public void actionPerformed(ActionEvent e) {
      SwingUtilities.invokeLater(
          () -> {
            var editor = getEditor(e);
            editor.getGraphComponent().setToolTips((boolean) getValue(Action.SELECTED_KEY));
          });
    }
  }

  public static class ToggleReLayoutAction extends AbstractAction {

    public ToggleReLayoutAction() {
      super("Automatic re-layout on edge insertion");
      putValue(Action.SELECTED_KEY, NESTWorkflowLayout.DEFAULT_EXECUTE_ON_EDGE_INSERTION);
      putValue(Action.SHORT_DESCRIPTION, mxResources.get("toggleReLayoutActionToolTip"));
    }

    @Override
    public void actionPerformed(ActionEvent actionEvent) {
      SwingUtilities.invokeLater(
          () -> {
            NESTWorkflowLayout layout =
                ((NESTWorkflowEditor.CustomGraph)
                    getEditor(actionEvent).getGraphComponent().getGraph())
                    .getLayout();
            layout.setExecuteOnEdgeInsertion((boolean) getValue(Action.SELECTED_KEY));
          });
    }
  }

  public static class EditStylesheetAction extends AbstractAction {

    public EditStylesheetAction() {
      super("Edit stylesheet");
      putValue(
          SHORT_DESCRIPTION,
          "<html><p width='300'>Opens an editor for the stylesheet XML</p></html>");
    }

    @Override
    public void actionPerformed(ActionEvent actionEvent) {
      SwingUtilities.invokeLater(
          () -> new StylesheetEditor(
              (CustomGraphComponent) getEditor(actionEvent).getGraphComponent())
      );
    }
  }

//  public static class RefreshNESTWorkflowAction extends AbstractAction {
//
//    public RefreshNESTWorkflowAction() {
//      super("Reload NESTWorkflow from reference");
//      putValue(
//          SHORT_DESCRIPTION,
//          "<html><p width='300'>Copies the NESTWorkflow from the reference given on instantiation of the editor and loads this copy into the editor. Be aware that any unsaved changes made to the workflow graph will be lost.</p></html>");
//    }
//
//    @Override
//    public void actionPerformed(ActionEvent actionEvent) {
//      SwingUtilities.invokeLater(
//          () -> {
//            NESTWorkflowEditor editor = (NESTWorkflowEditor) getEditor(actionEvent);
//            mxGraphComponent graphComponent = editor.getGraphComponent();
//            NESTWorkflowEditor.CustomGraph customGraph =
//                (NESTWorkflowEditor.CustomGraph) graphComponent.getGraph();
//            customGraph.clearWithoutEvents();
//            NESTWorkflowObject newCopyOfOriginalNESTWorkflow =
//                (NESTWorkflowObject) editor.getOriginalNestWorkflow().copy();
//            editor.setNestWorkflow(newCopyOfOriginalNESTWorkflow);
//            customGraph.loadNESTWorkflow(newCopyOfOriginalNESTWorkflow);
//            graphComponent.zoomTo(1.0, graphComponent.isCenterZoom());
//            editor.applyNodeVisibilitySettings();
//          });
//    }
//  }

  public static class ValidateNESTWorkflowAction extends AbstractAction {

    public ValidateNESTWorkflowAction() {
      super("Validate NESTWorkflow");
    }

    @Override
    public void actionPerformed(ActionEvent actionEvent) {
      NESTWorkflowEditor.NESTWorkflowValidatorGUI nestWorkflowValidatorGUI =
          ((NESTWorkflowEditor) getEditor(actionEvent)).getNestWorkflowValidatorGUI();
      nestWorkflowValidatorGUI.validateNESTWorkflow();
      nestWorkflowValidatorGUI.setVisible(true);
    }
  }

  public static class LayoutNESTWorkflowAction extends AbstractAction {

    NESTWorkflowEditor.CustomGraph customGraph;

    public LayoutNESTWorkflowAction() {
      super("Execute layout");
    }

    @Override
    public void actionPerformed(ActionEvent actionEvent) {
      mxGraphComponent graphComponent = getEditor(actionEvent).getGraphComponent();
      ((NESTWorkflowEditor.CustomGraph) graphComponent.getGraph()).executeLayout();
      graphComponent.refresh();
    }
  }

  public static class ToggleDataFlowEdgeStyleAction extends AbstractAction {

    private Object originalStyle;

    public ToggleDataFlowEdgeStyleAction() {
      super("Direct dataflow edges");
      putValue(Action.SELECTED_KEY, false);
    }

    @Override
    public void actionPerformed(ActionEvent e) {
      SwingUtilities.invokeLater(
          () -> {
            var graph = getEditor(e).getGraphComponent().getGraph();
            Map<String, Object> dataflowEdgeStyle =
                graph
                    .getStylesheet()
                    .getStyles()
                    .get(NESTWorkflowEditor.CellStyle.DATAFLOW_EDGE_STYLE.name());
            if ((boolean) getValue(Action.SELECTED_KEY)) { // turned direct edges on
              originalStyle =
                  dataflowEdgeStyle.get(mxConstants.STYLE_EDGE); // preserve original style
              dataflowEdgeStyle.put(mxConstants.STYLE_EDGE, mxConstants.NONE);
            } else { // turned direct edges off
              dataflowEdgeStyle.put(
                  mxConstants.STYLE_EDGE, originalStyle); // restore original style
            }
            graph.refresh();
          });
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ToggleRulersItem extends JCheckBoxMenuItem {

    /**
     *
     */
    public ToggleRulersItem(final BasicGraphEditor editor, String name) {
      super(name);
      setSelected(editor.getGraphComponent().getColumnHeader() != null);

      addActionListener(
          new ActionListener() {
            /** */
            public void actionPerformed(ActionEvent e) {
              mxGraphComponent graphComponent = editor.getGraphComponent();

              if (graphComponent.getColumnHeader() != null) {
                graphComponent.setColumnHeader(null);
                graphComponent.setRowHeader(null);
              } else {
                graphComponent.setColumnHeaderView(
                    new EditorRuler(graphComponent, EditorRuler.ORIENTATION_HORIZONTAL));
                graphComponent.setRowHeaderView(
                    new EditorRuler(graphComponent, EditorRuler.ORIENTATION_VERTICAL));
              }
            }
          });
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ToggleGridItem extends JCheckBoxMenuItem {

    /**
     *
     */
    public ToggleGridItem(final BasicGraphEditor editor, String name) {
      super(name);
      setSelected(false);
      mxGraphComponent graphComponent = editor.getGraphComponent();
      mxGraph graph = graphComponent.getGraph();
      graph.setGridEnabled(false);
      graphComponent.setGridVisible(false);

      addActionListener(
          new ActionListener() {
            /** */
            public void actionPerformed(ActionEvent e) {
              mxGraphComponent graphComponent = editor.getGraphComponent();
              mxGraph graph = graphComponent.getGraph();
              boolean enabled = !graph.isGridEnabled();

              graph.setGridEnabled(enabled);
              graphComponent.setGridVisible(enabled);
              graphComponent.repaint();
              setSelected(enabled);
            }
          });
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ToggleOutlineItem extends JCheckBoxMenuItem {

    /**
     *
     */
    public ToggleOutlineItem(final BasicGraphEditor editor, String name) {
      super(name);
      setSelected(true);

      addActionListener(
          new ActionListener() {
            /** */
            public void actionPerformed(ActionEvent e) {
              final mxGraphOutline outline = editor.getGraphOutline();
              outline.setVisible(!outline.isVisible());
              outline.revalidate();

              SwingUtilities.invokeLater(
                  new Runnable() {
                    /*
                     * (non-Javadoc)
                     * @see java.lang.Runnable#run()
                     */
                    public void run() {
                      if (outline.getParent() instanceof JSplitPane) {
                        if (outline.isVisible()) {
                          ((JSplitPane) outline.getParent())
                              .setDividerLocation(editor.getHeight() - 300);
                          ((JSplitPane) outline.getParent()).setDividerSize(6);
                        } else {
                          ((JSplitPane) outline.getParent()).setDividerSize(0);
                        }
                      }
                    }
                  });
            }
          });
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ExitAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      BasicGraphEditor editor = getEditor(e);

      if (editor != null) {
        editor.exit();
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class StylesheetAction extends AbstractAction {

    /**
     *
     */
    protected String stylesheet;

    /**
     *
     */
    public StylesheetAction(String stylesheet) {
      this.stylesheet = stylesheet;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        mxGraph graph = graphComponent.getGraph();
        mxCodec codec = new mxCodec();
        Document doc = mxUtils.loadDocument(EditorActions.class.getResource(stylesheet).toString());

        if (doc != null) {
          codec.decode(doc.getDocumentElement(), graph.getStylesheet());
          graph.refresh();
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ZoomPolicyAction extends AbstractAction {

    /**
     *
     */
    protected int zoomPolicy;

    /**
     *
     */
    public ZoomPolicyAction(int zoomPolicy) {
      this.zoomPolicy = zoomPolicy;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        graphComponent.setPageVisible(true);
        graphComponent.setZoomPolicy(zoomPolicy);
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class GridStyleAction extends AbstractAction {

    /**
     *
     */
    protected int style;

    /**
     *
     */
    public GridStyleAction(int style) {
      this.style = style;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        graphComponent.setGridStyle(style);
        graphComponent.repaint();
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class GridColorAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        Color newColor =
            JColorChooser.showDialog(
                graphComponent, mxResources.get("gridColor"), graphComponent.getGridColor());

        if (newColor != null) {
          graphComponent.setGridColor(newColor);
          graphComponent.repaint();
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ScaleAction extends AbstractAction {

    /**
     *
     */
    protected double scale;

    /**
     *
     */
    public ScaleAction(double scale) {
      this.scale = scale;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        double scale = this.scale;

        if (scale == 0) {
          String value =
              (String)
                  JOptionPane.showInputDialog(
                      graphComponent,
                      mxResources.get("value"),
                      mxResources.get("scale") + " (%)",
                      JOptionPane.PLAIN_MESSAGE,
                      null,
                      null,
                      "");

          if (value != null) {
            scale = Double.parseDouble(value.replace("%", "")) / 100;
          }
        }

        if (scale > 0) {
          graphComponent.zoomTo(scale, graphComponent.isCenterZoom());
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class PageSetupAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        PrinterJob pj = PrinterJob.getPrinterJob();
        PageFormat format = pj.pageDialog(graphComponent.getPageFormat());

        if (format != null) {
          graphComponent.setPageFormat(format);
          graphComponent.zoomAndCenter();
        }
      }
    }
  }

  @SuppressWarnings("serial")
  public static class PrintAction extends AbstractAction {

    public PrintAction() {
      this.putValue(SHORT_DESCRIPTION, mxResources.get("print"));
    }

    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        PrinterJob pj = PrinterJob.getPrinterJob();

        if (pj.printDialog()) {
          PageFormat pf = graphComponent.getPageFormat();
          Paper paper = new Paper();
          double margin = 36;
          paper.setImageableArea(
              margin, margin, paper.getWidth() - margin * 2, paper.getHeight() - margin * 2);
          pf.setPaper(paper);
          pj.setPrintable(graphComponent, pf);

          try {
            pj.print();
          } catch (PrinterException e2) {
            System.out.println(e2);
          }
        }
      }
    }
  }

  public static class SaveInObjectAction extends AbstractAction {

    public SaveInObjectAction() {
      this.putValue(SHORT_DESCRIPTION, mxResources.get("saveChangesToObject"));
    }

    @Override
    public void actionPerformed(ActionEvent e) {
      NESTWorkflowEditor editor = (NESTWorkflowEditor) getEditor(e);
      assert editor != null;
      editor.saveInObjectNESTWorkflow((NESTWorkflowObject) editor.getNESTWorkflow());
      editor.setModified(false);
      editor.fireGraphSaved();
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ExportAsAction extends AbstractAction {

    protected boolean showDialog;
    protected String lastDir = null;
    private List<DefaultFileFilter> fileFilters = new LinkedList<>();

    public ExportAsAction(boolean showDialog) {
      this.showDialog = showDialog;
      fileFilters.add(new DefaultFileFilter(".xml", "XML " + mxResources.get("file") + " (.xml)"));
      fileFilters.add(new DefaultFileFilter(".svg", "SVG " + mxResources.get("file") + " (.svg)"));
      this.putValue(SHORT_DESCRIPTION, mxResources.get("exportAs"));
    }

    public void actionPerformed(ActionEvent e) {
      NESTWorkflowEditor editor = (NESTWorkflowEditor) getEditor(e);

      if (editor != null) {
        mxGraphComponent graphComponent = editor.getGraphComponent();
        mxGraph graph = graphComponent.getGraph();

        if (!this.validateNESTWorkflow(editor)) {
          return; // NESTGraph is invalid and user chooses to not save
        }

        FileFilter selectedFilter = null;
        String filename = null;
        boolean dialogShown = false;

        if (showDialog || editor.getCurrentFile() == null) {
          String wd;

          if (lastDir != null) {
            wd = lastDir;
          } else if (editor.getCurrentFile() != null) {
            wd = editor.getCurrentFile().getParent();
          } else {
            wd = System.getProperty("user.dir");
          }

          JFileChooser fc = new JFileChooser(wd);
          fileFilters.forEach(fc::addChoosableFileFilter);
          fc.setFileFilter(fileFilters.get(0));

          int rc = fc.showDialog(null, mxResources.get("save"));
          dialogShown = true;

          if (rc != JFileChooser.APPROVE_OPTION) {
            return;
          } else {
            lastDir = fc.getSelectedFile().getParent();
          }

          filename = fc.getSelectedFile().getAbsolutePath();
          selectedFilter = fc.getFileFilter();

          if (selectedFilter instanceof DefaultFileFilter) {
            String ext = ((DefaultFileFilter) selectedFilter).getExtension();

            if (!filename.toLowerCase().endsWith(ext)) {
              filename += ext;
            }
          }

          if (new File(filename).exists()
              && JOptionPane.showConfirmDialog(
              graphComponent, mxResources.get("overwriteExistingFile"))
              != JOptionPane.YES_OPTION) {
            return;
          }
        } else {
          filename = editor.getCurrentFile().getAbsolutePath();
        }

        try {
          String ext = filename.substring(filename.lastIndexOf('.') + 1);

          if (ext.equalsIgnoreCase("svg")) {
            mxSvgCanvas canvas =
                (mxSvgCanvas)
                    mxCellRenderer.drawCells(
                        graph,
                        null,
                        1,
                        null,
                        new CanvasFactory() {
                          public mxICanvas createCanvas(int width, int height) {
                            mxSvgCanvas canvas =
                                new mxSvgCanvasWithStencilSupport(
                                    mxDomUtils.createSvgDocument(width, height));
                            canvas.setEmbedded(true);

                            return canvas;
                          }
                        });

            mxUtils.writeFile(mxXmlUtils.getXml(canvas.getDocument()), filename);
          } else if (ext.equalsIgnoreCase("xml")) {
            boolean writeSuccess =
                IOUtil.writeFile(
                    ((NESTWorkflowEditor.CustomGraph) editor.getGraphComponent().getGraph())
                        .getNestWorkflow(),
                    filename)
                    != null;
            if (!writeSuccess) {
              throw new IOException("XML serialization failed. Check stacktrace for details.");
            }
            editor.setModified(false);
            editor.setCurrentFile(new File(filename));

            // export geometry and style information as additional xml file to same path when option is set
            boolean isExportConfig = (boolean) EditorActions.getActionFor(
                    ToggleAutoImportExportConfigAction.class)
                .getValue(Action.SELECTED_KEY);
            if (isExportConfig) {
              NESTWorkflowEditorConfig config = new NESTWorkflowEditorConfig(
                  (NESTWorkflowEditor) getEditor(e));
              int extensionDelimiterPosition = filename.lastIndexOf('.');
              String filenameLayout = filename.substring(0, extensionDelimiterPosition) + ".config"
                  + filename.substring(extensionDelimiterPosition);
              FileUtils.writeStringToFile(
                  new File(filenameLayout),
                  JAXBUtil.marshall(config), // String method creates pretty XML
                  Charset.defaultCharset());
            }
          }
        } catch (Throwable ex) {
          ex.printStackTrace();
          JOptionPane.showMessageDialog(
              graphComponent, ex.toString(), mxResources.get("error"), JOptionPane.ERROR_MESSAGE);
        }
      }
    }

    /**
     * @param
     * @return true when graph is a valid workflow graph or the user wants to save an invalid graph,
     * false when the graph is invalid and the user chooses to not save
     */
    private boolean validateNESTWorkflow(NESTWorkflowEditor editor) {
      NESTWorkflowEditor.NESTWorkflowValidatorGUI validatorGUI = editor
          .getNestWorkflowValidatorGUI();
      if (!validatorGUI.validateNESTWorkflow()) {
        Object[] options = {"Save anyway", "Show validator"};
        int userChoice =
            JOptionPane.showOptionDialog(
                null,
                "Graph is not a valid workflow!",
                "Invalid Workflow Graph",
                JOptionPane.YES_NO_OPTION,
                JOptionPane.ERROR_MESSAGE,
                null,
                options,
                options[1]);
        if (userChoice == 0) { // "Save anyway"
          validatorGUI.dispatchEvent(new WindowEvent(validatorGUI, WindowEvent.WINDOW_CLOSING));
          return true;
        } else if (userChoice == 1) { // "Show validator"
          validatorGUI.setVisible(true);
          return false;
        } else if (userChoice == -1) { // window close button
          return false;
        }
      }
      return true;
    }

    public void setFileFilters(List<DefaultFileFilter> fileFilters) {
      this.fileFilters = fileFilters;
    }
  }

  @XmlRootElement(name = "NESTWorkflowEditorConfig")
  public static class NESTWorkflowEditorConfig {

    private NESTWorkflowEditor nestWorkflowEditor;

    @XmlElement(name = "GraphAppearance")
    GraphAppearance graphAppearance;

    @XmlElement(name = "LayoutConfig")
    NESTWorkflowLayout.LayoutConfig layoutConfig;

    public NESTWorkflowEditorConfig(NESTWorkflowEditor nestWorkflowEditor) {
      this.nestWorkflowEditor = nestWorkflowEditor;
      this.layoutConfig = ((CustomGraph) nestWorkflowEditor.getGraphComponent()
          .getGraph()).getLayout()
          .getLayoutConfig();
      this.graphAppearance = new GraphAppearance(
          (CustomGraphComponent) nestWorkflowEditor.getGraphComponent());
    }

    public NESTWorkflowEditorConfig() {
    } // needed for JAXB

    public GraphAppearance getGraphAppearance() {
      return graphAppearance;
    }

    public void applyTo(NESTWorkflowEditor nestWorkflowEditor) throws JAXBException {
      mxGraph graph = nestWorkflowEditor.getGraphComponent().getGraph();
      this.getGraphAppearance().applyTo(graph);
      String stylesheetXml = JAXBUtil.marshall(this.getGraphAppearance().getStylesheet());
      ((CustomGraphComponent) nestWorkflowEditor.getGraphComponent()).setStylesheet(stylesheetXml);

      NESTWorkflowLayout layout = ((CustomGraph) nestWorkflowEditor.getGraphComponent()
          .getGraph()).getLayout();
      layout.setLayoutConfig(this.layoutConfig);
    }
  }

  @XmlRootElement(name = "GraphAppearance")
  public static class GraphAppearance implements Serializable {

    static class Cell {

      @XmlElement(name = "id")
      String id;
      @XmlElement(name = "sourceId")
      String sourceId; // needed when cell is a semantic descriptor node connector
      @XmlElement(name = "targetId")
      String targetId; // needed when cell is a semantic descriptor node connector
      @XmlElement(name = "value")
      String value; // needed when cell is a semantic descriptor node
      @XmlElement(name = "geometry")
      mxGeometry geometry;
      @XmlElement(name = "style")
      String style;
    }

    @XmlRootElement(name = "mxStylesheet")
    public static class mxStylesheet {

      static class add {

        @XmlElement(name = "add")
        List<add> adds;

        @XmlAttribute(name = "as")
        String as;

        @XmlAttribute(name = "value")
        String value;

        @XmlAttribute(name = "extend")
        String extend;
      }

      @XmlElement(name = "add")
      List<add> adds;
    }

    @XmlElementWrapper(name = "Cells")
    @XmlElement(name = "Cell")
    List<Cell> cells = new LinkedList<>();

    @XmlElement(name = "mxStylesheet")
    mxStylesheet stylesheet;

    public GraphAppearance() {
    } // needed for JAXB

    public GraphAppearance(CustomGraphComponent graphComponent) {
      mxGraph graph = graphComponent.getGraph();
      Map<String, Object> cells = ((mxGraphModel) graph.getModel()).getCells();
      cells.values().stream()
          .map(mxICell.class::cast)
          .forEach(mxCell -> {
                Cell cell = new Cell();
                cell.id = mxCell.getId();
                cell.geometry = mxCell.getGeometry();
                cell.style = mxCell.getStyle();
                handleSemanticDescriptorNodeConnectorCell(cell, mxCell);
                handleSemanticDescriptorNodeCell(cell, mxCell);
                this.cells.add(cell);
              }
          );
      try {
        this.stylesheet = JAXBUtil.unmarshall(graphComponent.getStylesheet(), mxStylesheet.class);
      } catch (JAXBException e) {
        e.printStackTrace();
      }
    }

    /**
     * Checks whether the given mxCell is a semantic descriptor node connector cell and fills the
     * fields sourceId and targetId to the cell object, which is needed to connect the corresponding
     * cells when importing the GraphAppearance
     *
     * @param cell
     * @param mxCell
     */
    private void handleSemanticDescriptorNodeConnectorCell(Cell cell, mxICell mxCell) {
      boolean isSDNConnector = mxCell.getId().startsWith(
          AddSemanticDescriptorNodeAction.SEMANTIC_DESCRIPTOR_NODE_CONNECTOR_ID_PREFIX);
      if (isSDNConnector) {
        cell.sourceId = mxCell.getTerminal(true).getId();
        cell.targetId = mxCell.getTerminal(false).getId();
      }
    }

    private void handleSemanticDescriptorNodeCell(Cell cell, mxICell mxCell) {
      boolean isSDN = mxCell.getId().startsWith(
          AddSemanticDescriptorNodeAction.SEMANTIC_DESCRIPTOR_NODE_ID_PREFIX);
      if (isSDN) {
        cell.value = mxCell.getValue().toString();
      }
    }

    public void applyTo(mxGraph graph) {
      Map<String, Object> mxCells = ((mxGraphModel) graph.getModel()).getCells();
      try {
        graph.getModel().beginUpdate();
        this.cells.stream()
            .filter(cell -> mxCells.containsKey(cell.id))
            .forEach(cell -> this.applyCellAppearance(graph, cell, (mxICell) mxCells.get(cell.id)));
        this.cells.stream()
            .filter(cell -> !mxCells.containsKey(cell.id) && cell.id.startsWith(
                AddSemanticDescriptorNodeAction.SEMANTIC_DESCRIPTOR_NODE_ID_PREFIX))
            .forEach(cell -> this.insertSemanticDescriptorVertex(graph, cell));
        this.cells.stream()
            .filter(cell -> !mxCells.containsKey(cell.id) && cell.id.startsWith(
                AddSemanticDescriptorNodeAction.SEMANTIC_DESCRIPTOR_NODE_CONNECTOR_ID_PREFIX))
            .forEach(cell -> this.insertSemanticDescriptorNodeConnector(graph, cell));
      } finally {
        graph.getModel().endUpdate();
      }
    }

    private void applyCellAppearance(mxGraph graph, Cell cellAppearance, mxICell mxCell) {
      if (cellAppearance.geometry != null) {
        graph.getModel().setGeometry(mxCell, cellAppearance.geometry);
      }
      if (cellAppearance.style != null) {
        graph.getModel().setStyle(mxCell, cellAppearance.style);
      }
    }

    private void insertSemanticDescriptorVertex(mxGraph graph, Cell cellAppearance) {
      var test = graph.insertVertex(
          graph.getDefaultParent(),
          cellAppearance.id,
          cellAppearance.value,
          cellAppearance.geometry.getX(),
          cellAppearance.geometry.getY(),
          cellAppearance.geometry.getWidth(),
          cellAppearance.geometry.getHeight(),
          cellAppearance.style);
    }

    private void insertSemanticDescriptorNodeConnector(mxGraph graph, Cell cellAppearance) {
      Map<String, Object> mxCells = ((mxGraphModel) graph.getModel()).getCells();
      mxICell sourceCell = (mxICell) mxCells.get(cellAppearance.sourceId);
      mxICell targetCell = (mxICell) mxCells.get(cellAppearance.targetId);
      graph.insertEdge(
          graph.getDefaultParent(),
          cellAppearance.id,
          null,
          sourceCell,
          targetCell,
          cellAppearance.style);
    }

    public mxStylesheet getStylesheet() {
      return stylesheet;
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class SelectShortestPathAction extends AbstractAction {

    /**
     *
     */
    protected boolean directed;

    /**
     *
     */
    public SelectShortestPathAction(boolean directed) {
      this.directed = directed;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        mxGraph graph = graphComponent.getGraph();
        mxIGraphModel model = graph.getModel();

        Object source = null;
        Object target = null;

        Object[] cells = graph.getSelectionCells();

        for (int i = 0; i < cells.length; i++) {
          if (model.isVertex(cells[i])) {
            if (source == null) {
              source = cells[i];
            } else if (target == null) {
              target = cells[i];
            }
          }

          if (source != null && target != null) {
            break;
          }
        }

        if (source != null && target != null) {
          int steps = graph.getChildEdges(graph.getDefaultParent()).length;
          Object[] path =
              mxGraphAnalysis
                  .getInstance()
                  .getShortestPath(
                      graph, source, target, new mxDistanceCostFunction(), steps, directed);
          graph.setSelectionCells(path);
        } else {
          JOptionPane.showMessageDialog(
              graphComponent, mxResources.get("noSourceAndTargetSelected"));
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class SelectSpanningTreeAction extends AbstractAction {

    /**
     *
     */
    protected boolean directed;

    /**
     *
     */
    public SelectSpanningTreeAction(boolean directed) {
      this.directed = directed;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        mxGraph graph = graphComponent.getGraph();
        mxIGraphModel model = graph.getModel();

        Object parent = graph.getDefaultParent();
        Object[] cells = graph.getSelectionCells();

        for (int i = 0; i < cells.length; i++) {
          if (model.getChildCount(cells[i]) > 0) {
            parent = cells[i];
            break;
          }
        }

        Object[] v = graph.getChildVertices(parent);
        Object[] mst =
            mxGraphAnalysis
                .getInstance()
                .getMinimumSpanningTree(graph, v, new mxDistanceCostFunction(), directed);
        graph.setSelectionCells(mst);
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ToggleDirtyAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        graphComponent.showDirtyRectangle = !graphComponent.showDirtyRectangle;
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ToggleConnectModeAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        mxConnectionHandler handler = graphComponent.getConnectionHandler();
        handler.setHandleEnabled(!handler.isHandleEnabled());
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ToggleCreateTargetItem extends JCheckBoxMenuItem {

    /**
     *
     */
    public ToggleCreateTargetItem(final BasicGraphEditor editor, String name) {
      super(name);
      setSelected(true);

      addActionListener(
          new ActionListener() {
            /** */
            public void actionPerformed(ActionEvent e) {
              mxGraphComponent graphComponent = editor.getGraphComponent();

              if (graphComponent != null) {
                mxConnectionHandler handler = graphComponent.getConnectionHandler();
                handler.setCreateTarget(!handler.isCreateTarget());
                setSelected(handler.isCreateTarget());
              }
            }
          });
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class PromptPropertyAction extends AbstractAction {

    /**
     *
     */
    protected Object target;

    /**
     *
     */
    protected String fieldname, message;

    /**
     *
     */
    public PromptPropertyAction(Object target, String message) {
      this(target, message, message);
    }

    /**
     *
     */
    public PromptPropertyAction(Object target, String message, String fieldname) {
      this.target = target;
      this.message = message;
      this.fieldname = fieldname;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof Component) {
        try {
          Method getter = target.getClass().getMethod("get" + fieldname);
          Object current = getter.invoke(target);

          // TODO: Support other atomic types
          if (current instanceof Integer) {
            Method setter = target.getClass().getMethod("set" + fieldname, new Class[]{int.class});

            String value =
                (String)
                    JOptionPane.showInputDialog(
                        (Component) e.getSource(),
                        "Value",
                        message,
                        JOptionPane.PLAIN_MESSAGE,
                        null,
                        null,
                        current);

            if (value != null) {
              setter.invoke(target, Integer.parseInt(value));
            }
          }
        } catch (Exception ex) {
          ex.printStackTrace();
        }
      }

      // Repaints the graph component
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        graphComponent.repaint();
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class TogglePropertyItem extends JCheckBoxMenuItem {

    /**
     *
     */
    public TogglePropertyItem(Object target, String name, String fieldname) {
      this(target, name, fieldname, false);
    }

    /**
     *
     */
    public TogglePropertyItem(Object target, String name, String fieldname, boolean refresh) {
      this(target, name, fieldname, refresh, null);
    }

    /**
     *
     */
    public TogglePropertyItem(
        final Object target,
        String name,
        final String fieldname,
        final boolean refresh,
        ActionListener listener) {
      super(name);

      // Since action listeners are processed last to first we add the given
      // listener here which means it will be processed after the one below
      if (listener != null) {
        addActionListener(listener);
      }

      addActionListener(
          new ActionListener() {
            /** */
            public void actionPerformed(ActionEvent e) {
              execute(target, fieldname, refresh);
            }
          });

      PropertyChangeListener propertyChangeListener =
          new PropertyChangeListener() {

            /*
             * (non-Javadoc)
             * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
             */
            public void propertyChange(PropertyChangeEvent evt) {
              if (evt.getPropertyName().equalsIgnoreCase(fieldname)) {
                update(target, fieldname);
              }
            }
          };

      if (target instanceof mxGraphComponent) {
        ((mxGraphComponent) target).addPropertyChangeListener(propertyChangeListener);
      } else if (target instanceof mxGraph) {
        ((mxGraph) target).addPropertyChangeListener(propertyChangeListener);
      }

      update(target, fieldname);
    }

    /**
     *
     */
    public void update(Object target, String fieldname) {
      if (target != null && fieldname != null) {
        try {
          Method getter = target.getClass().getMethod("is" + fieldname);

          if (getter != null) {
            Object current = getter.invoke(target);

            if (current instanceof Boolean) {
              setSelected(((Boolean) current).booleanValue());
            }
          }
        } catch (Exception e) {
          // ignore
        }
      }
    }

    /**
     *
     */
    public void execute(Object target, String fieldname, boolean refresh) {
      if (target != null && fieldname != null) {
        try {
          Method getter = target.getClass().getMethod("is" + fieldname);
          Method setter =
              target.getClass().getMethod("set" + fieldname, new Class[]{boolean.class});

          Object current = getter.invoke(target);

          if (current instanceof Boolean) {
            boolean value = !((Boolean) current).booleanValue();
            setter.invoke(target, value);
            setSelected(value);
          }

          if (refresh) {
            mxGraph graph = null;

            if (target instanceof mxGraph) {
              graph = (mxGraph) target;
            } else if (target instanceof mxGraphComponent) {
              graph = ((mxGraphComponent) target).getGraph();
            }

            graph.refresh();
          }
        } catch (Exception e) {
          // ignore
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class HistoryAction extends AbstractAction {

    /**
     *
     */
    protected boolean undo;

    /**
     *
     */
    public HistoryAction(boolean undo) {
      this.undo = undo;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      BasicGraphEditor editor = getEditor(e);

      if (editor != null) {
        if (undo) {
          editor.getUndoManager().undo();
        } else {
          editor.getUndoManager().redo();
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class FontStyleAction extends AbstractAction {

    /**
     *
     */
    protected boolean bold;

    /**
     *
     */
    public FontStyleAction(boolean bold) {
      this.bold = bold;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        Component editorComponent = null;

        if (graphComponent.getCellEditor() instanceof mxCellEditor) {
          editorComponent = ((mxCellEditor) graphComponent.getCellEditor()).getEditor();
        }

        if (editorComponent instanceof JEditorPane) {
          JEditorPane editorPane = (JEditorPane) editorComponent;
          int start = editorPane.getSelectionStart();
          int ende = editorPane.getSelectionEnd();
          String text = editorPane.getSelectedText();

          if (text == null) {
            text = "";
          }

          try {
            HTMLEditorKit editorKit = new HTMLEditorKit();
            HTMLDocument document = (HTMLDocument) editorPane.getDocument();
            document.remove(start, (ende - start));
            editorKit.insertHTML(
                document,
                start,
                ((bold) ? "<b>" : "<i>") + text + ((bold) ? "</b>" : "</i>"),
                0,
                0,
                (bold) ? HTML.Tag.B : HTML.Tag.I);
          } catch (Exception ex) {
            ex.printStackTrace();
          }

          editorPane.requestFocus();
          editorPane.select(start, ende);
        } else {
          mxIGraphModel model = graphComponent.getGraph().getModel();
          model.beginUpdate();
          try {
            graphComponent.stopEditing(false);
            graphComponent
                .getGraph()
                .toggleCellStyleFlags(
                    mxConstants.STYLE_FONTSTYLE,
                    (bold) ? mxConstants.FONT_BOLD : mxConstants.FONT_ITALIC);
          } finally {
            model.endUpdate();
          }
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class WarningAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        Object[] cells = graphComponent.getGraph().getSelectionCells();

        if (cells != null && cells.length > 0) {
          String warning = JOptionPane.showInputDialog(mxResources.get("enterWarningMessage"));

          for (int i = 0; i < cells.length; i++) {
            graphComponent.setCellWarning(cells[i], warning);
          }
        } else {
          JOptionPane.showMessageDialog(graphComponent, mxResources.get("noCellSelected"));
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class NewNESTWorkflowAction extends AbstractAction {

    private NESTWorkflowClass nestWorkflowClass;

    public NewNESTWorkflowAction(NESTWorkflowClass nestWorkflowClass) {
      this.nestWorkflowClass = nestWorkflowClass;
    }

    public void actionPerformed(ActionEvent e) {
      NESTWorkflowObject newNestWorkflow = (NESTWorkflowObject) nestWorkflowClass.newObject();
      newNestWorkflow.setId("New_NESTWorkflow");
      new NESTWorkflowEditor(newNestWorkflow);
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ImportAction extends AbstractAction {

    /**
     *
     */
    protected String lastDir;

    /**
     * Loads and registers the shape as a new shape in mxGraphics2DCanvas and adds a new entry to
     * use that shape in the specified palette
     *
     * @param palette   The palette to add the shape to.
     * @param nodeXml   The raw XML of the shape
     * @param path      The path to the directory the shape exists in
     * @param cellValue The value the cell should have when newly created from the palette
     * @return the string name of the shape
     */
    public static String addStencilShape(
        EditorPalette palette, String nodeXml, String path, String style, Object cellValue) {

      // Some editors place a 3 byte BOM at the start of files
      // Ensure the first char is a "<"
      int lessthanIndex = nodeXml.indexOf("<");
      nodeXml = nodeXml.substring(lessthanIndex);
      mxStencilShape newShape = new mxStencilShape(nodeXml);
      String name = newShape.getName();
      ImageIcon icon = null;

      if (path != null) {
        try {
          icon = new ImageIcon(ImageIO.read(NESTWorkflowEditor.class.getResourceAsStream(path)));
        } catch (IOException e) {
          e.printStackTrace();
        }
      }

      // Registers the shape in the canvas shape registry
      mxGraphics2DCanvas.putShape(name, newShape);

      if (palette != null && icon != null) {
        palette.addTemplate(name, icon, style, 80, 80, cellValue);
      }

      return name;
    }

    public static String addStencilShape(
        EditorPalette palette, String nodeXml, String path, String style) {
      return addStencilShape(palette, nodeXml, path, style, "");
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      BasicGraphEditor editor = getEditor(e);

      if (editor != null) {
        String wd = (lastDir != null) ? lastDir : System.getProperty("user.dir");

        JFileChooser fc = new JFileChooser(wd);

        fc.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);

        // Adds file filter for Dia shape import
        fc.addChoosableFileFilter(
            new DefaultFileFilter(".shape", "Dia Shape " + mxResources.get("file") + " (.shape)"));

        int rc = fc.showDialog(null, mxResources.get("importStencil"));

        if (rc == JFileChooser.APPROVE_OPTION) {
          lastDir = fc.getSelectedFile().getParent();

          try {
            if (fc.getSelectedFile().isDirectory()) {
              EditorPalette palette = editor.insertPalette(fc.getSelectedFile().getName());

              for (File f :
                  fc.getSelectedFile()
                      .listFiles(
                          new FilenameFilter() {
                            public boolean accept(File dir, String name) {
                              return name.toLowerCase().endsWith(".shape");
                            }
                          })) {
                String nodeXml = mxUtils.readFile(f.getAbsolutePath());
                addStencilShape(palette, nodeXml, f.getParent() + File.separator, "");
              }

              JComponent scrollPane = (JComponent) palette.getParent().getParent();
              editor.getLibraryPane().setSelectedComponent(scrollPane);

              // FIXME: Need to update the size of the palette to force a layout
              // update. Re/in/validate of palette or parent does not work.
              // editor.getLibraryPane().revalidate();
            } else {
              String nodeXml = mxUtils.readFile(fc.getSelectedFile().getAbsolutePath());
              String name = addStencilShape(null, nodeXml, null, "");

              JOptionPane.showMessageDialog(
                  editor, mxResources.get("stencilImported", new String[]{name}));
            }
          } catch (IOException e1) {
            e1.printStackTrace();
          }
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class OpenAction extends AbstractAction {

    protected String lastDir;

    public OpenAction() {
      this.putValue(SHORT_DESCRIPTION, mxResources.get("openFile"));
    }

    protected void resetEditor(BasicGraphEditor editor) {
      editor.setModified(false);
      editor.getUndoManager().clear();
      editor.getGraphComponent().zoomAndCenter();
    }

    public void actionPerformed(ActionEvent e) {
      NESTWorkflowEditor editor = (NESTWorkflowEditor) getEditor(e);

      if (editor != null) {
        if (!editor.isModified()
            || JOptionPane.showConfirmDialog(
            editor,
            mxResources.get("loseChanges"),
            UIManager.getString("OptionPane.titleText"),
            JOptionPane.YES_NO_OPTION)
            == JOptionPane.YES_OPTION) {
          mxGraph graph = editor.getGraphComponent().getGraph();

          if (graph != null) {
            String wd = (lastDir != null) ? lastDir : System.getProperty("user.dir");

            JFileChooser fc = new JFileChooser(wd);

            // Adds file filter for supported file format
            DefaultFileFilter defaultFilter =
                new DefaultFileFilter(".xml", "Pool and NESTWorkflow XML definitions (.xml)");
            fc.addChoosableFileFilter(defaultFilter);
            fc.setFileFilter(defaultFilter);

            int rc = fc.showDialog(null, mxResources.get("openFile"));

            if (rc == JFileChooser.APPROVE_OPTION) {
              lastDir = fc.getSelectedFile().getParent();
              var nestWorkflowFromFile =
                  IOUtil.readFile(fc.getSelectedFile().getAbsolutePath(), NESTWorkflowObject.class);
              if (nestWorkflowFromFile == null) {
                JOptionPane.showMessageDialog(
                    editor.getGraphComponent(),
                    mxResources.get("invalidXMLContentError"),
                    mxResources.get("error"),
                    JOptionPane.ERROR_MESSAGE);
              } else {
                editor.setCurrentFile(fc.getSelectedFile());
                resetEditor(editor);
                editor.open(nestWorkflowFromFile);

                boolean isImportLayoutAndStyles = (boolean) EditorActions.getActionFor(
                        ToggleAutoImportExportConfigAction.class)
                    .getValue(Action.SELECTED_KEY);
                if (isImportLayoutAndStyles) {
                  this.importConfigFromFile(editor, fc.getSelectedFile());
                }

              }
            }
          }
        }
      }
    }

    /**
     * Import geometry and style information from accompanying .layout.xml file
     *
     * @param workflowXMLFile
     */
    private void importConfigFromFile(BasicGraphEditor editor, File workflowXMLFile) {
      String workflowXMLFilePath = workflowXMLFile.getPath();
      int extensionDelimiterPosition = workflowXMLFilePath.lastIndexOf('.');
      String layoutXMLFilePath =
          workflowXMLFilePath.substring(0, extensionDelimiterPosition) + ".config"
              + workflowXMLFilePath.substring(extensionDelimiterPosition);
      File configFile = new File(layoutXMLFilePath);
      if (configFile.exists()) {
        try {
          NESTWorkflowEditorConfig config = JAXBUtil.unmarshall(new FileInputStream(configFile),
              NESTWorkflowEditorConfig.class);
          config.applyTo((NESTWorkflowEditor) editor);
        } catch (JAXBException e) {
          JOptionPane.showMessageDialog(
              editor.getGraphComponent(),
              mxResources.get("invalidXMLContentError"),
              mxResources.get("error"),
              JOptionPane.ERROR_MESSAGE);
        } catch (FileNotFoundException e) {
          // do nothing when no accompanying .layout.xml file exists
        }
      }

    }

    private void importGraphGeometry(mxGraph graph, Map<String, mxGeometry> geometry) {
      Map<String, Object> cells = ((mxGraphModel) graph.getModel()).getCells();
      geometry.forEach(
          (id, mxGeometry) -> {
            mxICell cell = (mxICell) cells.get(id);
            if (cell != null) {
              graph.getModel().setGeometry(cell, mxGeometry);
            }
          });
    }

    private void importGraphStyle(mxGraph graph, Map<String, String> styles) {
      Map<String, Object> cells = ((mxGraphModel) graph.getModel()).getCells();
      styles.forEach(
          (id, style) -> {
            mxICell cell = (mxICell) cells.get(id);
            if (cell != null) {
              graph.getModel().setStyle(cell, style);
            }
          });
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ToggleAction extends AbstractAction {

    /**
     *
     */
    protected String key;

    /**
     *
     */
    protected boolean defaultValue;

    /**
     * @param key
     */
    public ToggleAction(String key) {
      this(key, false);
    }

    /**
     * @param key
     */
    public ToggleAction(String key, boolean defaultValue) {
      this.key = key;
      this.defaultValue = defaultValue;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      mxGraph graph = mxGraphActions.getGraph(e);

      if (graph != null) {
        graph.toggleCellStyles(key, defaultValue);
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class SetLabelPositionAction extends AbstractAction {

    /**
     *
     */
    protected String labelPosition, alignment;

    public SetLabelPositionAction(String labelPosition, String alignment) {
      this.labelPosition = labelPosition;
      this.alignment = alignment;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      mxGraph graph = mxGraphActions.getGraph(e);

      if (graph != null && !graph.isSelectionEmpty()) {
        graph.getModel().beginUpdate();
        try {
          // Checks the orientation of the alignment to use the correct constants
          if (labelPosition.equals(mxConstants.ALIGN_LEFT)
              || labelPosition.equals(mxConstants.ALIGN_CENTER)
              || labelPosition.equals(mxConstants.ALIGN_RIGHT)) {
            graph.setCellStyles(mxConstants.STYLE_LABEL_POSITION, labelPosition);
            graph.setCellStyles(mxConstants.STYLE_ALIGN, alignment);
          } else {
            graph.setCellStyles(mxConstants.STYLE_VERTICAL_LABEL_POSITION, labelPosition);
            graph.setCellStyles(mxConstants.STYLE_VERTICAL_ALIGN, alignment);
          }
        } finally {
          graph.getModel().endUpdate();
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class SetStyleAction extends AbstractAction {

    /**
     *
     */
    protected String value;

    public SetStyleAction(String value) {
      this.value = value;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      mxGraph graph = mxGraphActions.getGraph(e);

      if (graph != null && !graph.isSelectionEmpty()) {
        graph.setCellStyle(value);
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class KeyValueAction extends AbstractAction {

    /**
     *
     */
    protected String key, value;

    /**
     * @param key
     */
    public KeyValueAction(String key) {
      this(key, null);
    }

    /**
     * @param key
     */
    public KeyValueAction(String key, String value) {
      this.key = key;
      this.value = value;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      mxGraph graph = mxGraphActions.getGraph(e);

      if (graph != null && !graph.isSelectionEmpty()) {
        graph.setCellStyles(key, value);
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class PromptValueAction extends AbstractAction {

    /**
     *
     */
    protected String key, message;

    /**
     * @param key
     */
    public PromptValueAction(String key, String message) {
      this.key = key;
      this.message = message;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof Component) {
        mxGraph graph = mxGraphActions.getGraph(e);

        if (graph != null && !graph.isSelectionEmpty()) {
          String value =
              (String)
                  JOptionPane.showInputDialog(
                      (Component) e.getSource(),
                      mxResources.get("value"),
                      message,
                      JOptionPane.PLAIN_MESSAGE,
                      null,
                      null,
                      "");

          if (value != null) {
            if (value.equals(mxConstants.NONE)) {
              value = null;
            }

            graph.setCellStyles(key, value);
          }
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class AlignCellsAction extends AbstractAction {

    /**
     *
     */
    protected String align;

    public AlignCellsAction(String align) {
      this.align = align;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      mxGraph graph = mxGraphActions.getGraph(e);

      if (graph != null && !graph.isSelectionEmpty()) {
        graph.alignCells(align);
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class AutosizeAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      mxGraph graph = mxGraphActions.getGraph(e);

      if (graph != null && !graph.isSelectionEmpty()) {
        Object[] cells = graph.getSelectionCells();
        mxIGraphModel model = graph.getModel();

        model.beginUpdate();
        try {
          for (int i = 0; i < cells.length; i++) {
            graph.updateCellSize(cells[i]);
          }
        } finally {
          model.endUpdate();
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class ColorAction extends AbstractAction {

    /**
     *
     */
    protected String name, key;

    /**
     * @param key
     */
    public ColorAction(String name, String key) {
      this.name = name;
      this.key = key;
    }

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        mxGraph graph = graphComponent.getGraph();

        if (!graph.isSelectionEmpty()) {
          Color newColor = JColorChooser.showDialog(graphComponent, name, null);

          if (newColor != null) {
            graph.setCellStyles(key, mxUtils.hexString(newColor));
          }
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class BackgroundImageAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        String value =
            (String)
                JOptionPane.showInputDialog(
                    graphComponent,
                    mxResources.get("backgroundImage"),
                    "URL",
                    JOptionPane.PLAIN_MESSAGE,
                    null,
                    null,
                    "http://www.callatecs.com/images/background2.JPG");

        if (value != null) {
          if (value.length() == 0) {
            graphComponent.setBackgroundImage(null);
          } else {
            Image background = mxUtils.loadImage(value);
            // Incorrect URLs will result in no image.
            // TODO: provide feedback that the URL is not correct
            if (background != null) {
              graphComponent.setBackgroundImage(new ImageIcon(background));
            }
          }

          // Forces a repaint of the outline
          graphComponent.getGraph().repaint();
        }
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class BackgroundAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        Color newColor =
            JColorChooser.showDialog(graphComponent, mxResources.get("background"), null);

        if (newColor != null) {
          graphComponent.getViewport().setOpaque(true);
          graphComponent.getViewport().setBackground(newColor);
        }

        // Forces a repaint of the outline
        graphComponent.getGraph().repaint();
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class PageBackgroundAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        Color newColor =
            JColorChooser.showDialog(graphComponent, mxResources.get("pageBackground"), null);

        if (newColor != null) {
          graphComponent.setPageBackgroundColor(newColor);
        }

        // Forces a repaint of the component
        graphComponent.repaint();
      }
    }
  }

  /**
   *
   */
  @SuppressWarnings("serial")
  public static class StyleAction extends AbstractAction {

    /**
     *
     */
    public void actionPerformed(ActionEvent e) {
      if (e.getSource() instanceof mxGraphComponent) {
        mxGraphComponent graphComponent = (mxGraphComponent) e.getSource();
        mxGraph graph = graphComponent.getGraph();
        String initial = graph.getModel().getStyle(graph.getSelectionCell());
        String value =
            (String)
                JOptionPane.showInputDialog(
                    graphComponent,
                    mxResources.get("style"),
                    mxResources.get("style"),
                    JOptionPane.PLAIN_MESSAGE,
                    null,
                    null,
                    initial);

        if (value != null) {
          graph.setCellStyle(value);
        }
      }
    }
  }

  /**
   * {@link AbstractAction} for opening a {@link EditorSizeEdit} to change the size of selected
   * cells in the {@link mxGraph}
   */
  public static class EditSizeAction extends AbstractAction {

    /**
     * Active graph, where cells should be resized
     */
    mxGraph graph;

    /**
     * Constructor for a {@link AbstractAction} to resize selected cells in the {@link mxGraph}
     *
     * @param graph active {@link mxGraph}
     */
    public EditSizeAction(mxGraph graph) {
      super("Edit");
      this.graph = graph;
    }

    /**
     * Listener for when {@link EditSizeAction} {@link AbstractAction} is called. Opens a
     * {@link EditorSizeEdit} {@link JDialog}
     *
     * @param e Standard {@link ActionEvent}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      new EditorSizeEdit(graph);
    }
  }
}
