package de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.editor;

import static de.uni_trier.wi2.procake.gui.Constants.NESTWORKFLOW_RESOURCES;

import com.mxgraph.layout.hierarchical.mxHierarchicalLayout;
import com.mxgraph.layout.mxCircleLayout;
import com.mxgraph.layout.mxCompactTreeLayout;
import com.mxgraph.layout.mxEdgeLabelLayout;
import com.mxgraph.layout.mxIGraphLayout;
import com.mxgraph.layout.mxOrganicLayout;
import com.mxgraph.layout.mxParallelEdgeLayout;
import com.mxgraph.layout.mxPartitionLayout;
import com.mxgraph.layout.mxStackLayout;
import com.mxgraph.swing.handler.mxKeyboardHandler;
import com.mxgraph.swing.handler.mxRubberband;
import com.mxgraph.swing.mxGraphComponent;
import com.mxgraph.swing.mxGraphOutline;
import com.mxgraph.swing.util.mxMorphing;
import com.mxgraph.util.mxEvent;
import com.mxgraph.util.mxEventObject;
import com.mxgraph.util.mxEventSource.mxIEventListener;
import com.mxgraph.util.mxRectangle;
import com.mxgraph.util.mxResources;
import com.mxgraph.util.mxUndoManager;
import com.mxgraph.util.mxUndoableEdit;
import com.mxgraph.util.mxUndoableEdit.mxUndoableChange;
import com.mxgraph.view.mxGraph;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.NESTWorkflowEditor;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.utils.Utils;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.io.File;
import java.util.List;
import javax.swing.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BasicGraphEditor extends JPanel {

  protected static final Logger logger = LoggerFactory.getLogger(BasicGraphEditor.class);
  /**
   *
   */
  private static final long serialVersionUID = -6561623072112577140L;

  /** Adds required resources for i18n */
  static {
    try {
      mxResources.add("editor");
    } catch (Exception e) {
      logger.error("Error while adding resource.", e);
    }
  }

  protected final boolean frameless;

  /**
   *
   */
  protected mxGraphComponent graphComponent;

  /**
   *
   */
  protected mxGraphOutline graphOutline;

  /**
   *
   */
  protected JTabbedPane libraryPane;

  /**
   *
   */
  protected mxUndoManager undoManager;

  /**
   *
   */
  protected String appTitle;

  /**
   *
   */
  protected JLabel statusBar;

  /**
   *
   */
  protected File currentFile;

  /**
   * Flag indicating whether the current graph has been modified
   */
  protected boolean modified = false;

  /**
   *
   */
  protected mxRubberband rubberband;

  /**
   *
   */
  protected mxKeyboardHandler keyboardHandler;

  /**
   *
   */
  protected mxIEventListener undoHandler =
      new mxIEventListener() {
        public void invoke(Object source, mxEventObject evt) {
          undoManager.undoableEditHappened((mxUndoableEdit) evt.getProperty("edit"));
        }
      };

  /**
   *
   */
  protected mxIEventListener changeTracker =
      new mxIEventListener() {
        public void invoke(Object source, mxEventObject evt) {
          setModified(true);
        }
      };

  /**
   * Starting Point of Mouse when clicked
   */
  protected Point start;

  /**
   *
   */
  public BasicGraphEditor(String appTitle, mxGraphComponent component, boolean frameless,
      boolean generateToolbar) {
    // Stores and updates the frame title
    this.appTitle = appTitle;

    this.frameless = frameless;

    // Stores a reference to the graph and creates the command history
    graphComponent = component;
    final mxGraph graph = graphComponent.getGraph();
    undoManager = createUndoManager();

    // Do not change the scale and translation after files have been loaded
    graph.setResetViewOnRootChange(false);

    // Updates the modified flag if the graph model changes
    graph.getModel().addListener(mxEvent.CHANGE, changeTracker);

    // Adds the command history to the model and view
    graph.getModel().addListener(mxEvent.UNDO, undoHandler);
    graph.getView().addListener(mxEvent.UNDO, undoHandler);

    // Keeps the selection in sync with the command history
    mxIEventListener undoHandler =
        new mxIEventListener() {
          public void invoke(Object source, mxEventObject evt) {
            List<mxUndoableChange> changes =
                ((mxUndoableEdit) evt.getProperty("edit")).getChanges();
            graph.setSelectionCells(graph.getSelectionCellsForChanges(changes));
          }
        };

    undoManager.addListener(mxEvent.UNDO, undoHandler);
    undoManager.addListener(mxEvent.REDO, undoHandler);

    // Creates the graph outline component
    graphOutline = new mxGraphOutline(graphComponent);
    graphOutline.setPreferredSize(
        new Dimension((int) graphOutline.getPreferredSize().getWidth(), 250));
    graphOutline.setMinimumSize(
        new Dimension((int) graphOutline.getPreferredSize().getWidth(), 150));

    // Creates the library pane that contains the tabs with the palettes
    libraryPane = new JTabbedPane();
    libraryPane.setPreferredSize(
        new Dimension((int) libraryPane.getPreferredSize().getWidth(), 150));

    JPanel sidebar = new JPanel();
    sidebar.setLayout(new BorderLayout());
    sidebar.add(libraryPane, BorderLayout.PAGE_START);
    sidebar.add(
        new JScrollPane(new EditorControlsPanel((NESTWorkflowEditor.CustomGraph) graph)),
        BorderLayout.CENTER);

    // Creates the inner split pane that contains the library with the
    // palettes and the graph outline on the left side of the window
    JSplitPane inner = new JSplitPane(JSplitPane.VERTICAL_SPLIT, sidebar, graphOutline);
    //        inner.setDividerLocation(320);
    inner.setResizeWeight(1);
    inner.setDividerSize(6);
    inner.setBorder(null);

    // Creates the outer split pane that contains the inner split pane and
    // the graph component on the right side of the window
    JSplitPane outer = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, inner, graphComponent);
    outer.setOneTouchExpandable(true);
    outer.setDividerLocation(300);
    outer.setDividerSize(6);
    outer.setBorder(null);

    // Creates the status bar
    // status bar is disabled
    //        statusBar = createStatusBar();

    // Display some useful information about repaint events
    installRepaintListener();

    // Puts everything together
    setLayout(new BorderLayout());
    add(outer, BorderLayout.CENTER);
    //        add(statusBar, BorderLayout.SOUTH); // status bar is disabled

    if (generateToolbar) {
      installToolBar();
    }

    // Installs rubberband selection and handling for some special
    // keystrokes such as F2, Control-C, -V, X, A etc.

    installHandlers();

    installListeners();
    updateTitle();
  }

  public BasicGraphEditor(String appTitle, mxGraphComponent component, boolean frameless) {
    this(appTitle, component, frameless, false);
  }

  /**
   *
   */
  protected mxUndoManager createUndoManager() {
    return new mxUndoManager();
  }

  /**
   *
   */
  protected void installHandlers() {
    rubberband = new EditorRubberband(graphComponent);
    keyboardHandler = new EditorKeyboardHandler(graphComponent);
  }

  /**
   *
   */
  protected void installToolBar() {
    add(new EditorToolBar(this, JToolBar.HORIZONTAL), BorderLayout.NORTH);
  }

  /**
   *
   */
  protected JLabel createStatusBar() {
    JLabel statusBar = new JLabel(mxResources.get("ready"));
    statusBar.setBorder(BorderFactory.createEmptyBorder(2, 4, 2, 4));

    return statusBar;
  }

  /**
   *
   */
  protected void installRepaintListener() {
    graphComponent
        .getGraph()
        .addListener(
            mxEvent.REPAINT,
            new mxIEventListener() {
              public void invoke(Object source, mxEventObject evt) {
                String buffer = (graphComponent.getTripleBuffer() != null) ? "" : " (unbuffered)";
                mxRectangle dirty = (mxRectangle) evt.getProperty("region");

                if (dirty == null) {
                  status("Repaint all" + buffer);
                } else {
                  status(
                      "Repaint: x="
                          + (int) (dirty.getX())
                          + " y="
                          + (int) (dirty.getY())
                          + " w="
                          + (int) (dirty.getWidth())
                          + " h="
                          + (int) (dirty.getHeight())
                          + buffer);
                }
              }
            });
  }

  /**
   *
   */
  public EditorPalette insertPalette(String title) {
    final EditorPalette palette = new EditorPalette();
    final JScrollPane scrollPane = new JScrollPane(palette);
    scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
    scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
    libraryPane.add(title, scrollPane);

    // Updates the widths of the palettes if the container size changes
    libraryPane.addComponentListener(
        new ComponentAdapter() {
          /** */
          public void componentResized(ComponentEvent e) {
            int w = scrollPane.getWidth() - scrollPane.getVerticalScrollBar().getWidth();
            palette.setPreferredWidth(w);
          }
        });

    return palette;
  }

  /**
   *
   */
  protected void mouseWheelMoved(MouseWheelEvent e) {
    if (e.getWheelRotation() < 0) {
      graphComponent.zoomIn();
    } else {
      graphComponent.zoomOut();
    }

    status(
        mxResources.get("scale")
            + ": "
            + (int) (100 * graphComponent.getGraph().getView().getScale())
            + "%");
  }

  /**
   *
   */
  protected void showOutlinePopupMenu(MouseEvent e) {
    Point pt = SwingUtilities.convertPoint(e.getComponent(), e.getPoint(), graphComponent);
    JCheckBoxMenuItem item = new JCheckBoxMenuItem(mxResources.get("magnifyPage"));
    item.setSelected(graphOutline.isFitPage());

    item.addActionListener(
        new ActionListener() {
          /** */
          public void actionPerformed(ActionEvent e) {
            graphOutline.setFitPage(!graphOutline.isFitPage());
            graphOutline.repaint();
          }
        });

    JCheckBoxMenuItem item2 = new JCheckBoxMenuItem(mxResources.get("showLabels"));
    item2.setSelected(graphOutline.isDrawLabels());

    item2.addActionListener(
        new ActionListener() {
          /** */
          public void actionPerformed(ActionEvent e) {
            graphOutline.setDrawLabels(!graphOutline.isDrawLabels());
            graphOutline.repaint();
          }
        });

    JCheckBoxMenuItem item3 = new JCheckBoxMenuItem(mxResources.get("buffering"));
    item3.setSelected(graphOutline.isTripleBuffered());

    item3.addActionListener(
        new ActionListener() {
          /** */
          public void actionPerformed(ActionEvent e) {
            graphOutline.setTripleBuffered(!graphOutline.isTripleBuffered());
            graphOutline.repaint();
          }
        });

    JPopupMenu menu = new JPopupMenu();
    menu.add(item);
    menu.add(item2);
    menu.add(item3);
    menu.show(graphComponent, pt.x, pt.y);

    e.consume();
  }

  /**
   *
   */
  protected void showGraphPopupMenu(MouseEvent e) {
    Point pt = SwingUtilities.convertPoint(e.getComponent(), e.getPoint(), graphComponent);
    EditorPopupMenu menu = new EditorPopupMenu(BasicGraphEditor.this);
    menu.show(graphComponent, pt.x, pt.y);

    e.consume();
  }

  /**
   *
   */
  protected void mouseLocationChanged(MouseEvent e) {
    status(e.getX() + ", " + e.getY());
  }

  /**
   *
   */
  protected void installListeners() {
    // Installs mouse wheel listener for zooming
    MouseWheelListener wheelTracker =
        new MouseWheelListener() {
          /** */
          public void mouseWheelMoved(MouseWheelEvent e) {
            if (e.getSource() instanceof mxGraphOutline || e.isControlDown()) {
              BasicGraphEditor.this.mouseWheelMoved(e);
            }
          }
        };

    // Handles mouse wheel events in the outline and graph component
    graphOutline.addMouseWheelListener(wheelTracker);
    graphComponent.addMouseWheelListener(wheelTracker);

    // Installs the popup menu in the outline
    graphOutline.addMouseListener(
        new MouseAdapter() {

          /** */
          public void mousePressed(MouseEvent e) {
            // Handles context menu on the Mac where the trigger is on mousepressed
            mouseReleased(e);
          }

          /** */
          public void mouseReleased(MouseEvent e) {
            if (e.isPopupTrigger()) {
              showOutlinePopupMenu(e);
            }
          }
        });

    // Installs the popup menu in the graph component
    graphComponent
        .getGraphControl()
        .addMouseListener(
            new MouseAdapter() {

              public void mousePressed(MouseEvent e) {
                // Handles context menu on the Mac where the trigger is on mousepressed
                mouseReleased(e);
                Object cell = graphComponent.getCellAt(e.getX(), e.getY());
                if (SwingUtilities.isRightMouseButton(e) && cell == null) {
                  start = e.getPoint();
                  graphComponent.getGraphControl()
                      .setCursor(new Cursor(Cursor.MOVE_CURSOR));
                }
              }

              public void mouseReleased(MouseEvent e) {
                // Disabled so the graph popup menu does not show after new edges were created with right mouse button drag and drop, instead the menu is created on mouseClicked
//                if (e.isPopupTrigger()) {
//                    showGraphPopupMenu(e);
//                }
                start = null;
                graphComponent.getGraphControl()
                    .setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
              }

              public void mouseClicked(MouseEvent e) {
                if (SwingUtilities.isRightMouseButton(e)) {
                  boolean cellAtMouseInSelection = false;
                  Object cell = graphComponent.getCellAt(e.getX(), e.getY());
                  Object[] selection = graphComponent.getGraph().getSelectionCells();
                  for (Object o : selection) {
                    if (o == cell) {
                      cellAtMouseInSelection = true;
                    }
                  }
                  if (!cellAtMouseInSelection && cell != null) {
                    graphComponent.selectCellForEvent(cell, e);
                    cell = null;
                  }
                  showGraphPopupMenu(e);
                }
              }
            });

    // Installs a mouse motion listener to display the mouse location
    graphComponent
        .getGraphControl()
        .addMouseMotionListener(
            new MouseMotionListener() {

              /*
               * (non-Javadoc)
               * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
               */
              public void mouseDragged(MouseEvent e) {
                mouseLocationChanged(e);
                if (start != null) {
                  double dx = ((start.getX() - e.getX()));
                  double dy = ((start.getY() - e.getY()));

                  graphComponent.getHorizontalScrollBar().setValue(
                      graphComponent.getHorizontalScrollBar().getValue()
                          + (int) dx);
                  graphComponent.getVerticalScrollBar().setValue(
                      graphComponent.getVerticalScrollBar().getValue()
                          + (int) dy);
                }
              }

              /*
               * (non-Javadoc)
               * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
               */
              public void mouseMoved(MouseEvent e) {
                mouseDragged(e);
              }
            });
  }

  /**
   *
   */
  public File getCurrentFile() {
    return currentFile;
  }

  /**
   *
   */
  public void setCurrentFile(File file) {
    File oldValue = currentFile;
    currentFile = file;

    firePropertyChange("currentFile", oldValue, file);

    if (oldValue != file) {
      updateTitle();
    }
  }

  /**
   * @return whether or not the current graph has been modified
   */
  public boolean isModified() {
    return modified;
  }

  /**
   * @param modified
   */
  public void setModified(boolean modified) {
    boolean oldValue = this.modified;
    this.modified = modified;

    firePropertyChange("modified", oldValue, modified);

    if (oldValue != modified) {
      updateTitle();
    }
  }

  /**
   *
   */
  public mxGraphComponent getGraphComponent() {
    return graphComponent;
  }

  /**
   *
   */
  public mxGraphOutline getGraphOutline() {
    return graphOutline;
  }

  /**
   *
   */
  public JTabbedPane getLibraryPane() {
    return libraryPane;
  }

  /**
   *
   */
  public mxUndoManager getUndoManager() {
    return undoManager;
  }

  /**
   * @param name
   * @param action
   * @return a new Action bound to the specified string name
   */
  public Action bind(String name, final Action action) {
    return bind(name, action, null);
  }

  /**
   * @param name
   * @param action
   * @return a new Action bound to the specified string name and icon
   */
  @SuppressWarnings("serial")
  public Action bind(String name, final Action action, String iconUrl) {
    AbstractAction newAction =
        new AbstractAction(
            name,
            (iconUrl != null) ? new ImageIcon(BasicGraphEditor.class.getResource(iconUrl)) : null) {
          public void actionPerformed(ActionEvent e) {
            action.actionPerformed(
                new ActionEvent(getGraphComponent(), e.getID(), e.getActionCommand()));
          }
        };

    newAction.putValue(Action.SHORT_DESCRIPTION, action.getValue(Action.SHORT_DESCRIPTION));

    return newAction;
  }

  /**
   * @param msg
   */
  public void status(String msg) {
    // status bar is disabled
    //        statusBar.setText(msg);
  }

  /**
   *
   */
  public void updateTitle() {
    JFrame frame = (JFrame) SwingUtilities.windowForComponent(this);

    if (frame != null && !this.frameless) {
      String title = ((NESTWorkflowEditor) this).getNESTWorkflow().getId();

      //            if (modified) {
      //                title += "*";
      //            }

      frame.setTitle(title + " - " + appTitle);
    }
  }

  /**
   *
   */
  public void about() {
    JFrame frame = (JFrame) SwingUtilities.windowForComponent(this);

    if (frame != null) {
      EditorAboutFrame about = new EditorAboutFrame(frame);
      about.setModal(true);

      // Centers inside the application frame
      int x = frame.getX() + (frame.getWidth() - about.getWidth()) / 2;
      int y = frame.getY() + (frame.getHeight() - about.getHeight()) / 2;
      about.setLocation(x, y);

      // Shows the modal dialog and waits
      about.setVisible(true);
    }
  }

  /**
   *
   */
  public void exit() {
    JFrame frame = (JFrame) SwingUtilities.windowForComponent(this);

    if (frame != null) {
      frame.dispose();
    }
  }

  /**
   *
   */
  public void setLookAndFeel(String clazz) {
    JFrame frame = (JFrame) SwingUtilities.windowForComponent(this);

    if (frame != null) {
      try {
        UIManager.setLookAndFeel(clazz);
        Utils.setDefaultFontSize(11.5f); // resolves dpi scaling issue (label fonts too small)
        SwingUtilities.updateComponentTreeUI(frame);

        // Needs to assign the key bindings again
        keyboardHandler = new EditorKeyboardHandler(graphComponent);
      } catch (Exception e1) {
        e1.printStackTrace();
      }
    }
  }

  /**
   * Prepares a Frame and adds the View of the Editor to it
   *
   * @param menuBar   The menuBar that should be added to the Frame
   * @param splitPane Creates and initializes splitPane and adds the View to it, while adding it to
   *                  the Frame
   */
  public JFrame createFrame(JMenuBar menuBar, JSplitPane splitPane) {
    JFrame frame = new JFrame();

    frame.setIconImage(Toolkit.getDefaultToolkit()
        .getImage(getClass().getResource("/images/ProCAKE_Icon.png")));

    splitPane.setRightComponent(this);

    frame.getContentPane().add(splitPane);
    frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    frame.setJMenuBar(menuBar);
    //		frame.setSize(870, 640);
    frame.setExtendedState(frame.getExtendedState() | Frame.MAXIMIZED_BOTH);

    // Updates the frame title
    updateTitle();

    return frame;
  }

  /**
   * Creates an action that executes the specified layout.
   *
   * @param key Key to be used for getting the label from mxResources and also to create the layout
   *            instance for the commercial graph editor example.
   * @return an action that executes the specified layout
   */
  @SuppressWarnings("serial")
  public Action graphLayout(final String key, boolean animate) {
    final mxIGraphLayout layout = createLayout(key, animate);

    if (layout != null) {
      return new AbstractAction(mxResources.get(key)) {
        public void actionPerformed(ActionEvent e) {
          final mxGraph graph = graphComponent.getGraph();
          Object cell = graph.getSelectionCell();

          if (cell == null || graph.getModel().getChildCount(cell) == 0) {
            cell = graph.getDefaultParent();
          }

          graph.getModel().beginUpdate();
          try {
            long t0 = System.currentTimeMillis();
            layout.execute(cell);
            status("Layout: " + (System.currentTimeMillis() - t0) + " ms");
          } finally {
            mxMorphing morph = new mxMorphing(graphComponent, 20, 1.2, 20);

            morph.addListener(
                mxEvent.DONE,
                new mxIEventListener() {

                  public void invoke(Object sender, mxEventObject evt) {
                    graph.getModel().endUpdate();
                  }
                });

            morph.startAnimation();
          }
        }
      };
    } else {
      return new AbstractAction(mxResources.get(key)) {

        public void actionPerformed(ActionEvent e) {
          JOptionPane.showMessageDialog(graphComponent, mxResources.get("noLayout"));
        }
      };
    }
  }

  /**
   * Creates a layout instance for the given identifier.
   */
  protected mxIGraphLayout createLayout(String ident, boolean animate) {
    mxIGraphLayout layout = null;

    if (ident != null) {
      mxGraph graph = graphComponent.getGraph();

      if (ident.equals("verticalHierarchical")) {
        layout = new mxHierarchicalLayout(graph);
      } else if (ident.equals("horizontalHierarchical")) {
        layout = new mxHierarchicalLayout(graph, JLabel.WEST);
      } else if (ident.equals("verticalTree")) {
        layout = new mxCompactTreeLayout(graph, false);
      } else if (ident.equals("horizontalTree")) {
        layout = new mxCompactTreeLayout(graph, true);
      } else if (ident.equals("parallelEdges")) {
        layout = new mxParallelEdgeLayout(graph);
      } else if (ident.equals("placeEdgeLabels")) {
        layout = new mxEdgeLabelLayout(graph);
      } else if (ident.equals("organicLayout")) {
        layout = new mxOrganicLayout(graph);
      }
      if (ident.equals("verticalPartition")) {
        layout =
            new mxPartitionLayout(graph, false) {
              /** Overrides the empty implementation to return the size of the graph control. */
              public mxRectangle getContainerSize() {
                return graphComponent.getLayoutAreaSize();
              }
            };
      } else if (ident.equals("horizontalPartition")) {
        layout =
            new mxPartitionLayout(graph, true) {
              /** Overrides the empty implementation to return the size of the graph control. */
              public mxRectangle getContainerSize() {
                return graphComponent.getLayoutAreaSize();
              }
            };
      } else if (ident.equals("verticalStack")) {
        layout =
            new mxStackLayout(graph, false) {
              /** Overrides the empty implementation to return the size of the graph control. */
              public mxRectangle getContainerSize() {
                return graphComponent.getLayoutAreaSize();
              }
            };
      } else if (ident.equals("horizontalStack")) {
        layout =
            new mxStackLayout(graph, true) {
              /** Overrides the empty implementation to return the size of the graph control. */
              public mxRectangle getContainerSize() {
                return graphComponent.getLayoutAreaSize();
              }
            };
      } else if (ident.equals("circleLayout")) {
        layout = new mxCircleLayout(graph);
      }
    }

    return layout;
  }
}
