package de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor;

import com.mxgraph.model.mxICell;
import com.mxgraph.swing.mxGraphComponent;
import com.mxgraph.swing.view.mxICellEditor;
import com.mxgraph.view.mxCellState;
import de.uni_trier.wi2.procake.data.model.DataClass;
import de.uni_trier.wi2.procake.data.model.nest.NESTGraphItemClass;
import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTGraphItemObject;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.utils.Utils;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.swing.layouts.BasicGridLayout;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.swing.misc.AutoCompletion;

import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EventObject;
import java.util.Set;
import java.util.stream.Collectors;
import javax.swing.*;


public class SemanticDescriptorEditor implements mxICellEditor {

  public static final String[] SEMANTIC_DESCRIPTOR_CLASS_SELECTOR_BLACKLIST = new String[]{
      "NESTGraphElement", "AbstractWorkflowItem"};
  static String SEMANTIC_DESCRIPTOR_EDITOR_COMPONENT_NAME = "SemanticDescriptorEditorComponent";
  private mxGraphComponent graphComponent;
  private transient Object editingCell;

  SemanticDescriptorEditor(mxGraphComponent graphComponent) {
    this.graphComponent = graphComponent;
  }

  @Override
  public Object getEditingCell() {
    return editingCell;
  }

  @Override
  public void startEditing(Object cell, EventObject eventObject) {
    if (editingCell != null) {
      stopEditing(true);
    }
    new EditorWindow((mxICell) cell);
  }

  @Override
  public void stopEditing(boolean cancel) {
    editingCell = null;
  }

  class EditorWindow extends JDialog {

    private DataObjectEditor dataObjectEditor = new DataObjectEditor(null);
    private NESTGraphItemObject graphItem;
    private DataObject modifiedSemanticDescriptor;

    // TODO: https://stackoverflow.com/questions/24361899/frame-always-on-top-of-my-program-only
    // https://stackoverflow.com/questions/24476496/drag-and-resize-undecorated-jframe
    // https://stackoverflow.com/questions/16046824/making-a-java-swing-frame-movable-and-setundecorated
    // https://tips4java.wordpress.com/2009/06/14/moving-windows/
    // https://tips4java.wordpress.com/2009/09/13/resizing-components/
    public EditorWindow(mxICell cell) {
      super((JFrame) SwingUtilities.getWindowAncestor(graphComponent));
      this.setName(SEMANTIC_DESCRIPTOR_EDITOR_COMPONENT_NAME);
      this.graphItem = (NESTGraphItemObject) cell.getValue();
      this.modifiedSemanticDescriptor =
          graphItem.getSemanticDescriptor() != null
              ? graphItem.getSemanticDescriptor().copy()
              : null;

      this.setLayout(new BorderLayout());
      this.addWindowListener(
          new WindowAdapter() {
            public void windowClosed(WindowEvent e) {
              graphComponent.labelChanged(cell, graphItem, null);
              graphComponent.requestFocusInWindow();
            }

            public void windowClosing(WindowEvent e) {
              // do nothing
            }

          });

      this.init();
      this.pack();
      Point mouseLocation = MouseInfo.getPointerInfo().getLocation();
      this.setLocation(
          Math.max(0, (int) (mouseLocation.getX() - this.getWidth() / 2)),
          Math.max(0, (int) (mouseLocation.getY() - this.getHeight() / 2)));
      this.setDefaultCloseOperation(DISPOSE_ON_CLOSE);
      this.setVisible(true);
    }

    private void init() {
      this.getContentPane().removeAll();
      this.setTitle("ID: " + graphItem.getId() + " (" + graphItem.getDataClass() + ")");
      this.addSemanticDescriptorSelectorAndSaveButton(graphItem);
      this.addDataObjectEditor(this.modifiedSemanticDescriptor);
      this.revalidate();
    }

    private void addDataObjectEditor(DataObject dataObject) {
      this.dataObjectEditor.setDataObject(dataObject);
      this.dataObjectEditor.setAlignmentX(Component.LEFT_ALIGNMENT);
      int borderWidth = 20;
      this.dataObjectEditor.setBorder(
          BorderFactory.createEmptyBorder(borderWidth, borderWidth, borderWidth, borderWidth));
      JScrollPane scrollPane = new JScrollPane(this.dataObjectEditor);
      scrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
      scrollPane.setBorder(BorderFactory.createEmptyBorder());
      this.add(scrollPane, BorderLayout.CENTER);
      this.dataObjectEditor.reload();
    }

    private JButton createSemanticDescriptorSaveButton(NESTGraphItemObject graphItem) {
      JButton saveButton = new JButton("Save Changes");
      saveButton.setMargin(new Insets(2, 2, 2, 2));
      saveButton.addActionListener(
          e -> {
            graphItem.setSemanticDescriptor(modifiedSemanticDescriptor);
            this.dispose();
          });
      return saveButton;
    }

    private void addSemanticDescriptorSelectorAndSaveButton(NESTGraphItemObject graphItem) {
      JPanel panel = new JPanel(new BasicGridLayout(1, 2));
      panel.add(createSemanticDescriptorSelector(graphItem));
      panel.add(createSemanticDescriptorSaveButton(graphItem));
      this.add(panel, BorderLayout.PAGE_START);
    }

    private JComboBox createSemanticDescriptorSelector(NESTGraphItemObject graphItem) {
      java.util.List<DataClass> possibleSemanticDescriptorClasses = new ArrayList<>();
      DataClass semanticDescriptorClass = ((NESTGraphItemClass) graphItem.getDataClass()).getSemanticDescriptorClass();
      if (semanticDescriptorClass != null && semanticDescriptorClass.getSuperClass() != null) {
        possibleSemanticDescriptorClasses.add(semanticDescriptorClass);
        possibleSemanticDescriptorClasses.addAll(Utils.getSubClassesDeep(semanticDescriptorClass));
      } else {
        Set<DataClass> semanticDescriptorClassSelectorBlacklist = Arrays.stream(
                SEMANTIC_DESCRIPTOR_CLASS_SELECTOR_BLACKLIST)
            .map(className -> (DataClass) graphItem.getModel().getClass(className))
            .collect(Collectors.toSet());
        possibleSemanticDescriptorClasses.addAll(graphItem.getModel().getClasses().stream()
            .filter(dataClass -> !semanticDescriptorClassSelectorBlacklist.contains(dataClass))
            .filter(dataClass -> semanticDescriptorClassSelectorBlacklist.stream()
                .noneMatch(dataClass::isSubclassOf)
            ).toList());
      }
      possibleSemanticDescriptorClasses =
          possibleSemanticDescriptorClasses.stream()
              .filter(dataClass -> dataClass.isInstantiable() && !dataClass.isUnion())
              .sorted(Comparator.comparing(DataClass::getName, String.CASE_INSENSITIVE_ORDER))
              .collect(Collectors.toList());
      possibleSemanticDescriptorClasses.add(
          0, null); // for when semantic descriptor is null or for deleting semantic descriptor

      JComboBox semanticDescriptorSelector =
          new JComboBox(possibleSemanticDescriptorClasses.toArray(DataClass[]::new));
      ListCellRenderer originalRenderer = semanticDescriptorSelector.getRenderer();
      semanticDescriptorSelector.setRenderer(
          (list, value, index, isSelected, cellHasFocus) ->
              originalRenderer.getListCellRendererComponent(
                  list, value == null ? "null" : value, index, isSelected, cellHasFocus));
      DataClass selectedSemanticDescriptorClass =
          possibleSemanticDescriptorClasses.stream()
              .filter(
                  dataClass ->
                      this.modifiedSemanticDescriptor != null
                          && this.modifiedSemanticDescriptor.getDataClass() == dataClass)
              .findAny()
              .orElse(possibleSemanticDescriptorClasses.get(0));
      semanticDescriptorSelector.setSelectedItem(selectedSemanticDescriptorClass);

      semanticDescriptorSelector.addActionListener(
          e -> {
            DataClass selectedDataClass = (DataClass) semanticDescriptorSelector.getSelectedItem();
            if (selectedDataClass == null) {
              this.modifiedSemanticDescriptor = null;
              this.dataObjectEditor.setDataObject(null);
              this.dataObjectEditor.reload();
            } else if (this.modifiedSemanticDescriptor == null
                || !this.modifiedSemanticDescriptor
                .getDataClass()
                .getName()
                .equals(
                    selectedDataClass
                        .getName())) { // only overwrite SemanticDescriptor with a new object
              // when a different class from previously is selected
              DataObject newSemanticDescriptorObject = selectedDataClass.newObject();
              this.modifiedSemanticDescriptor = newSemanticDescriptorObject;
              this.dataObjectEditor.setDataObject(newSemanticDescriptorObject);
              this.dataObjectEditor.reload();
            }
          });
      AutoCompletion.enable(semanticDescriptorSelector);
      semanticDescriptorSelector.setAlignmentX(Component.LEFT_ALIGNMENT);
      return semanticDescriptorSelector;
    }

    private Rectangle getEditorBounds(mxCellState state) {
      int positionX = (int) state.getBoundingBox().getX();
      int positionY = (int) state.getBoundingBox().getY();
      int cellWidth = (int) state.getBoundingBox().getWidth();
      int cellHeight = (int) state.getBoundingBox().getHeight();
      Dimension editorSize = this.getPreferredSize();
      return new Rectangle(
          positionX,
          positionY,
          Math.max((int) editorSize.getWidth(), cellWidth),
          Math.max((int) editorSize.getHeight(), cellHeight));
    }

    public mxGraphComponent getGraphComponent() {
      return graphComponent;
    }
  }
}
