package de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor;

import de.uni_trier.wi2.procake.data.object.nest.*;
import de.uni_trier.wi2.procake.data.object.nest.controlflowNode.NESTControlflowNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.utils.impl.NESTSequentialWorkflowValidatorImpl;
import de.uni_trier.wi2.procake.data.object.nest.utils.impl.NESTWorkflowValidatorImpl;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.utils.Utils;
import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlTransient;
import org.apache.commons.lang3.ObjectUtils;
import java.awt.*;
import java.util.List;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * How to use: Implement abstract methods delivering node and edge dimensions. Init with
 * NESTWorkflowObject. Set layout options or leave defaults. Call execute() method. Retrieve node
 * positions and edge paths with the corresponding methods (getNodeXPositions, getNodeYPositions,
 * getEdgePaths)
 */


public abstract class NESTWorkflowLayout {

  @XmlRootElement
  @XmlAccessorType(XmlAccessType.FIELD)
  // prevent JAXB from using setters to write values to prevent execution of the "execute()" method
  public static class LayoutConfig {

    @XmlTransient
    private NESTWorkflowLayout nestWorkflowLayout;
    protected boolean combineReverseDataflowEdges = true;
    private int nodeWidth = 50;
    private int nodeHeight = 50;
    private int nodeVerticalSpacing = 50;
    private int taskNodeToDataNodeVerticalSpacing = 50;
    private int sequenceNodeVerticalSpacing = 0;
    private int graphLeftMargin = 50;
    private int graphTopMargin = 50;
    private int sequenceNodesHorizontalSpacing = 50;
    private int controlflowEdgeLabelHorizontalSpacing = 50;
    private boolean placeDataNodesVerticallyNearTaskNodes = true;
    private boolean alsoPlaceDataNodesAboveTaskNodes = false;
    private boolean orthogonalDataflowEdgeRouting = false;
    private int idealNudgingDistance = 15;
    private int shapeBufferDistance = 10;

    public LayoutConfig(NESTWorkflowLayout nestWorkflowLayout) {
      this.nestWorkflowLayout = nestWorkflowLayout;
    }

    public LayoutConfig() {
    } // needed for JAXB

    public int getNodeHeight() {
      return nodeHeight;
    }

    public void setNodeHeight(int nodeHeight) {
      this.nodeHeight = nodeHeight;
      nestWorkflowLayout.execute();
    }

    public int getNodeVerticalSpacing() {
      return nodeVerticalSpacing;
    }

    public void setNodeVerticalSpacing(int nodeVerticalSpacing) {
      this.nodeVerticalSpacing = nodeVerticalSpacing;
      nestWorkflowLayout.execute();
    }

    public void setIdealNudgingDistance(int idealNudgingDistance) {
      this.idealNudgingDistance = idealNudgingDistance;
      nestWorkflowLayout.execute();
    }

    public void setShapeBufferDistance(int shapeBufferDistance) {
      this.shapeBufferDistance = shapeBufferDistance;
      nestWorkflowLayout.execute();
    }

    public void setAlsoPlaceDataNodesAboveTaskNodes(boolean alsoPlaceDataNodesAboveTaskNodes) {
      this.alsoPlaceDataNodesAboveTaskNodes = alsoPlaceDataNodesAboveTaskNodes;
      nestWorkflowLayout.execute();
    }

    public int getGraphLeftMargin() {
      return graphLeftMargin;
    }

    public void setGraphLeftMargin(int graphLeftMargin) {
      this.graphLeftMargin = graphLeftMargin;
      nestWorkflowLayout.execute();
    }

    public void setSequenceNodeVerticalSpacing(int sequenceNodeVerticalSpacing) {
      this.sequenceNodeVerticalSpacing = sequenceNodeVerticalSpacing;
      nestWorkflowLayout.execute();
    }

    public void setTaskNodeToDataNodeVerticalSpacing(int taskNodeToDataNodeVerticalSpacing) {
      this.taskNodeToDataNodeVerticalSpacing = taskNodeToDataNodeVerticalSpacing;
      nestWorkflowLayout.execute();
    }

    public int getSequenceNodesHorizontalSpacing() {
      return sequenceNodesHorizontalSpacing;
    }

    public void setSequenceNodesHorizontalSpacing(int sequenceNodesHorizontalSpacing) {
      this.sequenceNodesHorizontalSpacing = sequenceNodesHorizontalSpacing;
      nestWorkflowLayout.execute();
    }

    public int getControlflowEdgeLabelHorizontalSpacing() {
      return controlflowEdgeLabelHorizontalSpacing;
    }

    public void setControlflowEdgeLabelHorizontalSpacing(
        int controlflowEdgeLabelHorizontalSpacing) {
      this.controlflowEdgeLabelHorizontalSpacing = controlflowEdgeLabelHorizontalSpacing;
      nestWorkflowLayout.execute();
    }

    public boolean isCombineReverseDataflowEdges() {
      return combineReverseDataflowEdges;
    }

    public void setCombineReverseDataflowEdges(boolean combineReverseDataflowEdges) {
      this.combineReverseDataflowEdges = combineReverseDataflowEdges;
      nestWorkflowLayout.execute();
    }

    public boolean isPlaceDataNodesVerticallyNearTaskNodes() {
      return placeDataNodesVerticallyNearTaskNodes;
    }

    public void setPlaceDataNodesVerticallyNearTaskNodes(
        boolean placeDataNodesVerticallyNearTaskNodes) {
      this.placeDataNodesVerticallyNearTaskNodes = placeDataNodesVerticallyNearTaskNodes;
      nestWorkflowLayout.execute();
    }

    public boolean isOrthogonalDataflowEdgeRouting() {
      return orthogonalDataflowEdgeRouting;
    }

    public void setOrthogonalDataflowEdgeRouting(boolean orthogonalDataflowEdgeRouting) {
      this.orthogonalDataflowEdgeRouting = orthogonalDataflowEdgeRouting;
      nestWorkflowLayout.execute();
    }

    public int getTaskNodeToDataNodeVerticalSpacing() {
      return taskNodeToDataNodeVerticalSpacing;
    }

    public int getNodeWidth() {
      return nodeWidth;
    }

    public int getSequenceNodeVerticalSpacing() {
      return sequenceNodeVerticalSpacing;
    }

    public int getGraphTopMargin() {
      return graphTopMargin;
    }

    public void setGraphTopMargin(int graphTopMargin) {
      this.graphTopMargin = graphTopMargin;
    }

    public boolean isAlsoPlaceDataNodesAboveTaskNodes() {
      return alsoPlaceDataNodesAboveTaskNodes;
    }

    public int getIdealNudgingDistance() {
      return idealNudgingDistance;
    }

    public int getShapeBufferDistance() {
      return shapeBufferDistance;
    }

    public void setNestWorkflowLayout(NESTWorkflowLayout nestWorkflowLayout) {
      this.nestWorkflowLayout = nestWorkflowLayout;
    }
  }

  protected LayoutConfig layoutConfig = new LayoutConfig(this);
  public static boolean DEFAULT_EXECUTE_ON_EDGE_INSERTION = false;
  private boolean executeOnEdgeInsertion = DEFAULT_EXECUTE_ON_EDGE_INSERTION;

  protected NESTAbstractWorkflowObject nestWorkflow;
  protected Map<NESTNodeObject, Integer> nodeYPositions = new HashMap<>();
  protected Map<NESTNodeObject, Integer> nodeXPositions = new HashMap<>();
  protected Map<String, Integer> nodeYPositionsById = new HashMap<>();
  protected Map<String, Integer> nodeXPositionsById = new HashMap<>();
  protected Map<NESTEdgeObject, List<java.awt.Point>> edgePaths = new HashMap<>();

  private int currentBranchXOffset = 0; // needed for ordering of data nodes in their sequence

  public NESTWorkflowLayout(NESTAbstractWorkflowObject nestWorkflow) {
    this.nestWorkflow = nestWorkflow;
  }

  public void execute() {
    nodeYPositions.clear();
    nodeXPositions.clear();
    edgePaths.clear();
    this.layoutAllBranches(this.nestWorkflow);
    this.layoutWorkflowNodesX();
    this.resolveNodeOverlapsX();
    this.shiftGraphToPositivePositions();
    if (this.layoutConfig.isOrthogonalDataflowEdgeRouting()) {
      // this.routeDataflowEdges();
    }
  }

  private void layoutAllBranches(NESTAbstractWorkflowObject nestWorkflow) {
    Set<NESTSequenceNodeObject> allStartNodes = nestWorkflow.getStartNodes();
    int[] maximumX = {0};
    this.traverseAllWorkflowTrees(
        nestWorkflow,
        workflowNode -> {
          List<NESTSequenceNodeObject> workflowStartNodes =
              allStartNodes.stream()
                  .filter(
                      startNode ->
                          startNode.getOutgoingEdges(NESTEdgeObject::isNESTPartOfEdge).stream()
                              .anyMatch(
                                  partOfEdge ->
                                      partOfEdge.getPost()
                                          == workflowNode)) // all start nodes that are part of the
                  // current workflowNode
                  .sorted(Comparator.comparing(NESTSequenceNodeObject::getId))
                  .collect(Collectors.toList());
          workflowStartNodes.forEach(
              workflowStartNode -> {
                Branch workflowBranch = new Branch(workflowStartNode);
                maximumX[0] = this.layoutBranchX(workflowBranch, maximumX[0]);
                this.layoutNodesY(workflowBranch);
                this.layoutControlflowEdges(workflowBranch);
                allStartNodes.remove(workflowStartNode);
              });
        });

    // catch start nodes that are not connected to a (sub)workflow node
    allStartNodes.forEach(
        startNode -> {
          Branch branch = new Branch(startNode);
          maximumX[0] = this.layoutBranchX(branch, maximumX[0]);
          this.layoutNodesY(branch);
          this.layoutControlflowEdges(branch);
        });
  }

  private void layoutControlflowEdges(Branch branch) {
    this.edgePaths.putAll(branch.getControlflowEdgePaths());
  }

  private void traverseAllWorkflowTrees(
      NESTAbstractWorkflowObject nestWorkflow, Consumer<NESTNodeObject> consumer) {
    this.getWorkflowRootNodes(nestWorkflow)
        .forEach(rootNode -> this.traverseWorkflowTree(rootNode, consumer));
  }

  /**
   * @param nestWorkflow
   * @return workflow and subworkflow root nodes in the graph
   */
  private List<NESTNodeObject> getWorkflowRootNodes(NESTAbstractWorkflowObject nestWorkflow) {
    return nestWorkflow
        .getGraphNodes(node -> node.isNESTWorkflowNode() || node.isNESTSubWorkflowNode())
        .stream()
        .map(this::getRootWorkflowNode)
        .collect(Collectors.toSet())
        .stream() // remove duplicates
        .sorted(
            Comparator.comparing(NESTNodeObject::isNESTSubWorkflowNode)
                .thenComparing(NESTNodeObject::getId)) // sort to ensure determinism
        .collect(Collectors.toList());
  }

  /**
   * @param node
   * @return The root workflow or subworkflow node of the node
   */
  private NESTNodeObject getRootWorkflowNode(NESTNodeObject node) {
    if (node == null) {
      return null;
    }
    NESTNodeObject parentWorkflowNode =
        node.getOutgoingEdges(NESTEdgeObject::isNESTPartOfEdge).stream()
            .map(NESTEdgeObject::getPost)
            .filter(postNode -> postNode.isNESTWorkflowNode() || postNode.isNESTSubWorkflowNode())
            .findAny()
            .orElse(null);
    if (parentWorkflowNode == null) {
      return node.isNESTWorkflowNode() || node.isNESTSubWorkflowNode() ? node : null;
    }
    return getRootWorkflowNode(parentWorkflowNode);
  }

  /**
   * @param node     A NESTWorkflowNode or a NESTSubWorkflowNode
   * @param consumer The function that is applied to each (sub)workflow node
   */
  private void traverseWorkflowTree(NESTNodeObject node, Consumer<NESTNodeObject> consumer) {
    consumer.accept(node);
    List<NESTNodeObject> childWorkflowNodes =
        node.getIngoingEdges(NESTEdgeObject::isNESTPartOfEdge).stream()
            .map(NESTEdgeObject::getPre)
            .filter(Objects::nonNull)
            .filter(preNode -> preNode.isNESTWorkflowNode() || preNode.isNESTSubWorkflowNode())
            .sorted(Comparator.comparing(NESTNodeObject::getId)) // sort to ensure determinism
            .collect(Collectors.toList());
    childWorkflowNodes.forEach(workflowNode -> traverseWorkflowTree(workflowNode, consumer));
  }

  private int layoutBranchX(Branch branch, int startX) {
    if (layoutConfig.isPlaceDataNodesVerticallyNearTaskNodes()) {
      branch.assignDataNodesToSequences(this.nestWorkflow.getDataNodes());
    }
    branch.calculateSequenceNodeHorizontalPositions();

    if (layoutConfig.isPlaceDataNodesVerticallyNearTaskNodes()
        && layoutConfig.isAlsoPlaceDataNodesAboveTaskNodes()) {
      this.nodeXPositions.putAll(branch.getNodesXPositions(startX));
      branch.distributeDataNodes();
    }

    this.nodeXPositions.putAll(branch.getNodesXPositions(startX));

    if (!layoutConfig.isPlaceDataNodesVerticallyNearTaskNodes()) {
      layoutDataNodesX();
    }
    return branch.getMaximumX();
  }

  // TODO: As soon as libavoid is integrated into the next Eclipse ELK release
/*
  private LibavoidLayoutProvider getLayoutProvider() {
    LibavoidLayoutProvider layoutProvider = new LibavoidLayoutProvider();
    layoutProvider.initialize(null);
    LayoutMetaDataService service = LayoutMetaDataService.getInstance();
    // https://www.eclipse.org/elk/reference/algorithms/org-eclipse-elk-alg-libavoid.html
    service.registerLayoutMetaDataProviders(new LibavoidMetaDataProvider());
    return layoutProvider;
  }

  private void routeDataflowEdges() {
    // Build temporary graph to pass to adaptagrams/libavoid via ELK
    Set<NESTNodeObject> graphNodes = this.nestWorkflow.getGraphNodes();
    Map<NESTNodeObject, ElkNode> shapeRefs = new HashMap<>();

    // Set libavoid algorithm options
    ElkNode root = ElkGraphUtil.createGraph();
    root.setProperty(CoreOptions.ALGORITHM, LibavoidOptions.ALGORITHM_ID);
    root.setProperty(
        LibavoidOptions.IDEAL_NUDGING_DISTANCE,
        ((double) this.layoutConfig.getIdealNudgingDistance())
    );
    root.setProperty(
        LibavoidOptions.NUDGE_ORTHOGONAL_SEGMENTS_CONNECTED_TO_SHAPES,
        true
    );
    root.setProperty(
        LibavoidOptions.SHAPE_BUFFER_DISTANCE,
        ((double) this.layoutConfig.getShapeBufferDistance())
    );
    root.setProperty(LibavoidOptions.SEGMENT_PENALTY, 1.0);
    root.setProperty(LibavoidOptions.EDGE_ROUTING, EdgeRouting.ORTHOGONAL);

    // Attach NESTNodes to temporary root node
    graphNodes.forEach(
        nestNodeObject -> {
          var node = ElkGraphUtil.createNode(root);
          node.setLocation(
              this.nodeXPositions.get(nestNodeObject),
              this.nodeYPositions.get(nestNodeObject)
          );
          node.setDimensions(
              this.getNodeSize(nestNodeObject).getWidth(),
              this.getNodeSize(nestNodeObject).getHeight()
          );
          node.setProperty(CoreOptions.PORT_CONSTRAINTS, PortConstraints.FIXED_POS);
          node.setProperty(CoreOptions.PORT_ALIGNMENT_DEFAULT, PortAlignment.CENTER);
          ElkGraphUtil.createPort(node).setProperty(CoreOptions.PORT_SIDE, PortSide.NORTH);
          ElkGraphUtil.createPort(node).setProperty(CoreOptions.PORT_SIDE, PortSide.SOUTH);
          // Add shape to router and save mapping between NESTNodeObject and ShapeRef
          shapeRefs.put(nestNodeObject, node);
        }
    );

    Set<NESTDataflowEdgeObject> dataflowEdges = this.nestWorkflow.getDataflowEdges();
    Map<NESTEdgeObject, ElkEdge> connRefs = new HashMap<>();
    dataflowEdges.forEach(
        edgeObject -> {
          NESTNodeObject pre = edgeObject.getPre();
          NESTNodeObject post = edgeObject.getPost();
          if (pre != null && post != null) {
            if (!layoutConfig.isCombineReverseDataflowEdges()
                || findReverseDataflowEdge(edgeObject).isEmpty()
                || !connRefs.containsKey(findReverseDataflowEdge(edgeObject).get())) {
              boolean isPostCellBelow = nodeYPositions.get(pre) < nodeYPositions.get(post);
              // Select ports on the nodes, so the edges point in the right direction
              ElkEdge edge = ElkGraphUtil.createSimpleEdge(
                  shapeRefs.get(pre).getPorts().get(isPostCellBelow ? 1 : 0),
                  shapeRefs.get(post).getPorts().get(isPostCellBelow ? 0 : 1)
              );

              connRefs.put(edgeObject, edge);
            }
          }
        });

    // Perform orthogonal edge routing
    LibavoidLayoutProvider layoutProvider = getLayoutProvider();
    layoutProvider.layout(root, new BasicProgressMonitor());
    layoutProvider.dispose();

    // Update edge paths
    var edgePaths = (
        connRefs.entrySet().stream().collect(
            Collectors.toMap(
                Map.Entry::getKey,
                edge -> {
                  var route = edge.getValue().getSections();
                  List<Point> points = new LinkedList<>();
                  // ELK edge section structure: start - bendpoint* - end
                  for (ElkEdgeSection section : route) {
                    points.add(new Point((int) section.getStartX(), (int) section.getStartY()));
                    for (ElkBendPoint bendPoint : section.getBendPoints()) {
                      points.add(new Point((int) bendPoint.getX(), (int) bendPoint.getY()));
                    }
                    points.add(new Point((int) section.getEndX(), (int) section.getEndY()));
                  }
                  return points;
                }
            )
        ));
    this.edgePaths.putAll(edgePaths);
  }
 */

  protected Optional<NESTEdgeObject> findReverseDataflowEdge(NESTEdgeObject edge) {
    return findReverseEdge(edge, NESTEdgeObject::isNESTDataflowEdge);
  }

  private Optional<NESTEdgeObject> findReverseEdge(
      NESTEdgeObject edge, Predicate<? super NESTEdgeObject> filter) {
    return edge.getPre() == null
        ? Optional.empty()
        : edge.getPre().getIngoingEdges(filter).stream()
            .filter(inEdge -> inEdge.getPre() == edge.getPost())
            .findAny();
  }

  /**
   * The layout only works properly when certain constraints (concerning the controlflow) on the
   * graph are met. This method is used to check the NESTWorkflow for these constraints to determine
   * whether an alternative layout should be run.
   *
   * @return whether the needed constraints to apply NESTWorkflowLayout are met
   */
  public boolean isApplicable() {
    if (this.nestWorkflow.isNESTWorkflow()) {
      var validator = new NESTWorkflowValidatorImpl((NESTWorkflowObject) this.nestWorkflow);
      return validator.hasCorrectControlflowEdges() && validator.checkControlflowBlocks();

    } else if (this.nestWorkflow.isNESTSequentialWorkflow()) {
      return new NESTSequentialWorkflowValidatorImpl(this.nestWorkflow).isValidSequentialWorkflow();
    }

    return false;

  }

  /**
   * @param edge A NESTEdgeObject
   * @return dimension of the label of the given NESTEdgeObject
   */
  public abstract Dimension getEdgeLabelSize(NESTEdgeObject edge);

  /**
   * @param node A NESTNodeObject
   * @return dimension of the visual representation of the given NESTNodeObject
   */
  public abstract Dimension getNodeSize(NESTNodeObject node);

  private int getMaxNodeHeight(Collection<? extends NESTNodeObject> nodes) {
    return nodes.stream()
        .map(node -> this.getNodeSize(node).getHeight())
        .max(Double::compareTo)
        .orElse(0.0)
        .intValue();
  }

  private void layoutNodesY(Branch branch) {
    Map<NESTNodeObject, Integer> nodeVerticalPositions = calculateNodeVerticalPositions(branch);

    // ensure that all root branches start on the same height whether they have a top data node
    // layer or not:
    int minPosition =
        nodeVerticalPositions.entrySet().stream()
            .filter(entry -> entry.getKey().isNESTDataNode() || entry.getKey().isNESTSequenceNode())
            .map(Map.Entry::getValue)
            .min(Integer::compareTo)
            .orElse(0);
    if (layoutConfig.isPlaceDataNodesVerticallyNearTaskNodes()
        && layoutConfig.isAlsoPlaceDataNodesAboveTaskNodes()
        && nodeVerticalPositions.entrySet().stream()
        .noneMatch(
            entry -> entry.getValue() == minPosition && entry.getKey().isNESTDataNode())) {
      nodeVerticalPositions =
          nodeVerticalPositions.entrySet().stream()
              .collect(
                  Collectors.toMap(
                      Map.Entry::getKey,
                      e ->
                          e.getValue()
                              + layoutConfig.getNodeHeight()
                              + layoutConfig.getTaskNodeToDataNodeVerticalSpacing())); // shift root branches without
      // top data node layer
      // downwards by 1
    }
    this.nodeYPositions.putAll(nodeVerticalPositions);
  }

  private Map<NESTNodeObject, Integer> calculateNodeVerticalPositions(Branch branch) {
    Map<NESTNodeObject, Integer> positions = new HashMap<>();
    Map<NESTNodeObject, Integer> sequenceNodePositions = branch.getNodeVerticalPositions();
    Map<NESTNodeObject, Integer> workflowAndSubworkflowLevels =
        getWorkflowAndSubworkflowLevels(this.nestWorkflow.getWorkflowNode());

    int minSequenceNodeY =
        sequenceNodePositions.values().stream().min(Integer::compareTo).orElse(0);
    int maxWorkflowAndSubworkflowLevels =
        workflowAndSubworkflowLevels.values().stream().max(Integer::compareTo).orElse(0);
    Map<NESTNodeObject, Integer> workflowAndSubworkflowPositions =
        workflowAndSubworkflowLevels.entrySet().stream()
            .collect(
                Collectors.toMap(
                    Map.Entry::getKey,
                    e ->
                        minSequenceNodeY
                            - ((maxWorkflowAndSubworkflowLevels + 2)
                            * (layoutConfig.getNodeHeight()
                            + layoutConfig.getNodeVerticalSpacing()))
                            + e.getValue()
                            * (layoutConfig.getNodeHeight()
                            + layoutConfig.getNodeVerticalSpacing()))); // shift (sub)workflow nodes above
    // sequence nodes (and leave empty
    // space)

    positions.putAll(sequenceNodePositions);
    positions.putAll(workflowAndSubworkflowPositions);
    if (!layoutConfig.isPlaceDataNodesVerticallyNearTaskNodes()) {
      int maxNodePosition = positions.values().stream().max(Integer::compareTo).orElse(0);
      for (NESTNodeObject dataNode : this.nestWorkflow.getDataNodes()) {
        positions.put(
            dataNode,
            maxNodePosition
                + layoutConfig.getTaskNodeToDataNodeVerticalSpacing()); // put data nodes below everything else
      }
    }
    int minNodePosition = positions.values().stream().min(Integer::compareTo).orElse(0);
    positions =
        positions.entrySet().stream()
            .collect(
                Collectors.toMap(
                    Map.Entry::getKey,
                    e -> e.getValue() - minNodePosition + 1)); // shift positions to positive values

    return positions;
  }

  private Map<NESTNodeObject, Integer> getWorkflowAndSubworkflowLevels(
      NESTNodeObject node, int level) {
    Map<NESTNodeObject, Integer> levels = new HashMap<>();
    if (node == null) {
      return levels;
    } // NESTWorkflow is invalid due to missing Workflow node, but layout should still be applicable
    if (node.isNESTWorkflowNode() || node.isNESTSubWorkflowNode()) {
      levels.put(node, level);
      node.getIngoingEdges(NESTEdgeObject::isNESTPartOfEdge)
          .forEach(
              partOfEdge ->
                  levels.putAll(getWorkflowAndSubworkflowLevels(partOfEdge.getPre(), level + 1)));
    }
    return levels;
  }

  private Map<NESTNodeObject, Integer> getWorkflowAndSubworkflowLevels(NESTNodeObject node) {
    return getWorkflowAndSubworkflowLevels(node, 0);
  }

  private void layoutDataNodesX() {
    IntervalOverlapResolver overlapResolver = new IntervalOverlapResolver(10);
    Set<NESTDataNodeObject> dataNodes = this.nestWorkflow.getDataNodes();
    Map<NESTDataNodeObject, Integer> dataNodesCenterXPositions =
        this.getDataNodesCenterXPositions(dataNodes);
    dataNodes.forEach(
        node -> {
          int intervalStart =
              dataNodesCenterXPositions.get(node) - (int) (this.getNodeSize(node).getWidth() / 2);
          overlapResolver.addInterval(
              intervalStart, intervalStart + (int) this.getNodeSize(node).getWidth(), node.getId());
        });

    List<IntervalOverlapResolver.Interval> nonOverlappingIntervals = overlapResolver.resolve();
    nonOverlappingIntervals.forEach(
        interval ->
            nodeXPositions.put(nestWorkflow.getGraphNode(interval.getId()), interval.getStart()));
  }

  private Map<NESTDataNodeObject, Integer> getDataNodesCenterXPositions(
      Collection<NESTDataNodeObject> dataNodes) {
    return dataNodes.stream()
        .collect(
            Collectors.toMap(
                Function.identity(),
                dataNode -> {
                  List<Double> inNeighboursXPositionsCentre =
                      dataNode.getIngoingEdges(NESTEdgeObject::isNESTDataflowEdge).stream()
                          .map(
                              edge -> {
                                NESTNodeObject preNode = edge.getPre();
                                return nodeXPositions.get(preNode)
                                    + this.getNodeSize(preNode).getWidth() / 2;
                              })
                          .collect(Collectors.toList());
                  List<Double> outNeighboursXPositionsCentre =
                      dataNode.getOutgoingEdges(NESTEdgeObject::isNESTDataflowEdge).stream()
                          .filter(edge -> edge.getPost() != null)
                          .map(
                              edge -> {
                                NESTNodeObject postNode = edge.getPost();
                                return nodeXPositions.get(postNode)
                                    + this.getNodeSize(postNode).getWidth() / 2;
                              })
                          .collect(Collectors.toList());
                  List<Double> neighboursXPositions =
                      new ArrayList<>() {
                        {
                          addAll(inNeighboursXPositionsCentre);
                          addAll(outNeighboursXPositionsCentre);
                        }
                      };
                  return (int)
                      neighboursXPositions.stream().mapToDouble(val -> val).average().orElse(0);
                }));
  }

  private Map<NESTTaskNodeObject, Integer> getTaskNodesCenterXPositions() {
    return this.nestWorkflow.getTaskNodes().stream()
        .filter(taskNode -> nodeXPositions.get(taskNode) != null)
        .collect(
            Collectors.toMap(
                Function.identity(),
                taskNode ->
                    (int)
                        (nodeXPositions.get(taskNode)
                            + this.getNodeSize(taskNode).getWidth() / 2)));
  }

  private void layoutWorkflowNodesX() {
    Set<NESTNodeObject> workflowNodes = new HashSet<>(this.nestWorkflow.getSubWorkflowNodes());
    workflowNodes.add(this.nestWorkflow.getWorkflowNode());
    IntervalOverlapResolver overlapResolver = new IntervalOverlapResolver(10);

    workflowNodes.stream()
        .filter(Objects::nonNull)
        .forEach(
            node -> {
              List<Integer> inNeighboursXPositions =
                  node.getIngoingEdges(NESTEdgeObject::isNESTPartOfEdge).stream()
                      .filter(edge -> edge.getPre() != null)
                      .map(edge -> nodeXPositions.getOrDefault(edge.getPre(), 0))
                      .collect(Collectors.toList());
              int averageXPosition =
                  (int) inNeighboursXPositions.stream().mapToDouble(val -> val).average().orElse(0);
              overlapResolver.addInterval(
                  averageXPosition, averageXPosition + this.getNodeSize(node).width, node.getId());
            });

    overlapResolver
        .resolve()
        .forEach(
            interval ->
                nodeXPositions
                    .put(nestWorkflow.getGraphNode(interval.getId()), interval.getStart()));
  }

  private void resolveNodeOverlapsX() {
    Collection<List<NESTNodeObject>> nodeLayers =
        this.nestWorkflow.getGraphNodes().stream()
            .collect(Collectors.groupingBy(node -> nodeYPositions.getOrDefault(node, 0)))
            .values();

    nodeLayers.forEach(
        nodes -> {
          IntervalOverlapResolver overlapResolver = new IntervalOverlapResolver(5);
          nodes.forEach(
              node -> {
                int intervalStart = nodeXPositions.getOrDefault(node, 0);
                overlapResolver.addInterval(
                    intervalStart, intervalStart + this.getNodeSize(node).width, node.getId());
              });

          overlapResolver
              .resolve()
              .forEach(
                  interval ->
                      nodeXPositions.put(
                          nestWorkflow.getGraphNode(interval.getId()), interval.getStart()));
        });
  }

  private void shiftGraphToPositivePositions() {
    int minX = nodeXPositions.values().stream().min(Integer::compareTo).orElse(0);
    int shiftDistanceX = layoutConfig.getGraphLeftMargin() - minX;
    int minY = nodeYPositions.values().stream().min(Integer::compareTo).orElse(0);
    int minEdgePathY =
        edgePaths.values().stream()
            .flatMap(Collection::stream)
            .map(java.awt.Point::getY)
            .min(Double::compareTo)
            .orElse(0.0)
            .intValue();
    minY = Math.min(minY, minEdgePathY);
    int shiftDistanceY = minY < layoutConfig.getGraphTopMargin() ? layoutConfig.getGraphTopMargin()
        - minY : 0;

    this.nodeXPositions =
        this.nodeXPositions.entrySet().stream()
            .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue() + shiftDistanceX));
    this.nodeYPositions =
        this.nodeYPositions.entrySet().stream()
            .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue() + shiftDistanceY));
    this.edgePaths.values().stream()
        .flatMap(Collection::stream)
        .forEach(
            point -> {
              point.setLocation(point.x + shiftDistanceX, point.y + shiftDistanceY);
            });
  }

  public Map<NESTNodeObject, Integer> getNodeXPositions() {
    return this.nodeXPositions;
  }

  public int getNodeXPosition(NESTNodeObject node) {
    return this.nodeXPositions.get(node);
  }

  public Map<NESTNodeObject, Integer> getNodeYPositions() {
    return this.nodeYPositions;
  }

  public int getNodeYPosition(NESTNodeObject node) {
    return this.nodeYPositions.get(node);
  }

  public Map<NESTEdgeObject, List<java.awt.Point>> getEdgePaths() {
    return this.edgePaths;
  }

  public List<java.awt.Point> getEdgePath(NESTEdgeObject edge) {
    return this.edgePaths.get(edge);
  }

  public LayoutConfig getLayoutConfig() {
    return layoutConfig;
  }

  public void setLayoutConfig(LayoutConfig layoutConfig) {
    this.layoutConfig = layoutConfig;
    layoutConfig.setNestWorkflowLayout(this);
    this.execute();
  }

  public boolean isExecuteOnEdgeInsertion() {
    return executeOnEdgeInsertion;
  }

  public void setExecuteOnEdgeInsertion(boolean executeOnEdgeInsertion) {
    this.executeOnEdgeInsertion = executeOnEdgeInsertion;
  }

  //    interface

  public void setNestWorkflow(NESTAbstractWorkflowObject nestWorkflow) {
    this.nestWorkflow = nestWorkflow;
  }

  private int getCenterY(NESTNodeObject node) {
    return (int) (nodeYPositions.get(node) + getNodeSize(node).getHeight() / 2);
  }

  private int getCenterX(NESTNodeObject node) {
    return (int) (nodeXPositions.get(node) + getNodeSize(node).getWidth() / 2);
  }

  /**
   * interface to group sequences, splits and branches
   */
  interface Controlflow {

    Map<NESTNodeObject, Integer> getNodeVerticalPositions();

    int getHeight();

    int getTopHeight();

    int getBottomHeight();

    Map<NESTSequenceNodeObject, Sequence> getSequenceNodesToSequencesMapping();

    /**
     * Assigns each data node to either the top or bottom layer of its branch
     */
    void distributeDataNodes();

    void calculateSequenceNodeHorizontalPositions();

    /**
     * Returns the x positions of all sequence and data nodes contained in this controlflow. Note:
     * calculateSequenceNodesXPositions needs to be executed prior for data node position
     * calculation to work
     *
     * @param startX The x position the laying out should start
     * @return The x positions for all sequence and data nodes contained in this Controlflow
     */
    Map<NESTNodeObject, Integer> getNodesXPositions(int startX);
  }

  static class IntervalOverlapResolver extends HashSet<IntervalOverlapResolver.Interval> {

    private int buffer;

    public IntervalOverlapResolver(int buffer) {
      this.buffer = buffer;
    }

    public void addInterval(int start, int end, String id) {
      this.add(new Interval(start, end, id));
    }

    public List<Interval> resolve() {
      SortedSet<IntervalContainer> intervalContainers = new TreeSet<>();
      for (var interval : this) {
        intervalContainers.add(new IntervalContainer(interval));
      }

      boolean modified = true;
      while (modified) {
        modified = false;
        IntervalContainer last = null;
        for (IntervalContainer next : intervalContainers) {
          if (last != null) {
            if (last.isOverlapping(next, buffer)) {
              // remove containers
              intervalContainers.remove(last);
              intervalContainers.remove(next);

              // determine shift directions
              var leftShiftIntervalContainer =
                  last.getShiftDistanceWeight() <= next.getShiftDistanceWeight() ? last : next;
              var rightShiftIntervalContainer =
                  last.getShiftDistanceWeight() > next.getShiftDistanceWeight() ? last : next;

              leftShiftIntervalContainer.resolveOverlap(rightShiftIntervalContainer, buffer);
              // merge containers and re-add to maintain order
              leftShiftIntervalContainer.addAll(rightShiftIntervalContainer);
              intervalContainers.add(leftShiftIntervalContainer);

              modified = true;
              break;
            }
          }
          last = next;
        }
      }

      List<Interval> result = new ArrayList<>(this.size());
      for (var intervalContainer : intervalContainers) {
        result.addAll(intervalContainer);
      }

      return result;
    }

    static class Interval implements Comparable<Interval> {

      private int start;
      private int end;
      private String id;

      public Interval(int start, int end, String id) {
        this.start = start;
        this.end = end;
        this.id = id;
      }

      @Override
      public int compareTo(Interval other) {
        return Comparator.comparing(Interval::getStart)
            .thenComparing(Interval::getId)
            .compare(this, other);
      }

      public int getShiftDistanceWeight() {
        return 0;
      }

      public int getStart() {
        return start;
      }

      public void setStart(int start) {
        this.start = start;
      }

      public int getEnd() {
        return end;
      }

      public void setEnd(int end) {
        this.end = end;
      }

      public String getId() {
        return id;
      }
    }

    class IntervalContainer extends TreeSet<Interval> implements Comparable<IntervalContainer> {

      public IntervalContainer(Interval initialElement) {
        this.add(initialElement);
      }

      /**
       * 'this' is shifted to the left, 'other' is shifted to the right
       *
       * @param other
       * @param buffer
       */
      public void resolveOverlap(IntervalContainer other, int buffer) {
        int distanceToResolve = this.getShiftDistance(other) + buffer;
        // the container with fewer intervals has to move more than the container with more
        // intervals:
        int otherShiftDistance =
            (int)
                Math.ceil(
                    (((double) this.size()) / (this.size() + other.size()) * distanceToResolve));
        int meShiftDistance =
            (int)
                Math.ceil(
                    (((double) other.size()) / (this.size() + other.size()) * distanceToResolve));
        this.shift(-meShiftDistance);
        other.shift(otherShiftDistance);
      }

      private void shift(int distance) {
        this.forEach(
            interval -> {
              interval.setStart(interval.getStart() + distance);
              interval.setEnd(interval.getEnd() + distance);
            });
      }

      private boolean isOverlapping(IntervalContainer other, int buffer) {
        List<IntervalContainer> intervals = Arrays.asList(this, other);
        Collections.sort(intervals);
        return (intervals.get(0).getEnd() + buffer) > intervals.get(1).getStart();
      }

      /**
       * @param other
       * @return the distance "other" has to be shifted to the right to resolve the overlap
       */
      private int getShiftDistance(IntervalContainer other) {
        List<IntervalContainer> intervals = Arrays.asList(this, other);
        Collections.sort(intervals);
        return intervals.get(0).getEnd() - intervals.get(1).getStart();
      }

      public int getShiftDistanceWeight() {
        return this.stream()
            .map(Interval::getShiftDistanceWeight)
            .mapToInt(Integer::intValue)
            .sum();
      }

      public int getStart() {
        return this.first().getStart();
      }

      public int getEnd() {
        return this.last().getEnd();
      }

      @Override
      public int compareTo(IntervalContainer other) {
        return Comparator.comparing(IntervalContainer::getStart)
            .thenComparing(IntervalContainer::first)
            .compare(this, other);
      }
    }
  }

  class DataNodeDistribution {

    Map<NESTDataNodeObject, Integer> xPositions = new HashMap<>();
    Set<NESTDataNodeObject> topLayer = new HashSet<>();
    Set<NESTDataNodeObject> bottomLayer = new HashSet<>();
    Map<NESTNodeObject, Integer> nodePositions = new HashMap<>();

    //        public DataNodeDistribution(Set<NESTDataNodeObject> dataNodes) {
    //            // dummy: // TODO: implement distribution which minimizes crossings of dataflow
    // edges and then overlap distance of data nodes, exhaustive search would produce 2^n solutions
    // which have to be rated
    //            int i = 0;
    //            for(NESTDataNodeObject dataNode : dataNodes)  {
    //                if(i % 2 == 0) {
    //                    bottomLayer.add(dataNode);
    //                } else {
    //                    topLayer.add(dataNode);
    //                }
    //                i++;
    //            }
    //        }

    public DataNodeDistribution(Set<NESTDataNodeObject> dataNodes) {
      if (dataNodes.size() > 2) {
        this.distribute(dataNodes);
      } else {
        bottomLayer.addAll(dataNodes);
      }
    }

    private void distribute(Set<NESTDataNodeObject> dataNodes) {
      nodePositions.putAll(getDataNodesCenterXPositions(dataNodes));
      nodePositions.putAll(getTaskNodesCenterXPositions());

      Map<List<List<NESTDataNodeObject>>, Integer> solutions = new HashMap<>();
      for (List<List<NESTDataNodeObject>> solution :
          new PartitionIterable<>(new LinkedList<>(dataNodes), 2)) {
        solutions.put(solution, this.edgeCrossingsInSolution(solution));
      }
      int minCrossings = solutions.values().stream().min(Integer::compareTo).get();
      List<List<NESTDataNodeObject>> solution =
          solutions.entrySet().stream()
              .filter(entry -> entry.getValue() == minCrossings)
              .findFirst()
              .get()
              .getKey();

      if (solution.get(0).size() < solution.get(1).size()) {
        topLayer.addAll(solution.get(0));
        bottomLayer.addAll(solution.get(1));
      } else {
        topLayer.addAll(solution.get(1));
        bottomLayer.addAll(solution.get(0));
      }
    }

    private int edgeCrossingsInSolution(List<List<NESTDataNodeObject>> solution) {
      return solution.stream().map(this::getEdgeCrossings).mapToInt(Integer::intValue).sum();
    }

    private int getEdgeCrossings(List<NESTDataNodeObject> dataNodes) {
      Set<NESTEdgeObject> dataflowEdges =
          dataNodes.stream()
              .map(
                  dataNode ->
                      dataNode.getEdges(NESTEdgeObject::isNESTDataflowEdge).stream()
                          .filter(edge -> edge.getPost().isNESTTaskNode())
                          .collect(
                              Collectors
                                  .toSet())) // only consider valid dataflow edges which end in a
              // task node
              .flatMap(Collection::stream)
              .collect(Collectors.toSet());
      return dataflowEdges.stream()
          .map(
              currentEdge ->
                  dataflowEdges.stream()
                      .filter(otherEdge -> isCrossing(currentEdge, otherEdge))
                      .count())
          .mapToInt(Long::intValue)
          .sum();
    }

    private boolean isCrossing(NESTEdgeObject edgeA, NESTEdgeObject edgeB) {
      Set<NESTNodeObject> endpoints =
          new HashSet<>(
              Arrays.asList(edgeA.getPre(), edgeA.getPost(), edgeB.getPre(), edgeB.getPost()));
      if (endpoints.size() < 4) { // at least one common endpoint
        return false;
      }
      NESTNodeObject edgeATaskNode =
          edgeA.getPre().isNESTTaskNode() ? edgeA.getPre() : edgeA.getPost();
      NESTNodeObject edgeADataNode =
          edgeA.getPre().isNESTDataNode() ? edgeA.getPre() : edgeA.getPost();
      NESTNodeObject edgeBTaskNode =
          edgeB.getPre().isNESTTaskNode() ? edgeB.getPre() : edgeB.getPost();
      NESTNodeObject edgeBDataNode =
          edgeB.getPre().isNESTDataNode() ? edgeB.getPre() : edgeB.getPost();

      return ObjectUtils.allNotNull(edgeATaskNode, edgeADataNode, edgeBTaskNode, edgeBDataNode)
          && (((nodePositions.get(edgeATaskNode) < nodePositions.get(edgeBTaskNode))
          && (nodePositions.get(edgeADataNode) > nodePositions.get(edgeBDataNode)))
          || ((nodePositions.get(edgeATaskNode) > nodePositions.get(edgeBTaskNode))
          && (nodePositions.get(edgeADataNode) < nodePositions.get(edgeBDataNode))));
    }

    public Map<NESTDataNodeObject, Integer> getxPositions() {
      return xPositions;
    }

    public Set<NESTDataNodeObject> getTopLayer() {
      return topLayer;
    }

    public Set<NESTDataNodeObject> getBottomLayer() {
      return bottomLayer;
    }
  }

  class Split extends LinkedList<Branch> implements Controlflow {

    public Split(List<Branch> branches) {
      super(branches);
    }

    @Override
    public Map<NESTNodeObject, Integer> getNodeVerticalPositions() {
      Map<NESTNodeObject, Integer> verticalPositions = new HashMap<>();
      if (this.size() > 0) {
        Branch topBranch = this.get(0);
        Map<NESTNodeObject, Integer> topBranchVerticalPositions =
            topBranch.getNodeVerticalPositions();
        verticalPositions.putAll(
            topBranchVerticalPositions.entrySet().stream()
                .collect(
                    Collectors.toMap(
                        Map.Entry::getKey,
                        e ->
                            e.getValue()
                                - (topBranch.getBottomHeight()
                                + this.getControlflowNodeHeight() / 2
                                + NESTWorkflowLayout.this.layoutConfig.getSequenceNodeVerticalSpacing()))));
      }
      if (this.size() > 1) {
        Branch bottomBranch = this.get(1);
        Map<NESTNodeObject, Integer> bottomBranchVerticalPositions =
            bottomBranch.getNodeVerticalPositions();
        verticalPositions.putAll(
            bottomBranchVerticalPositions.entrySet().stream()
                .collect(
                    Collectors.toMap(
                        Map.Entry::getKey,
                        e ->
                            e.getValue()
                                + (bottomBranch.getTopHeight()
                                + this.getControlflowNodeHeight() / 2
                                + NESTWorkflowLayout.this.layoutConfig.getSequenceNodeVerticalSpacing()))));
      }
      return verticalPositions;
    }

    private int getControlflowNodeHeight() {
      return (int)
          NESTWorkflowLayout.this.getNodeSize(this.get(0).getIngoingEdge().getPre()).getHeight();
    }

    @Override
    public int getHeight() {
      int topBranchHeight =
          this.size() > 0 ? this.get(0).getHeight()
              : NESTWorkflowLayout.this.layoutConfig.getNodeHeight();
      int bottomBranchHeight =
          this.size() > 1 ? this.get(1).getHeight()
              : NESTWorkflowLayout.this.layoutConfig.getNodeHeight();
      return topBranchHeight
          + bottomBranchHeight
          + this.getControlflowNodeHeight()
          + 2 * NESTWorkflowLayout.this.layoutConfig.getSequenceNodeVerticalSpacing();
    }

    @Override
    public int getTopHeight() {
      int topBranchHeight =
          this.size() > 0 ? this.get(0).getHeight()
              : NESTWorkflowLayout.this.layoutConfig.getNodeHeight();
      return topBranchHeight
          + this.getControlflowNodeHeight() / 2
          + NESTWorkflowLayout.this.layoutConfig.getSequenceNodeVerticalSpacing();
    }

    @Override
    public int getBottomHeight() {
      int bottomBranchHeight =
          this.size() > 1 ? this.get(1).getHeight()
              : NESTWorkflowLayout.this.layoutConfig.getNodeHeight();
      return bottomBranchHeight
          + this.getControlflowNodeHeight() / 2
          + NESTWorkflowLayout.this.layoutConfig.getSequenceNodeVerticalSpacing();
    }

    @Override
    public Map<NESTSequenceNodeObject, Sequence> getSequenceNodesToSequencesMapping() {
      Map<NESTSequenceNodeObject, Sequence> result = new HashMap<>();
      this.forEach(branch -> result.putAll(branch.getSequenceNodesToSequencesMapping()));
      return result;
    }

    @Override
    public void distributeDataNodes() {
      this.forEach(Branch::distributeDataNodes);
    }

    @Override
    public void calculateSequenceNodeHorizontalPositions() {
      this.forEach(Branch::calculateSequenceNodeHorizontalPositions);
    }

    @Override
    public Map<NESTNodeObject, Integer> getNodesXPositions(int startX) {
      Map<NESTNodeObject, Integer> xPositions = new HashMap<>();
      this.forEach(branch -> xPositions.putAll(branch.getNodesXPositions(startX)));
      return xPositions;
    }

    public Map<NESTEdgeObject, List<java.awt.Point>> getControlflowEdgePaths() {
      Map<NESTEdgeObject, List<java.awt.Point>> edgePaths = new HashMap<>();
      this.forEach(branch -> edgePaths.putAll(branch.getControlflowEdgePaths()));

      this.forEach(
          branch -> {
            List<java.awt.Point> ingoingEdgePath = new LinkedList<>();
            NESTEdgeObject ingoingEdge = branch.getIngoingEdge();
            NESTEdgeObject outgoingEdge = branch.getOutgoingEdge();
            NESTNodeObject controlflowStartNode = ingoingEdge.getPre();
            NESTNodeObject controlflowEndNode =
                outgoingEdge != null ? outgoingEdge.getPost() : null;

            if (ingoingEdge == outgoingEdge) { // empty branch
              Branch otherBranch =
                  this.get(0) == branch ? this.size() >= 2 ? this.get(1) : null : this.get(0);
              int sign = otherBranch == null ? -1 : Integer.signum(branch.compareTo(otherBranch));
              int bendVerticalPosition =
                  (int)
                      (getCenterY(controlflowStartNode)
                          + sign
                          * (getNodeSize(controlflowStartNode).getHeight() / 2
                          + layoutConfig.getSequenceNodeVerticalSpacing()
                          + layoutConfig.getNodeHeight() / 2));
              ingoingEdgePath.add(
                  new java.awt.Point(getCenterX(controlflowStartNode), bendVerticalPosition));
              ingoingEdgePath.add(
                  new java.awt.Point(getCenterX(controlflowEndNode), bendVerticalPosition));
            } else {
              int bendVerticalPosition = getCenterY(ingoingEdge.getPost());
              ingoingEdgePath.add(
                  new java.awt.Point(getCenterX(controlflowStartNode), bendVerticalPosition));

              if (controlflowEndNode != null) {
                edgePaths.put(
                    outgoingEdge,
                    new LinkedList<>(
                        Collections.singletonList(
                            new java.awt.Point(
                                getCenterX(controlflowEndNode), bendVerticalPosition))));
              }
            }
            edgePaths.put(ingoingEdge, ingoingEdgePath);
          });
      return edgePaths;
    }
  }

  class LoopSplit extends Split {

    public LoopSplit(List<Branch> branches) {
      super(branches);
    }

    @Override
    public Map<NESTNodeObject, Integer> getNodeVerticalPositions() {
      return this.size() <= 0 ? new HashMap<>() : this.get(0).getNodeVerticalPositions();
    }

    @Override
    public int getHeight() {
      return this.get(0).getHeight() + layoutConfig.getSequenceNodeVerticalSpacing()
          + layoutConfig.getNodeHeight() / 2;
    }

    @Override
    public int getTopHeight() {
      return this.get(0).getTopHeight() + layoutConfig.getSequenceNodeVerticalSpacing()
          + layoutConfig.getNodeHeight() / 2;
    }

    @Override
    public int getBottomHeight() {
      return this.get(0).getBottomHeight();
    }

    @Override
    public Map<NESTEdgeObject, List<java.awt.Point>> getControlflowEdgePaths() {
      Map<NESTEdgeObject, List<java.awt.Point>> edgePaths = new HashMap<>();
      this.forEach(branch -> edgePaths.putAll(branch.getControlflowEdgePaths()));

      if (this.size() > 0) {
        Branch branch = this.get(0);
        NESTEdgeObject loopReturnEdge =
            branch.getOutgoingEdge().getPost().getOutgoingEdges().stream()
                .filter(Utils::isEdgeLoopReturnEdge)
                .findAny()
                .get();
        NESTNodeObject loopEndNode = loopReturnEdge.getPre();
        NESTNodeObject loopStartNode = loopReturnEdge.getPost();
        int loopEndNodeCenterY =
            nodeYPositions.get(loopEndNode) - getNodeSize(loopEndNode).height / 2;
        int bendVerticalPosition =
            loopEndNodeCenterY
                - branch.getTopHeight()
                - layoutConfig.getSequenceNodeVerticalSpacing()
                - layoutConfig.getNodeHeight() / 2;
        edgePaths.put(
            loopReturnEdge,
            new LinkedList<>(
                Arrays.asList(
                    new java.awt.Point(
                        nodeXPositions.get(loopEndNode) + getNodeSize(loopEndNode).width / 2,
                        bendVerticalPosition),
                    new java.awt.Point(
                        nodeXPositions.get(loopStartNode) + getNodeSize(loopStartNode).width / 2,
                        bendVerticalPosition))));
      }
      return edgePaths;
    }
  }

  /**
   * Controlflow Branch. Can consist of multiple sequences and child branches according to its
   * controlflow splits. Contains the controlflow consisting of sequences and child branches in an
   * ordered manner for easier traversal of the controlflow.
   */
  class Branch implements Comparable<Branch>, Controlflow {

    private List<Controlflow> controlflow = new LinkedList<>();
    private NESTEdgeObject ingoingEdge; // for creating "empty" branches without nodes
    private int maximumX; // furthest spot on the right this branch consumes on the x axis

    public Branch(NESTSequenceNodeObject firstNode) {
      this.initIngoingEdge(firstNode);
      this.initControlflow(firstNode);
    }

    private Branch(NESTEdgeObject ingoingEdge) { // for creating "empty" branches without nodes
      this.ingoingEdge = ingoingEdge;
    }

    private void initControlflow(NESTSequenceNodeObject currentNode) {
      Sequence currentSequence = new Sequence(currentNode);
      controlflow.add(currentSequence);

      NESTSequenceNodeObject lastNode = currentSequence.getLastNode();
      if (lastNode.isNESTControlflowNode()
          && ((NESTControlflowNodeObject) lastNode).isStartControlflowNode()) { // split
        NESTControlflowNodeObject startControlflowNode = (NESTControlflowNodeObject) lastNode;
        NESTControlflowNodeObject endControlflowNode =
            startControlflowNode.getMatchingBlockControlflowNode();
        List<Branch> branches =
            startControlflowNode.getOutgoingEdges(NESTEdgeObject::isNESTControlflowEdge).stream()
                .map(
                    outEdge ->
                        outEdge.getPost() == endControlflowNode
                            ? new Branch(outEdge) // add edge-only branch
                            : new Branch((NESTSequenceNodeObject) outEdge.getPost()))
                .sorted() // ensure determinism of layout, split branch order has to be always be
                // the same
                .collect(Collectors.toList());
        controlflow.add(
            startControlflowNode.isLoopNode()
                ? new LoopSplit(branches)
                : new Split(
                    branches)); // make distinction here so the custom layout behavior for loop
        // blocks can be implemented in own class

        if (endControlflowNode != null) {
          this.initControlflow(endControlflowNode);
        }
      }
    }

    private List<Sequence> getSequences() {
      return controlflow.stream()
          .filter(Sequence.class::isInstance)
          .map(Sequence.class::cast)
          .collect(Collectors.toList());
    }

    private List<Split> getSplits() {
      return controlflow.stream()
          .filter(Split.class::isInstance)
          .map(Split.class::cast)
          .collect(Collectors.toList());
    }

    public Map<NESTSequenceNodeObject, Sequence> getSequenceNodesToSequencesMapping() {
      Map<NESTSequenceNodeObject, Sequence> result = new HashMap<>();
      controlflow.forEach(
          controlflow -> result.putAll(controlflow.getSequenceNodesToSequencesMapping()));
      return result;
    }

    @Override
    public void distributeDataNodes() {
      this.controlflow.forEach(Controlflow::distributeDataNodes);
    }

    @Override
    public void calculateSequenceNodeHorizontalPositions() {
      controlflow.forEach(Controlflow::calculateSequenceNodeHorizontalPositions);
    }

    @Override
    public Map<NESTNodeObject, Integer> getNodesXPositions(int startX) {
      currentBranchXOffset = startX;
      Map<NESTNodeObject, Integer> xPositions = new HashMap<>();
      for (Controlflow controlflow : controlflow) {
        Map<NESTNodeObject, Integer> controlflowXPositions = controlflow.getNodesXPositions(startX);
        if (controlflowXPositions.size()
            > 0) { // branches can be empty and thus have no x positions (e.g. in an empty
          // controlflow block)
          this.maximumX =
              controlflowXPositions.entrySet().stream()
                  .map(entry -> entry.getValue() + (int) getNodeSize(entry.getKey()).getWidth())
                  .max(Integer::compareTo)
                  .get();
          startX = this.maximumX;
          xPositions.putAll(controlflowXPositions);
        }
      }
      return xPositions;
    }

    private void initIngoingEdge(NESTSequenceNodeObject firstNode) {
      this.ingoingEdge =
          firstNode.getIngoingEdges().stream()
              .filter(edge -> edge.isNESTControlflowEdge() && !Utils.isEdgeLoopReturnEdge(edge))
              .findAny()
              .orElse(null);
    }

    public void assignDataNodesToSequences(Set<NESTDataNodeObject> dataNodes) {
      Map<NESTSequenceNodeObject, Sequence> sequenceNodesToSequencesMapping =
          this.getSequenceNodesToSequencesMapping();
      dataNodes.forEach(
          dataNode -> { // assign the data node to the sequence it has the most connections to
            Map<Sequence, Integer> associationCount = new HashMap<>();
            dataNode
                .getConnectedTasks()
                .forEach(
                    connectedTask -> {
                      Sequence taskSequence = sequenceNodesToSequencesMapping.get(connectedTask);
                      associationCount.put(
                          taskSequence, associationCount.getOrDefault(taskSequence, 0) + 1);
                    });
            int maxAssociationCount =
                associationCount.values().stream().max(Integer::compareTo).orElse(0);
            List<Sequence> fittingSequences =
                associationCount.entrySet().stream()
                    .filter(
                        entry ->
                            entry.getKey()
                                != null // happens when there are multiple start nodes present
                                // and thus the sequenceNodesToSequencesMapping is
                                // incomplete
                                && entry.getValue() == maxAssociationCount)
                    .map(Map.Entry::getKey)
                    .sorted()
                    .collect(Collectors.toList());
            if (fittingSequences.size() > 0) {
              fittingSequences.get(0).getBottomDataNodes().add(dataNode);
            }
          });
    }

    public NESTEdgeObject getIngoingEdge() {
      return ingoingEdge;
    }

    public NESTEdgeObject getOutgoingEdge() {
      if (controlflow.size() == 0) {
        return ingoingEdge;
      }
      Controlflow lastCf = this.controlflow.get(this.controlflow.size() - 1);
      if (lastCf instanceof Split) {
        return null;
      }
      return ((Sequence) lastCf)
          .getLastNode().getOutgoingEdges(NESTEdgeObject::isNESTControlflowEdge).stream()
          .filter(edge -> !Utils.isEdgeLoopReturnEdge(edge))
          .findAny()
          .orElse(null);
    }

    public int getMaximumX() {
      return maximumX;
    }

    @Override
    public Map<NESTNodeObject, Integer> getNodeVerticalPositions() {
      Map<NESTNodeObject, Integer> verticalPositions = new HashMap<>();
      this.controlflow.forEach(cf -> verticalPositions.putAll(cf.getNodeVerticalPositions()));
      return verticalPositions;
    }

    @Override
    public int getHeight() {
      return this.controlflow.stream()
          .map(Controlflow::getHeight)
          .max(Integer::compareTo)
          .orElse(layoutConfig.getNodeHeight());
    }

    @Override
    public int getTopHeight() {
      return this.controlflow.stream()
          .map(Controlflow::getTopHeight)
          .max(Integer::compareTo)
          .orElse(layoutConfig.getNodeHeight() / 2);
    }

    @Override
    public int getBottomHeight() {
      return this.controlflow.stream()
          .map(Controlflow::getBottomHeight)
          .max(Integer::compareTo)
          .orElse(layoutConfig.getNodeHeight() / 2);
    }

    public Map<NESTEdgeObject, List<java.awt.Point>> getControlflowEdgePaths() {
      Map<NESTEdgeObject, List<java.awt.Point>> edgePaths = new HashMap<>();
      this.getSplits().forEach(split -> edgePaths.putAll(split.getControlflowEdgePaths()));
      return edgePaths;
    }

    @Override
    public int compareTo(Branch other) {
      return this.getIngoingEdge()
          .getId()
          .compareTo(
              other
                  .getIngoingEdge()
                  .getId()); // compare edge ids so newly inserted nodes at the beginning of a
      // branch do not change its order (presuming the first edge stays the
      // same)
    }
  }

  /**
   * Linear sequence of sequence nodes without any splits in the controlflow. Begins with start node
   * or the first node after a controlflow start node or a controlflow end node. Ends with end node
   * or the last node before a controlflow end node. Example: ________|-b1-b2-b3-\ a1-a2-a3 d1-d2-d3
   * ________\-c1-c2-c3-/
   */
  class Sequence implements Comparable<Sequence>, Controlflow {

    private List<NESTSequenceNodeObject> sequence = new LinkedList<>();
    private Set<NESTDataNodeObject> bottomDataNodes = new HashSet<>();
    private Set<NESTDataNodeObject> topDataNodes = new HashSet<>();
    private Map<NESTSequenceNodeObject, Integer> sequenceNodesXPositions = new HashMap<>();

    public Sequence(NESTSequenceNodeObject firstNode) {
      initSequence(firstNode);
    }

    private void initSequence(NESTSequenceNodeObject currentNode) {
      this.sequence.add(currentNode);
      if (currentNode.getNextNodes().size() == 0 // end node
          || (currentNode.isNESTControlflowNode()
          && ((NESTControlflowNodeObject) currentNode)
          .isStartControlflowNode())) { // or start controlflow node
        return;
      }
      NESTSequenceNodeObject nextNode = currentNode.getNextNodes().iterator().next();
      if (nextNode.isNESTTaskNode()) {
        this.initSequence(nextNode);
        return;
      }
      if (nextNode.isNESTControlflowNode()
          && ((NESTControlflowNodeObject) nextNode).isStartControlflowNode()) {
        this.sequence.add(nextNode);
      }

      // nextNode should be an end controlflow node
    }

    @Override
    public int getHeight() {
      int topDataNodesHeight = NESTWorkflowLayout.this.getMaxNodeHeight(this.topDataNodes);
      int bottomDataNodesHeight = NESTWorkflowLayout.this.getMaxNodeHeight(this.bottomDataNodes);
      return NESTWorkflowLayout.this.getMaxNodeHeight(this.sequence)
          + (topDataNodesHeight > 0
          ? topDataNodesHeight
          + NESTWorkflowLayout.this.layoutConfig.getTaskNodeToDataNodeVerticalSpacing()
          : 0)
          + (bottomDataNodesHeight > 0
          ? bottomDataNodesHeight
          + NESTWorkflowLayout.this.layoutConfig.getTaskNodeToDataNodeVerticalSpacing()
          : 0);
    }

    @Override
    public int getTopHeight() {
      int topDataNodesHeight = NESTWorkflowLayout.this.getMaxNodeHeight(this.topDataNodes);
      return NESTWorkflowLayout.this.getMaxNodeHeight(this.sequence) / 2
          + (topDataNodesHeight > 0
          ? topDataNodesHeight
          + NESTWorkflowLayout.this.layoutConfig.getTaskNodeToDataNodeVerticalSpacing()
          : 0);
    }

    @Override
    public int getBottomHeight() {
      int bottomDataNodesHeight = NESTWorkflowLayout.this.getMaxNodeHeight(this.bottomDataNodes);
      return NESTWorkflowLayout.this.getMaxNodeHeight(this.sequence) / 2
          + (bottomDataNodesHeight > 0
          ? bottomDataNodesHeight
          + NESTWorkflowLayout.this.layoutConfig.getTaskNodeToDataNodeVerticalSpacing()
          : 0);
    }

    @Override
    public Map<NESTSequenceNodeObject, Sequence> getSequenceNodesToSequencesMapping() {
      Map<NESTSequenceNodeObject, Sequence> result = new HashMap<>();
      sequence.forEach(node -> result.put(node, this));
      return result;
    }

    @Override
    public void distributeDataNodes() {
      DataNodeDistribution distribution = new DataNodeDistribution(this.bottomDataNodes);
      this.bottomDataNodes = distribution.getBottomLayer();
      this.topDataNodes = distribution.getTopLayer();
    }

    @Override
    public Map<NESTNodeObject, Integer> getNodeVerticalPositions() {
      Map<NESTNodeObject, Integer> verticalPositions = new HashMap<>();
      int sequenceHeight = NESTWorkflowLayout.this.getMaxNodeHeight(this.sequence);
      this.sequence.forEach(
          sequenceNode ->
              verticalPositions.put(
                  sequenceNode,
                  -(int) NESTWorkflowLayout.this.getNodeSize(sequenceNode).getHeight() / 2));
      this.bottomDataNodes.forEach(
          dataNode ->
              verticalPositions.put(
                  dataNode,
                  sequenceHeight / 2
                      + NESTWorkflowLayout.this.layoutConfig.getTaskNodeToDataNodeVerticalSpacing()));
      this.topDataNodes.forEach(
          dataNode ->
              verticalPositions.put(
                  dataNode,
                  -sequenceHeight / 2
                      - ((int) NESTWorkflowLayout.this.getNodeSize(dataNode).getHeight()
                      + +NESTWorkflowLayout.this.layoutConfig.getTaskNodeToDataNodeVerticalSpacing())));
      return verticalPositions;
    }

    @Override
    public void calculateSequenceNodeHorizontalPositions() {
      Map<NESTSequenceNodeObject, Integer> xPositions = new HashMap<>();
      int currentX = 0;
      for (NESTSequenceNodeObject node : sequence) {
        Set<NESTEdgeObject> inControlflowEdges =
            node.getIngoingEdges(NESTEdgeObject::isNESTControlflowEdge);
        int maxEdgeLabelWidth =
            inControlflowEdges.stream()
                .map(edge -> (int) NESTWorkflowLayout.this.getEdgeLabelSize(edge).getWidth())
                .max(Integer::compareTo)
                .orElse(0);
        int maxEdgeWidth =
            inControlflowEdges.size() <= 0
                ? 0
                : maxEdgeLabelWidth + layoutConfig.getControlflowEdgeLabelHorizontalSpacing();
        int nodeDistance = Math.max(maxEdgeWidth, layoutConfig.getSequenceNodesHorizontalSpacing());
        int nodeX = currentX + nodeDistance;
        xPositions.put(node, nodeX);
        currentX = nodeX + (int) getNodeSize(node).getWidth();
      }
      nodeXPositions.putAll(xPositions);
      this.sequenceNodesXPositions = xPositions;
    }

    private Map<NESTSequenceNodeObject, Integer> getSequenceNodesXPositions() {
      return this.sequenceNodesXPositions;
    }

    /**
     * @param sequenceNodesXPositions X positions of the sequence nodes associated with the data
     *                                nodes in this sequence, needed for laying out the data nodes
     * @return
     */
    private Map<NESTDataNodeObject, Integer> getDataNodesXPositions(
        Map<NESTSequenceNodeObject, Integer> sequenceNodesXPositions) {
      Map<NESTDataNodeObject, Integer> xPositions = new HashMap<>();
      Stream.of(this.getTopDataNodes(), this.getBottomDataNodes())
          .forEach(
              dataNodeLayer -> {
                IntervalOverlapResolver overlapResolver = new IntervalOverlapResolver(10);
                dataNodeLayer.forEach(
                    dataNode -> {
                      int optimalX = getOptimalDataNodeXPosition(dataNode, sequenceNodesXPositions);
                      overlapResolver.add(
                          new IntervalOverlapResolver.Interval(
                              optimalX,
                              optimalX + (int) getNodeSize(dataNode).getWidth(),
                              dataNode.getId()) {
                            @Override
                            public int getShiftDistanceWeight() {
                              // tasks which are connected to this data node but are not part of the
                              // current sequence
                              List<NESTTaskNodeObject> foreignTasks =
                                  dataNode.getConnectedTasks().stream()
                                      .filter(taskNode -> !sequence.contains(taskNode))
                                      .collect(Collectors.toList());

                              // count tasks connected to this data node located left with -1 and
                              // the ones located to the right with 1
                              return foreignTasks.stream()
                                  .mapToInt(
                                      taskNode -> {
                                        if (nodeXPositions.get(taskNode) == null) {
                                          return 1; // if no position for this task node was
                                        }
                                        // calculated yet we assume the node is located
                                        // in a sequence to the right TODO would not be
                                        // true for parallel conotrolflows which
                                        // exchange data, are these possible?
                                        return (int)
                                            Math.signum(
                                                nodeXPositions.get(taskNode)
                                                    - (currentBranchXOffset + this.getStart()));
                                      })
                                  .sum();
                            }
                          });
                    });
                List<IntervalOverlapResolver.Interval> nonOverlappingIntervals =
                    overlapResolver.resolve();
                nonOverlappingIntervals.forEach(
                    interval -> {
                      NESTDataNodeObject dataNode =
                          dataNodeLayer.stream()
                              .filter(node -> node.getId().equals(interval.getId()))
                              .findAny()
                              .get();
                      xPositions.put(dataNode, interval.getStart());
                    });
              });
      return xPositions;
    }

    private int getOptimalDataNodeXPosition(
        NESTDataNodeObject dataNode, Map<NESTSequenceNodeObject, Integer> sequenceNodesXPositions) {
      Set<NESTNodeObject> validNeighbours =
          dataNode.getEdges(NESTEdgeObject::isNESTDataflowEdge).stream()
              .flatMap(edge -> Stream.of(edge.getPost(), edge.getPre()))
              .filter(
                  sequenceNodesXPositions
                      ::containsKey) // only use neighbours present in the sequence the data node
              // was assigned to
              .collect(Collectors.toSet());
      return (int)
          (validNeighbours.stream()
              .mapToDouble(
                  neighbour ->
                      sequenceNodesXPositions.get(neighbour)
                          + getNodeSize(neighbour).getWidth() / 2)
              .average()
              .getAsDouble()
              - getNodeSize(dataNode).getWidth() / 2);
    }

    @Override
    public Map<NESTNodeObject, Integer> getNodesXPositions(int startX) {
      Map<NESTSequenceNodeObject, Integer> sequenceNodesXPositions =
          this.getSequenceNodesXPositions();
      // combine data and sequence nodes
      Map<NESTNodeObject, Integer> xPositions = new HashMap<>();
      xPositions.putAll(sequenceNodesXPositions);
      xPositions.putAll(this.getDataNodesXPositions(sequenceNodesXPositions));
      int minX = xPositions.values().stream().min(Integer::compareTo).orElse(0);
      int shiftDistance = minX >= 0 ? 0 : -minX;
      // add the required offset to the x values and shift nodes to the right when the data nodes
      // are positioned too far left of the sequence nodes and thus could intersect with elements of
      // other sequences
      return xPositions.entrySet().stream()
          .collect(
              Collectors.toMap(
                  Map.Entry::getKey, entry -> entry.getValue() + startX + shiftDistance));
    }

    public NESTSequenceNodeObject getFirstNode() {
      return sequence.get(0);
    }

    public NESTSequenceNodeObject getLastNode() {
      return sequence.get(sequence.size() - 1);
    }

    public List<NESTSequenceNodeObject> getSequence() {
      return sequence;
    }

    public Set<NESTDataNodeObject> getBottomDataNodes() {
      return bottomDataNodes;
    }

    public Set<NESTDataNodeObject> getTopDataNodes() {
      return topDataNodes;
    }

    public boolean equals(Object o) {
      // Sequences should be considered equal when they have the same first node
      return (o instanceof Sequence) && ((Sequence) o).getFirstNode() == this.getFirstNode();
    }

    public int hashCode() {
      return this.getFirstNode().hashCode();
    }

    @Override
    public int compareTo(Sequence other) {
      return this.getFirstNode()
          .getId()
          .compareTo(
              other
                  .getFirstNode()
                  .getId()); // compare first node ids so that data nodes are always assigned to the
      // same sequence when associationCount is the same between multiple
      // sequences (see Branch.assignDataNodes())
    }
  }

  /**
   * https://codereview.stackexchange.com/questions/119976/an-iterator-returning-all-possible-partitions-of-a-list-in-java
   * This class implements an {@code Iterable} over all partitions of a given list.
   *
   * @param <T> The actual element type.
   * @author Rodion "rodde" Efremov
   * @version 1.6 (Feb 14, 2016 a.k.a. Friend Edition)
   */
  public class PartitionIterable<T> implements Iterable<List<List<T>>> {

    private final List<T> allElements = new ArrayList<>();
    private final int blocks;

    public PartitionIterable(List<T> allElements, int blocks) {
      checkNumberOfBlocks(blocks, allElements.size());
      this.allElements.addAll(allElements);
      this.blocks = blocks;
    }

    @Override
    public Iterator<List<List<T>>> iterator() {
      return new PartitionIterator<>(allElements, blocks);
    }

    private void checkNumberOfBlocks(int blocks, int numberOfElements) {
      if (blocks < 1) {
        throw new IllegalArgumentException(
            "The number of blocks should be at least 1, received: " + blocks);
      }

      if (blocks > numberOfElements) {
        throw new IllegalArgumentException(
            "The number of blocks should be at most " + numberOfElements + ", received: " + blocks);
      }
    }

    private final class PartitionIterator<T> implements Iterator<List<List<T>>> {

      private final List<T> allElements = new ArrayList<>();
      private final int blocks;
      private final int[] s;
      private final int[] m;
      private final int n;
      private List<List<T>> nextPartition;

      PartitionIterator(List<T> allElements, int blocks) {
        this.allElements.addAll(allElements);
        this.blocks = blocks;
        this.n = allElements.size();

        s = new int[n];
        m = new int[n];

        if (n != 0) {
          for (int i = 0; i < n - blocks + 1; ++i) {
            s[i] = 0;
            m[i] = 0;
          }

          for (int i = n - blocks + 1; i < n; ++i) {
            s[i] = m[i] = i - n + blocks;
          }

          loadPartition();
        }
      }

      @Override
      public boolean hasNext() {
        return nextPartition != null;
      }

      @Override
      public List<List<T>> next() {
        if (nextPartition == null) {
          throw new NoSuchElementException("No more partitions left.");
        }

        List<List<T>> partition = nextPartition;
        generateNextPartition();
        return partition;
      }

      private void loadPartition() {
        nextPartition = new ArrayList<>(blocks);

        for (int i = 0; i < blocks; ++i) {
          nextPartition.add(new ArrayList<>());
        }

        for (int i = 0; i < n; ++i) {
          nextPartition.get(s[i]).add(allElements.get(i));
        }
      }

      private void generateNextPartition() {
        for (int i = n - 1; i > 0; --i) {
          if (s[i] < blocks - 1 && s[i] <= m[i - 1]) {
            s[i]++;
            m[i] = Math.max(m[i], s[i]);

            for (int j = i + 1; j < n - blocks + m[i] + 1; ++j) {
              s[j] = 0;
              m[j] = m[i];
            }

            for (int j = n - blocks + m[i] + 1; j < n; ++j) {
              s[j] = m[j] = blocks - n + j;
            }

            loadPartition();
            return;
          }
        }

        nextPartition = null;
      }
    }
  }
}
