package de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor;

import static de.uni_trier.wi2.procake.gui.Constants.NESTWORKFLOW_RESOURCES;

import com.mxgraph.io.mxCodec;
import com.mxgraph.layout.hierarchical.mxHierarchicalLayout;
import com.mxgraph.model.mxCell;
import com.mxgraph.model.mxGeometry;
import com.mxgraph.model.mxGraphModel;
import com.mxgraph.model.mxICell;
import com.mxgraph.swing.handler.mxCellHandler;
import com.mxgraph.swing.handler.mxConnectionHandler;
import com.mxgraph.swing.handler.mxEdgeHandler;
import com.mxgraph.swing.handler.mxElbowEdgeHandler;
import com.mxgraph.swing.handler.mxVertexHandler;
import com.mxgraph.swing.mxGraphComponent;
import com.mxgraph.swing.util.mxGraphTransferable;
import com.mxgraph.swing.util.mxSwingConstants;
import com.mxgraph.swing.view.mxICellEditor;
import com.mxgraph.util.mxConstants;
import com.mxgraph.util.mxEvent;
import com.mxgraph.util.mxEventObject;
import com.mxgraph.util.mxEventSource.mxIEventListener;
import com.mxgraph.util.mxPoint;
import com.mxgraph.util.mxRectangle;
import com.mxgraph.util.mxResources;
import com.mxgraph.util.mxUtils;
import com.mxgraph.util.mxXmlUtils;
import com.mxgraph.view.mxCellState;
import com.mxgraph.view.mxEdgeStyle;
import com.mxgraph.view.mxEdgeStyle.mxEdgeStyleFunction;
import com.mxgraph.view.mxGraph;
import com.mxgraph.view.mxStylesheet;
import de.uni_trier.wi2.procake.data.model.DataClass;
import de.uni_trier.wi2.procake.data.model.nest.NESTGraphClass;
import de.uni_trier.wi2.procake.data.model.nest.NESTGraphItemClass;
import de.uni_trier.wi2.procake.data.model.nest.controlflowNode.NESTControlflowNodeClass;
import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.object.base.AggregateObject;
import de.uni_trier.wi2.procake.data.object.base.AtomicObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTAbstractWorkflowObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTControlflowEdgeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTEdgeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTGraphItemObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTSubWorkflowNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTTaskNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTWorkflowObject;
import de.uni_trier.wi2.procake.data.object.nest.controlflowNode.NESTControlflowNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.utils.impl.NESTSequentialWorkflowValidatorImpl;
import de.uni_trier.wi2.procake.data.object.nest.utils.impl.NESTWorkflowValidatorImpl;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.editor.BasicGraphEditor;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.editor.EditorActions;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.editor.EditorMenuBar;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.editor.EditorPalette;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.swing.layouts.BasicGridLayout;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.utils.Utils;
import de.uni_trier.wi2.procake.utils.io.IOUtil;
import java.awt.*;
import java.awt.datatransfer.DataFlavor;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collection;
import java.util.EventObject;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.imageio.ImageIO;
import javax.swing.*;
import org.w3c.dom.Document;

public class NESTWorkflowEditor extends BasicGraphEditor {

  public static final String DEFAULT_STYLE_XML_PATH = NESTWORKFLOW_RESOURCES + "default-style.xml";
  private static final long serialVersionUID = -4601740824088314699L;
  private static final Object windowCloseLock = new Object();
  private NESTAbstractWorkflowObject nestWorkflow;
  private JFrame editorFrame;
  private JSplitPane mainSplitPane;

  protected NESTWorkflowValidatorGUI nestWorkflowValidatorGUI;
  private Set<GraphSaveListener> graphSaveListeners = new HashSet<>();

  public NESTWorkflowEditor(NESTAbstractWorkflowObject nestWorkflow) {
    this(nestWorkflow, false, false);
  }

  public NESTWorkflowEditor(NESTAbstractWorkflowObject nestWorkflow,
      boolean blockThreadUntilEditorWindowClosed) {
    this(nestWorkflow, false, blockThreadUntilEditorWindowClosed);
  }

  public NESTWorkflowEditor(NESTAbstractWorkflowObject nestWorkflow, boolean frameless,
      boolean blockThreadUntilEditorWindowClosed) {
    this("NESTWorkflow Editor", new CustomGraphComponent(new CustomGraph()), frameless, true,
        (NESTGraphClass) nestWorkflow.getDataClass());
    this.nestWorkflow = (NESTAbstractWorkflowObject) nestWorkflow;
    try {
      SwingUtilities.invokeLater(this::initEditor);
    } catch (Exception e) {
      e.printStackTrace();
    }
    this.nestWorkflowValidatorGUI = new NESTWorkflowValidatorGUI(this);

    if (!frameless && blockThreadUntilEditorWindowClosed) {
      this.waitUntilEditorWindowClosed();
    }
  }

  public NESTWorkflowEditor(NESTAbstractWorkflowObject nestWorkflow, boolean frameless,
      boolean blockThreadUntilEditorWindowClosed, boolean useWorkingCopy, boolean generateToolbar) {
    this("NESTWorkflow Editor", new CustomGraphComponent(new CustomGraph()), frameless,
        generateToolbar, (NESTGraphClass) nestWorkflow.getDataClass());
    if (useWorkingCopy) {
      this.nestWorkflow = nestWorkflow;
    } else {
      this.nestWorkflow = (NESTAbstractWorkflowObject) nestWorkflow.copy();
    }
    try {
      SwingUtilities.invokeLater(this::initEditor);
    } catch (Exception e) {
      e.printStackTrace();
    }
    this.nestWorkflowValidatorGUI = new NESTWorkflowValidatorGUI(this);

    if (!frameless && blockThreadUntilEditorWindowClosed) {
      this.waitUntilEditorWindowClosed();
    }
  }

  public NESTWorkflowEditor(String appTitle, mxGraphComponent component, boolean frameless,
      boolean generateToolbar, NESTGraphClass nestClass) {
    super(appTitle, component, frameless, generateToolbar);

    final mxGraph graph = graphComponent.getGraph();

    // Creates the shapes palette
    EditorPalette shapesPalette = insertPalette(mxResources.get("shapes"));

    // Sets the edge template to be used for creating new edges if an edge
    // is clicked in the shape palette
    shapesPalette.addListener(
        mxEvent.SELECT,
        new mxIEventListener() {
          public void invoke(Object sender, mxEventObject evt) {
            Object tmp = evt.getProperty("transferable");

            if (tmp instanceof mxGraphTransferable) {
              mxGraphTransferable t = (mxGraphTransferable) tmp;
              Object cell = t.getCells()[0];

              if (graph.getModel().isEdge(cell)) {
                ((CustomGraph) graph).setEdgeTemplate(cell);
              }
            }
          }
        }
    );

    Collection<NESTGraphItemClass> itemClasses = nestClass.getNESTGraphItemClasses();

    // Add template cells for dropping into the graph
    try {

      if (itemClasses.stream().anyMatch(DataClass::isNESTWorkflowNode)) {
        shapesPalette.addTemplate(
            "Workflow",
            new ImageIcon(ImageIO.read(
                NESTWorkflowEditor.class.getResourceAsStream(
                    NESTWORKFLOW_RESOURCES + "images/rhombus.png"))),
            CellStyle.WORKFLOW_NODE_STYLE.name(),
            50,
            50,
            NodeInsertType.WORKFLOW);
      }
      if (itemClasses.stream().anyMatch(DataClass::isNESTSubWorkflowNode)) {
        shapesPalette.addTemplate(
            "Subworkflow",
            new ImageIcon(ImageIO.read(
                NESTWorkflowEditor.class.getResourceAsStream(
                    NESTWORKFLOW_RESOURCES + "images/rhombus.png"))),
            CellStyle.SUB_WORKFLOW_NODE_STYLE.name(),
            50,
            50,
            NodeInsertType.SUBWORKFLOW);
      }
    } catch (IOException e) {
      System.err.println("Error while loading template");
      e.printStackTrace();
    }

    // Add stencils for different node types
    try {

      String nodeXml;

      if (itemClasses.stream().anyMatch(DataClass::isNESTTaskNode)) {
        nodeXml = mxUtils.readInputStream(
            NESTWorkflowEditor.class.getResourceAsStream(
                NESTWORKFLOW_RESOURCES + "stencils/Task.shape"));
        EditorActions.ImportAction.addStencilShape(
            shapesPalette,
            nodeXml,
            NESTWORKFLOW_RESOURCES + "stencils/Task.png",
            CellStyle.TASK_NODE_STYLE.name(),
            NodeInsertType.TASK);
      }

      if (itemClasses.stream().anyMatch(DataClass::isNESTDataNode)) {
        shapesPalette.addTemplate(
            "Data",
            new ImageIcon(ImageIO.read(
                NESTWorkflowEditor.class.getResourceAsStream(
                    NESTWORKFLOW_RESOURCES + "images/ellipse.png"))),
            CellStyle.DATA_NODE_STYLE.name(),
            80,
            50,
            NodeInsertType.DATA);
      }

      if (itemClasses.stream().anyMatch(
          n -> n.isNESTControlflowNode() && ((NESTControlflowNodeClass) n).isAndNode())) {
        nodeXml = mxUtils.readInputStream(
            NESTWorkflowEditor.class.getResourceAsStream(
                NESTWORKFLOW_RESOURCES + "stencils/AND.shape"));
        EditorActions.ImportAction.addStencilShape(
            shapesPalette,
            nodeXml,
            NESTWORKFLOW_RESOURCES + "stencils/AND.png",
            CellStyle.AND_SPLIT_NODE_STYLE.name(),
            NodeInsertType.AND_BLOCK);
      }

      if (itemClasses.stream().anyMatch(
          n -> n.isNESTControlflowNode() && ((NESTControlflowNodeClass) n).isXorNode())) {
        nodeXml = mxUtils.readInputStream(NESTWorkflowEditor.class
            .getResourceAsStream(NESTWORKFLOW_RESOURCES + "stencils/XOR.shape"));
        EditorActions.ImportAction.addStencilShape(
            shapesPalette,
            nodeXml,
            NESTWORKFLOW_RESOURCES + "stencils/XOR.png",
            CellStyle.XOR_SPLIT_NODE_STYLE.name(),
            NodeInsertType.XOR_BLOCK);
      }

      if (itemClasses.stream().anyMatch(
          n -> n.isNESTControlflowNode() && ((NESTControlflowNodeClass) n).isOrNode())) {
        nodeXml = mxUtils.readInputStream(NESTWorkflowEditor.class
            .getResourceAsStream(NESTWORKFLOW_RESOURCES + "stencils/OR.shape"));
        EditorActions.ImportAction.addStencilShape(
            shapesPalette,
            nodeXml,
            NESTWORKFLOW_RESOURCES + "stencils/OR.png",
            CellStyle.OR_SPLIT_NODE_STYLE.name(),
            NodeInsertType.OR_BLOCK);
      }

      if (itemClasses.stream().anyMatch(
          n -> n.isNESTControlflowNode() && ((NESTControlflowNodeClass) n).isLoopNode())) {
        nodeXml = mxUtils.readInputStream(
            NESTWorkflowEditor.class.getResourceAsStream(
                NESTWORKFLOW_RESOURCES + "stencils/LOOP.shape"));
        EditorActions.ImportAction.addStencilShape(
            shapesPalette,
            nodeXml,
            NESTWORKFLOW_RESOURCES + "stencils/LOOP.png",
            CellStyle.LOOP_SPLIT_NODE_STYLE.name(),
            NodeInsertType.LOOP_BLOCK);
      }
    } catch (IOException e) {
      System.err.println("Error while loading stencil");
      e.printStackTrace();
    }
  }

  private void initEditor() {
    // make hotspot of cells cover their whole size for easier edge creation, cells are moved with
    // left mouse button drag,
    // new edges are created with right mouse button drag
    // see custom createConnectionHandler in CustomGraphComponent
    this.getGraphComponent().getConnectionHandler().getMarker().setHotspot(0.95);

    mxSwingConstants.SHADOW_COLOR = Color.LIGHT_GRAY;
    mxConstants.W3C_SHADOWCOLOR = "#D3D3D3";

    CustomGraph customGraph = (CustomGraph) super.getGraphComponent().getGraph();
    customGraph.loadNESTWorkflow(this.nestWorkflow);
    this.applyNodeVisibilitySettings();

    if (!super.frameless) {
      this.editorFrame = this.createFrame(new EditorMenuBar(this),
          mainSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT));
      this.editorFrame.addWindowListener(new WindowAdapter() {
        @Override
        public void windowClosing(WindowEvent arg0) {
          synchronized (NESTWorkflowEditor.windowCloseLock) {
            editorFrame.setVisible(false);
            NESTWorkflowEditor.windowCloseLock.notify();
            NESTWorkflowEditor.this.getNestWorkflowValidatorGUI().dispose();
          }
        }
      });
      super.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      this.editorFrame.setSize(800, 600);
      this.editorFrame.setVisible(true);
      this.editorFrame.toFront();
      this.editorFrame.repaint();
    }

    graphComponent.zoomTo(1.0, graphComponent.isCenterZoom());
  }

  private void waitUntilEditorWindowClosed() {
    try {
      while (editorFrame == null || !editorFrame.isVisible()) {
        Thread.sleep(100);
      }
    } catch (InterruptedException e) {
      e.printStackTrace();
    }
    synchronized (windowCloseLock) {
      while (editorFrame.isVisible()) {
        try {
          windowCloseLock.wait();
        } catch (InterruptedException e) {
          e.printStackTrace();
        }
      }
    }
  }

  public void open(Object parsedObject) {
    if (parsedObject instanceof NESTWorkflowObject) {
      CustomGraph customGraph = (CustomGraph) super.getGraphComponent().getGraph();
      customGraph.clearWithoutEvents();
      this.nestWorkflow = (NESTWorkflowObject) parsedObject;
      customGraph.loadNESTWorkflow(this.nestWorkflow);
      this.applyNodeVisibilitySettings();
      graphComponent.zoomTo(1.0, graphComponent.isCenterZoom());
      super.updateTitle();
    }
  }

  public void saveInObjectNESTWorkflow(NESTWorkflowObject nestWorkflow) {

  }

  public void applyNodeVisibilitySettings() {
    ActionEvent e = new ActionEvent(this, 0, "applyNodeVisibilitySettings");
    EditorActions.getActionFor(EditorActions.ToggleWorkflowNodeVisibilityAction.class)
        .actionPerformed(e);
    EditorActions.getActionFor(EditorActions.ToggleDataNodeVisibilityAction.class)
        .actionPerformed(e);
    EditorActions.getActionFor(EditorActions.ToggleSequenceNodeVisibilityAction.class)
        .actionPerformed(e);
    EditorActions.getActionFor(EditorActions.ToggleIdVisibilityAction.class)
        .actionPerformed(e);
    EditorActions.getActionFor(EditorActions.ToggleEdgeLabelVisibilityAction.class)
        .actionPerformed(e);
    EditorActions.getActionFor(EditorActions.ToggleCellTooltipVisibilityAction.class)
        .actionPerformed(e);
  }

  public NESTAbstractWorkflowObject getNESTWorkflow() {
    return ((NESTWorkflowEditor.CustomGraph) this.getGraphComponent().getGraph()).getNestWorkflow();
  }

  public void setNestWorkflow(NESTWorkflowObject nestWorkflow) {
    this.nestWorkflow = nestWorkflow;
  }

  public JFrame getEditorFrame() {
    return editorFrame;
  }

  public NESTWorkflowValidatorGUI getNestWorkflowValidatorGUI() {
    return nestWorkflowValidatorGUI;
  }

//  public NESTAbstractWorkflowObject getOriginalNestWorkflow() {
//    return originalNestWorkflow;
//  }

  public void addGraphSaveListener(GraphSaveListener listener) {
    this.graphSaveListeners.add(listener);
  }

  public void removeGraphSaveListener(GraphSaveListener listener) {
    this.graphSaveListeners.remove(listener);
  }

  public void fireGraphSaved() {
    graphSaveListeners.forEach(listener -> listener.graphSaved(this.nestWorkflow));
  }

  public enum CellStyle {
    // node style names (have to exist in xml style file)
    WORKFLOW_NODE_STYLE,
    SUB_WORKFLOW_NODE_STYLE,

    TASK_NODE_STYLE,
    DATA_NODE_STYLE,

    AND_JOIN_NODE_STYLE,
    AND_SPLIT_NODE_STYLE,

    XOR_JOIN_NODE_STYLE,
    XOR_SPLIT_NODE_STYLE,

    OR_JOIN_NODE_STYLE,
    OR_SPLIT_NODE_STYLE,

    LOOP_JOIN_NODE_STYLE,
    LOOP_SPLIT_NODE_STYLE,

    UNKNOWN_NODE_STYLE,

    // edge style names (have to exist in xml style file)
    CONTROLFLOW_EDGE_STYLE,
    DATAFLOW_EDGE_STYLE,
    PART_OF_EDGE_STYLE,
    CONSTRAINT_EDGE_STYLE,
    UNKNOWN_EDGE_STYLE;

    public static String get(NESTEdgeObject edge) {
      if (edge.isNESTControlflowEdge()) {
        return CellStyle.CONTROLFLOW_EDGE_STYLE.name();
      } else if (edge.isNESTDataflowEdge()) {
        return CellStyle.DATAFLOW_EDGE_STYLE.name();
      } else if (edge.isNESTPartOfEdge()) {
        return CellStyle.PART_OF_EDGE_STYLE.name();
      } else if (edge.isNESTConstraintEdge()) {
        return CellStyle.CONSTRAINT_EDGE_STYLE.name();
      } else {
        return CellStyle.UNKNOWN_EDGE_STYLE.name();
      }
    }

    public static String get(NESTNodeObject node) {
      if (node.isNESTTaskNode()) {
        return CellStyle.TASK_NODE_STYLE.name();
      } else if (node.isNESTDataNode()) {
        return CellStyle.DATA_NODE_STYLE.name();
      } else if (node.isNESTWorkflowNode()) {
        return CellStyle.WORKFLOW_NODE_STYLE.name();
      } else if (node.isNESTSubWorkflowNode()) {
        return CellStyle.SUB_WORKFLOW_NODE_STYLE.name();
      } else if (node.isNESTControlflowNode()) {
        NESTControlflowNodeObject cfNode = (NESTControlflowNodeObject) node;
        if (cfNode.isAndNode()) {
          return cfNode.isEndControlflowNode()
              ? CellStyle.AND_JOIN_NODE_STYLE.name()
              : CellStyle.AND_SPLIT_NODE_STYLE.name();
        }
        if (cfNode.isXorNode()) {
          return cfNode.isEndControlflowNode()
              ? CellStyle.XOR_JOIN_NODE_STYLE.name()
              : CellStyle.XOR_SPLIT_NODE_STYLE.name();
        }
        if (cfNode.isOrNode()) {
          return cfNode.isEndControlflowNode()
              ? CellStyle.OR_JOIN_NODE_STYLE.name()
              : CellStyle.OR_SPLIT_NODE_STYLE.name();
        }
        if (cfNode.isLoopNode()) {
          return cfNode.isEndControlflowNode()
              ? CellStyle.LOOP_JOIN_NODE_STYLE.name()
              : CellStyle.LOOP_SPLIT_NODE_STYLE.name();
        }
        return CellStyle.UNKNOWN_NODE_STYLE.name();
      } else {
        return CellStyle.UNKNOWN_NODE_STYLE.name();
      }
    }
  }

  public interface GraphSaveListener {

    public void graphSaved(NESTAbstractWorkflowObject nestWorkflow);
  }

  /**
   *
   */
  public static class CustomGraphComponent extends mxGraphComponent {

    /**
     *
     */
    private static final long serialVersionUID = -6833603133512882012L;

    static String NAME = "CustomGraphComponent";
    private String stylesheet;

    static {
      try { // prevent NotSerializable exception by using local reference flavor instead of
        // serialization
        mxGraphTransferable.dataFlavor =
            new DataFlavor(
                DataFlavor.javaJVMLocalObjectMimeType
                    + "; class=com.mxgraph.swing.util.mxGraphTransferable");
      } catch (ClassNotFoundException cnfe) {
        System.err.println(cnfe);
      }
    }

    /**
     * @param graph
     */
    public CustomGraphComponent(mxGraph graph) {
      super(graph);
      this.setName(NAME);
      this.setTolerance(10); // allows for higher reliability when splitting edges
      this.getConnectionHandler().addListener(mxEvent.CONNECT, new EdgeCreationListener(
          graph)); // register CellConnectListener which reflects edge creation to the underlying NESTWorkflow
      graph.addListener(mxEvent.CELLS_REMOVED, new CellRemoveListener());
      graph.addListener(mxEvent.CELLS_ADDED, new CellAddListener());
      graph.addListener(mxEvent.SPLIT_EDGE, new EdgeSplitListener());

      // Sets switches typically used in an editor
      //            setPageVisible(true);
      //			setGridVisible(true);
      setToolTips(false);
      ToolTipManager.sharedInstance().setDismissDelay(30000);
      getConnectionHandler().setCreateTarget(false);

      // Loads the default stylesheet from an external file
      try {
        String defaultStyle = new String(
            IOUtil.getInputStream(DEFAULT_STYLE_XML_PATH).readAllBytes(), StandardCharsets.UTF_8);
        this.setStylesheet(defaultStyle);
      } catch (IOException e) {
        System.err.println("Default style could not be loaded from " + DEFAULT_STYLE_XML_PATH);
        e.printStackTrace();
      }

      // Sets the background to white
      getViewport().setOpaque(true);
      getViewport().setBackground(Color.WHITE);
      this.getConnectionHandler().getMarker().addListener(mxEvent.MARK, new Highlighter());
    }

    public void setStylesheet(String xml) {
      this.stylesheet = xml;
      mxStylesheet newStylesheet = new mxStylesheet();
      mxCodec codec = new mxCodec();
      Document doc = mxXmlUtils.parseXml(xml);
      codec.decode(doc.getDocumentElement(), newStylesheet);
      graph.setStylesheet(newStylesheet);
    }

    public String getStylesheet() {
      return stylesheet;
    }

    /**
     * Override so that CustomVertexHandler is used
     */
    @Override
    public mxCellHandler createHandler(mxCellState state) {
      {
        if (graph.getModel().isVertex(state.getCell())) {
          // returns custom mxVertexHandler here
          return new CustomVertexHandler(this, state);
        } else if (graph.getModel().isEdge(state.getCell())) {
          mxEdgeStyleFunction style = graph.getView().getEdgeStyle(state,
              null, null, null);

          if (graph.isLoop(state) || style == mxEdgeStyle.ElbowConnector
              || style == mxEdgeStyle.SideToSide
              || style == mxEdgeStyle.TopToBottom) {
            return new mxElbowEdgeHandler(this, state);
          }

          return new mxEdgeHandler(this, state);
        }

        return new mxCellHandler(this, state);
      }
    }

    @Override
    protected mxICellEditor createCellEditor() {
      return new mxICellEditor() {
        SemanticDescriptorEditor semanticDescriptorEditor =
            new SemanticDescriptorEditor(CustomGraphComponent.this);

        @Override
        public Object getEditingCell() {
          return null;
        }

        @Override
        public void startEditing(Object cell, EventObject trigger) {
          Object valueToEdit = ((mxICell) cell).getValue();
          if (valueToEdit instanceof NESTGraphItemObject) {
            semanticDescriptorEditor.startEditing(cell, trigger);
          } else if (valueToEdit instanceof String) {
            new StringCellEditor((mxICell) cell, CustomGraphComponent.this);
          }
        }

        @Override
        public void stopEditing(boolean cancel) {
          // do nothing
        }

      };

    }

    public void updateAllCellLabels() {
      var cells = this.getGraph().getChildCells(this.getGraph().getDefaultParent(), true, true);
      Arrays.stream(cells)
          .forEach(cell -> this.labelChanged(cell, ((mxICell) cell).getValue(), null));
    }

    @Override
    protected mxConnectionHandler createConnectionHandler() {
      // ConnectionHandler that does nothing on left click and only initiates connection mode on
      // right click
      // this allows the cell to be moved when dragged with left mouse down
      return new mxConnectionHandler(this) {
        @Override
        public void mousePressed(MouseEvent e) {
          if (SwingUtilities.isLeftMouseButton(e)) {
            // do nothing, as connection mode should not be initiated
            // this will cause the other handler to take over the action and move the cell
          } else {
            super.mousePressed(e);
          }
        }
      };
    }

    class Highlighter implements mxIEventListener {

      final int OPACITY_HIGH = 100;
      final int OPACITY_LOW = 20;

      @Override
      public void invoke(Object connectionHandler, mxEventObject event) {
        Object[] allCells = graph.getChildCells(graph.getDefaultParent());
        mxCellState cellState = (mxCellState) event.getProperty("state");
        if (cellState != null) { // cell is marked
          mxICell markedCell = (mxICell) cellState.getCell();
          if (!(markedCell.getValue() instanceof NESTGraphItemObject)) {
            return;
          }
          Arrays.asList(allCells)
              .forEach(cell -> setCellOpacity(cell, OPACITY_LOW)); // lowlight everything
          setCellOpacity(markedCell, OPACITY_HIGH); // highlight hovered cell
          highlightPartOfEdgeAndWorkflowNode((NESTNodeObject) markedCell.getValue());
          highlightFor((NESTNodeObject) markedCell.getValue());
        } else { // cell is unmarked
          Arrays.asList(allCells)
              .forEach(cell -> setCellOpacity(cell, OPACITY_HIGH)); // highlight everything
        }
        mxRectangle bounds = graph.getBoundsForCells(allCells, true, true, true);
        graph.repaint(bounds);
      }

      private void highlightPartOfEdgeAndWorkflowNode(NESTNodeObject nestNode) {
        mxGraphModel model = (mxGraphModel) graph.getModel();
        nestNode
            .getOutgoingEdges(NESTEdgeObject::isNESTPartOfEdge)
            .forEach(
                edge -> {
                  setCellOpacity(model.getCell(edge.getId()), OPACITY_HIGH);
                  if (edge.getPost() != null) {
                    setCellOpacity(model.getCell(edge.getPost().getId()), OPACITY_HIGH);
                  }
                });
      }

      private void highlightFor(NESTNodeObject nestNode) {
        if (nestNode.isNESTSubWorkflowNode() || nestNode.isNESTWorkflowNode()) {
          highlightForWorkflowNodeAndSubworkflowNode(nestNode);
        } else if (nestNode.isNESTTaskNode()) {
          highlightForTaskNode(nestNode);
        } else if (nestNode.isNESTDataNode()) {
          highlightForDataNode(nestNode);
        } else if (nestNode.isNESTControlflowNode()) {
          highlightForControlflowNode(nestNode);
        }
      }

      private void setCellOpacity(Object cell, int opacity) {
        if (graph.getView().getState(cell) != null) {
          graph.getView().getState(cell).getStyle().put(mxConstants.STYLE_OPACITY, opacity);
          graph.getView().getState(cell).getStyle().put(mxConstants.STYLE_TEXT_OPACITY, opacity);
        }
      }

      private void highlightForControlflowNode(NESTNodeObject nestNode) {
        mxGraphModel model = (mxGraphModel) graph.getModel();
        NESTControlflowNodeObject controlflowNode = (NESTControlflowNodeObject) nestNode;
        if (controlflowNode.getMatchingBlockControlflowNode() != null) {
          setCellOpacity(
              model.getCell(controlflowNode.getMatchingBlockControlflowNode().getId()),
              OPACITY_HIGH);
        }
        var innerBlockElements = controlflowNode.getInnerBlockElements();
        if (innerBlockElements == null) {
          return; // fix for open branches
        }
        innerBlockElements
            .forEach(node -> setCellOpacity(model.getCell(node.getId()), OPACITY_HIGH));
        innerBlockElements.stream()
            .flatMap(node -> node.getEdges(NESTEdgeObject::isNESTControlflowEdge).stream())
            .forEach(edge -> setCellOpacity(model.getCell(edge.getId()), OPACITY_HIGH));
        controlflowNode
            .getEdges()
            .forEach(
                edge ->
                    setCellOpacity(
                        model.getCell(edge.getId()), OPACITY_HIGH)); // catch empty branch edges
      }

      private void highlightForTaskNode(NESTNodeObject nestNode) {
        mxGraphModel model = (mxGraphModel) graph.getModel();
        var dataflowEdges = nestNode.getEdges(NESTEdgeObject::isNESTDataflowEdge);
        dataflowEdges.forEach(edge -> setCellOpacity(model.getCell(edge.getId()), OPACITY_HIGH));
        dataflowEdges.stream()
            .filter(edge -> edge.getPre() != null && edge.getPost() != null)
            .map(
                edge ->
                    Stream.of(edge.getPost(), edge.getPre())
                        .filter(NESTNodeObject::isNESTDataNode)
                        .findFirst()
                        .get()) // Data nodes connected to this task node
            .forEach(dataNode -> setCellOpacity(model.getCell(dataNode.getId()), OPACITY_HIGH));
      }

      private void highlightForDataNode(NESTNodeObject nestNode) {
        mxGraphModel model = (mxGraphModel) graph.getModel();
        var dataflowEdges = nestNode.getEdges(NESTEdgeObject::isNESTDataflowEdge);
        dataflowEdges.forEach(edge -> setCellOpacity(model.getCell(edge.getId()), OPACITY_HIGH));
        dataflowEdges.stream()
            .filter(edge -> edge.getPre() != null && edge.getPost() != null)
            .map(
                edge ->
                    Stream.of(edge.getPost(), edge.getPre())
                        .filter(NESTNodeObject::isNESTTaskNode)
                        .findFirst()
                        .get()) // Task nodes connected to this data node
            .forEach(taskNode -> setCellOpacity(model.getCell(taskNode.getId()), OPACITY_HIGH));
      }

      private void highlightForWorkflowNodeAndSubworkflowNode(NESTNodeObject nestNode) {
        var ingoingPartOfEdges = nestNode.getIngoingEdges(NESTEdgeObject::isNESTPartOfEdge);
        mxGraphModel model = (mxGraphModel) graph.getModel();
        ingoingPartOfEdges.forEach(
            partOfEdge -> {
              var partOfEdgeCell = model.getCell(partOfEdge.getId());
              setCellOpacity(partOfEdgeCell, OPACITY_HIGH);

              if (partOfEdge.getPre() == null) {
                return;
              }

              mxICell connectedCell = (mxICell) model.getCell(partOfEdge.getPre().getId());
              setCellOpacity(connectedCell, OPACITY_HIGH);

              var containedEdges =
                  partOfEdge
                      .getPre()
                      .getEdges(
                          edge ->
                              Stream.of(edge.getPost(), edge.getPre())
                                  .filter(Objects::nonNull)
                                  .allMatch(
                                      node ->
                                          node.getOutgoingEdges(
                                                  adjacentEdge ->
                                                      adjacentEdge.isNESTPartOfEdge()
                                                          && adjacentEdge.getPost() == nestNode)
                                              .size()
                                              >= 1));
              containedEdges.forEach(
                  edge -> setCellOpacity(model.getCell(edge.getId()), OPACITY_HIGH));
              partOfEdge
                  .getPre()
                  .getOutgoingEdges()
                  .forEach(edge -> setCellOpacity(model.getCell(edge.getId()), OPACITY_HIGH));
              var connectedNESTNode = ((NESTNodeObject) connectedCell.getValue());
              if (connectedNESTNode.isNESTSubWorkflowNode()) {
                highlightForWorkflowNodeAndSubworkflowNode(connectedNESTNode);
              }
            });
      }
    }
  }

  /**
   * A graph that creates new edges from a given template edge.
   */
  public static class CustomGraph extends mxGraph {

    /**
     * Holds the edge to be used as a template for inserting new edges.
     */
    protected Object edgeTemplate;

    private CellLabelGenerator cellLabelGenerator = new CellLabelGenerator();
    private NESTAbstractWorkflowObject nestWorkflow;
    private NESTWorkflowLayout layout;

    /**
     * Custom graph that defines the alternate edge style to be used when the middle control point
     * of edges is double clicked (flipped).
     */
    public CustomGraph() {
      setAlternateEdgeStyle("edgeStyle=mxEdgeStyle.ElbowConnector;elbow=vertical");
      this.setAllowDanglingEdges(false);
      //            this.setHtmlLabels(true);
      layout = new NESTWorkflowLayoutForMxGraph(this, nestWorkflow);
      this.addListener(mxEvent.CONNECT_CELL, new EdgeTerminalModificationListener());
    }

    /**
     * Sets the edge template to be used to inserting edges.
     */
    public void setEdgeTemplate(Object template) {
      edgeTemplate = template;
    }

    /**
     * Prints out some useful information about the cell in the tooltip.
     */
    public String getToolTipForCell(Object cell) {
      NESTGraphItemObject item = (NESTGraphItemObject) ((mxICell) cell).getValue();
      DataObject semanticDescriptor = item.getSemanticDescriptor();
      String heading = "ID: " + item.getId() + " (" + item.getDataClass().getName() + ")";

      FontMetrics metrics = new Canvas().getFontMetrics((Font) UIManager.get("ToolTip.font"));
      int headingWidth = metrics.stringWidth(heading); //

      String tooltip =
          "<html><p width="
              + headingWidth
              + ">ID: "
              + item.getId()
              + " ("
              + item.getDataClass().getName()
              + ")</p><hr>"; // without width set, in some cases an unwanted line break happens
      if (semanticDescriptor != null && semanticDescriptor.isAggregate()) {
        tooltip += this.aggregateToHtmlTable((AggregateObject) semanticDescriptor, true);
      } else {
        tooltip += Objects.toString(semanticDescriptor);
      }
      return tooltip + "<br></html>"; // new line as workaround for bug in JDK:
      // https://bugs.openjdk.java.net/browse/JDK-8213786
    }

    public String aggregateToHtmlTable(
        AggregateObject aggregateObject,
        boolean includeDataClassNames,
        boolean includeNullAttributes) {
      StringBuilder sb = new StringBuilder("<table cellpadding='0' border='0' cellspacing='0'>");
      for (String attributeName : aggregateObject.getAggregateClass().getAttributeNames()) {
        DataObject dataObject = aggregateObject.getAttributeValue(attributeName);
        if (dataObject != null || includeNullAttributes) {
          sb.append("<tr><td align='right' valign='top'><b>")
              .append(attributeName)
              .append(": </b></td><td>");
          if (dataObject == null) {
            sb.append("null");
          } else if (dataObject.isAggregate()) {
            sb.append(
                this.aggregateToHtmlTable((AggregateObject) dataObject, includeDataClassNames));
          } else if (dataObject.isAtomic()) {
            sb.append(
                includeDataClassNames
                    ? dataObject.toString()
                    : ((AtomicObject) dataObject).getNativeObject().toString());
          } else {
            sb.append(dataObject.toString());
          }
          sb.append("</td></tr>");
        }
      }
      sb.append("</table>");
      return sb.toString();
    }

    public String aggregateToHtmlTable(
        AggregateObject aggregateObject, boolean includeDataClassNames) {
      return this.aggregateToHtmlTable(aggregateObject, includeDataClassNames, true);
    }

    /**
     * Overrides the method to use the currently selected edge template for new edges.
     *
     * @param parent
     * @param id
     * @param value
     * @param source
     * @param target
     * @param style
     * @return
     */
    public Object createEdge(
        Object parent, String id, Object value, Object source, Object target, String style) {
      if (edgeTemplate != null) {
        mxCell edge = (mxCell) cloneCells(new Object[]{edgeTemplate})[0];
        edge.setId(id);

        return edge;
      }

      return super.createEdge(parent, id, value, source, target, style);
    }

    public Object splitEdge(Object edge, Object[] cells, double dx, double dy) {
      // clone cell when a template cell from the palette is dropped on an edge
      // otherwise the reference to the template cell is replaced by the actual node instance and
      // thus nodes newly
      // inserted thereafter all point to the same cell instance
      // happens because of switch of DataFlavor from serialization to local reference (which was
      // introduced to prevent
      // NotSerializableException)
      if (Arrays.stream(cells)
          .anyMatch(cell -> ((mxICell) cell).getValue() instanceof NodeInsertType)) {
        cells = this.cloneCells(cells);
      }
      return super.splitEdge(edge, cells, dx, dy);
    }

    // Overrides method to provide a cell label in the display
    public String convertValueToString(Object cell) {
      if (cell instanceof mxCell) {
        Object value = ((mxCell) cell).getValue();
        if (value instanceof NESTGraphItemObject) {
          String label = cellLabelGenerator.getLabelFor((NESTGraphItemObject) value);
          // A Label may contain a semantic description consisting of key-value-pairs, separated by
          // line breaks. This does not play nice with the automatic node size calculation in
          // getPreferredSizeForCell(). Therefore, HTML-labels are used.
          this.setHtmlLabels(true);
          label = label.replace("\n", "<br>");
          return "<html>" + label + "</html>";
        }
      }
      return super.convertValueToString(cell);
    }

    @Override
    public String getEdgeValidationError(Object edge, Object source, Object target) {
      abstract class ValidationRule {

        NESTNodeObject sourceNode;
        NESTNodeObject targetNode;
        NESTEdgeObject edge;
        String constraintDescription;
        boolean changingSource;
        boolean changingTarget;

        public ValidationRule(
            NESTNodeObject sourceNode,
            NESTNodeObject targetNode,
            NESTEdgeObject edge,
            String constraintDescription) {
          this(sourceNode, targetNode, edge, constraintDescription, false, false);
        }

        public ValidationRule(
            NESTNodeObject sourceNode,
            NESTNodeObject targetNode,
            NESTEdgeObject edge,
            String constraintDescription,
            boolean changingSource,
            boolean changingTarget) {
          this.sourceNode = sourceNode;
          this.targetNode = targetNode;
          this.edge = edge;
          this.constraintDescription = constraintDescription;
          this.changingSource = changingSource;
          this.changingTarget = changingTarget;
        }

        public String getConstraintDescription() {
          return constraintDescription;
        }

        abstract boolean isApplicable();

        abstract boolean isViolated();
      }
      class MaxOneOutgoingPartOfEdge extends ValidationRule {

        public MaxOneOutgoingPartOfEdge(
            NESTNodeObject sourceNode,
            NESTNodeObject targetNode,
            NESTEdgeObject edge,
            boolean changingTarget) {
          super(
              sourceNode,
              targetNode,
              edge,
              "Only one outgoing partOf edge for this source node allowed.",
              false,
              changingTarget);
        }

        @Override
        boolean isApplicable() {
          return edge.isNESTPartOfEdge()
              && (sourceNode.isNESTTaskNode()
              || sourceNode.isNESTControlflowNode()
              || sourceNode.isNESTDataNode()
              || sourceNode.isNESTSubWorkflowNode());
        }

        @Override
        boolean isViolated() {
          return sourceNode.getOutgoingEdges(NESTEdgeObject::isNESTPartOfEdge).size()
              - (changingTarget ? 1 : 0)
              >= 1;
        }
      }
      class MaxOneOutgoingControlflowEdge extends ValidationRule {

        public MaxOneOutgoingControlflowEdge(
            NESTNodeObject sourceNode,
            NESTNodeObject targetNode,
            NESTEdgeObject edge,
            boolean changingTarget) {
          super(
              sourceNode,
              targetNode,
              edge,
              "Only one outgoing controlflow edge for this source node allowed.",
              false,
              changingTarget);
        }

        @Override
        boolean isApplicable() {
          if (!edge.isNESTControlflowEdge()
              || !(sourceNode.isNESTControlflowNode() || sourceNode.isNESTTaskNode())) {
            return false;
          }
          if (sourceNode.isNESTTaskNode()) {
            return true;
          }
          NESTControlflowNodeObject controlflowNode = (NESTControlflowNodeObject) this.sourceNode;
          return controlflowNode.isLoopStartNode()
              || controlflowNode.isXorEndNode()
              || controlflowNode.isAndEndNode()
              || controlflowNode.isOrEndNode();
        }

        @Override
        boolean isViolated() {
          return sourceNode.getOutgoingEdges(NESTEdgeObject::isNESTControlflowEdge).size()
              - (changingTarget ? 1 : 0)
              >= 1;
        }
      }
      class MaxTwoOutgoingControlflowEdges extends ValidationRule {

        public MaxTwoOutgoingControlflowEdges(
            NESTNodeObject sourceNode,
            NESTNodeObject targetNode,
            NESTEdgeObject edge,
            boolean changingTarget) {
          super(
              sourceNode,
              targetNode,
              edge,
              "Only two outgoing controlflow edges for this source node allowed.",
              false,
              changingTarget);
        }

        @Override
        boolean isApplicable() {
          if (!edge.isNESTControlflowEdge() || !sourceNode.isNESTControlflowNode()) {
            return false;
          }
          NESTControlflowNodeObject controlflowNode = (NESTControlflowNodeObject) this.sourceNode;
          return controlflowNode.isAndStartNode()
              || controlflowNode.isXorStartNode()
              || controlflowNode.isLoopEndNode()
              || controlflowNode.isOrStartNode();
        }

        @Override
        boolean isViolated() {
          return sourceNode.getOutgoingEdges(NESTEdgeObject::isNESTControlflowEdge).size()
              - (changingTarget ? 1 : 0)
              >= 2;
        }
      }
      class MaxOneIngoingControlflowEdge extends ValidationRule {

        public MaxOneIngoingControlflowEdge(
            NESTNodeObject sourceNode,
            NESTNodeObject targetNode,
            NESTEdgeObject edge,
            boolean changingSource) {
          super(
              sourceNode,
              targetNode,
              edge,
              "Only one ingoing controlflow edge for this target node allowed.",
              changingSource,
              false);
        }

        @Override
        boolean isApplicable() {
          if (!edge.isNESTControlflowEdge()
              || !(targetNode.isNESTControlflowNode() || targetNode.isNESTTaskNode())) {
            return false;
          }
          if (targetNode.isNESTTaskNode()) {
            return true;
          }
          NESTControlflowNodeObject controlflowNode = (NESTControlflowNodeObject) this.targetNode;
          return controlflowNode.isAndStartNode()
              || controlflowNode.isXorStartNode()
              || controlflowNode.isLoopEndNode()
              || controlflowNode.isOrStartNode();
        }

        @Override
        boolean isViolated() {
          return targetNode.getIngoingEdges(NESTEdgeObject::isNESTControlflowEdge).size()
              - (changingSource ? 1 : 0)
              >= 1;
        }
      }
      class MaxTwoIngoingControlflowEdges extends ValidationRule {

        public MaxTwoIngoingControlflowEdges(
            NESTNodeObject sourceNode,
            NESTNodeObject targetNode,
            NESTEdgeObject edge,
            boolean changingSource) {
          super(
              sourceNode,
              targetNode,
              edge,
              "Only two ingoing controlflow edge for this target node allowed.",
              changingSource,
              false);
        }

        @Override
        boolean isApplicable() {
          if (!edge.isNESTControlflowEdge() || !targetNode.isNESTControlflowNode()) {
            return false;
          }
          NESTControlflowNodeObject controlflowNode = (NESTControlflowNodeObject) this.targetNode;
          return controlflowNode.isLoopStartNode()
              || controlflowNode.isXorEndNode()
              || controlflowNode.isAndEndNode()
              || controlflowNode.isOrEndNode();
        }

        @Override
        boolean isViolated() {
          return targetNode.getIngoingEdges(NESTEdgeObject::isNESTControlflowEdge).size()
              - (changingSource ? 1 : 0)
              >= 2;
        }
      }
      class NoDuplicateDataflowEdges extends ValidationRule {

        public NoDuplicateDataflowEdges(
            NESTNodeObject sourceNode, NESTNodeObject targetNode, NESTEdgeObject edge) {
          super(sourceNode, targetNode, edge, "No duplicate dataflow edges allowed.");
        }

        @Override
        boolean isApplicable() {
          return edge.isNESTDataflowEdge();
        }

        @Override
        boolean isViolated() {
          return sourceNode.getOutgoingEdges(NESTEdgeObject::isNESTDataflowEdge).stream()
              .anyMatch(edge -> edge.getPost() == targetNode);
        }
      }
      class NoDuplicateLoopReturnEdges extends ValidationRule {

        public NoDuplicateLoopReturnEdges(
            NESTNodeObject sourceNode, NESTNodeObject targetNode, NESTEdgeObject edge) {
          super(sourceNode, targetNode, edge, "No duplicate loop return edges allowed.");
        }

        @Override
        boolean isApplicable() {
          return edge.isNESTControlflowEdge()
              && sourceNode.isNESTControlflowNode()
              && ((NESTControlflowNodeObject) this.sourceNode).isLoopEndNode()
              && ((NESTControlflowNodeObject) this.sourceNode).getMatchingBlockControlflowNode()
              == targetNode;
        }

        @Override
        boolean isViolated() {
          return sourceNode.getOutgoingEdges(Utils::isEdgeLoopReturnEdge).size() >= 1;
        }
      }
      class NoDirectEndToStartControlflowEdges extends ValidationRule {

        public NoDirectEndToStartControlflowEdges(
            NESTNodeObject sourceNode, NESTNodeObject targetNode, NESTEdgeObject edge) {
          super(sourceNode, targetNode, edge, "Invalid edge.");
        }

        @Override
        boolean isApplicable() {
          return edge.isNESTControlflowEdge()
              && sourceNode.isNESTControlflowNode()
              && (((NESTControlflowNodeObject) this.sourceNode).isXorEndNode()
              || ((NESTControlflowNodeObject) this.sourceNode).isOrEndNode()
              || ((NESTControlflowNodeObject) this.sourceNode).isAndEndNode())
              && ((NESTControlflowNodeObject) this.sourceNode).getMatchingBlockControlflowNode()
              == targetNode;
        }

        @Override
        boolean isViolated() {
          return true;
        }
      }
      if (source != null
          && target != null
          && ((mxICell) source).getValue() instanceof NESTNodeObject
          && ((mxICell) target).getValue() instanceof NESTNodeObject) {
        NESTNodeObject sourceNode = (NESTNodeObject) ((mxICell) source).getValue();
        NESTNodeObject targetNode = (NESTNodeObject) ((mxICell) target).getValue();
        String sourceDataClassName = sourceNode.getDataClass().getName();
        String targetDataClassName = targetNode.getDataClass().getName();
        String validEdgeClassName = Utils.getEdgeClassForNodeConnection(sourceNode, targetNode);
        mxCell edgeCell =
            (mxCell)
                edge; // can be a new edge (id==null) or an existing edge whose endpoint shall be
        // reattached
        if (validEdgeClassName == null
            || (edgeCell.getId() != null
            && !((NESTEdgeObject) edgeCell.getValue())
            .getDataClass()
            .getName()
            .equals(validEdgeClassName))) {
          return "No suitable edge class for node connection of type "
              + sourceDataClassName
              + " -> "
              + targetDataClassName
              + " found.";
        }

        NESTEdgeObject edgeToCheck =
            edgeCell.getId() == null
                ? sourceNode.getGraph().getModel().createObject(validEdgeClassName)
                : (NESTEdgeObject) edgeCell.getValue();

        final boolean changingTarget =
            edgeCell.getTarget() != null && edgeCell.getTarget() != target;
        final boolean changingSource =
            edgeCell.getSource() != null && edgeCell.getSource() != source;
        Set<ValidationRule> rulesToCheck =
            new HashSet<>() {
              {
                add(
                    new MaxOneOutgoingPartOfEdge(
                        sourceNode, targetNode, edgeToCheck, changingTarget));
                add(
                    new MaxOneOutgoingControlflowEdge(
                        sourceNode, targetNode, edgeToCheck, changingTarget));
                add(
                    new MaxTwoOutgoingControlflowEdges(
                        sourceNode, targetNode, edgeToCheck, changingTarget));
                add(
                    new MaxOneIngoingControlflowEdge(
                        sourceNode, targetNode, edgeToCheck, changingSource));
                add(
                    new MaxTwoIngoingControlflowEdges(
                        sourceNode, targetNode, edgeToCheck, changingSource));
                add(new NoDuplicateDataflowEdges(sourceNode, targetNode, edgeToCheck));
                add(new NoDuplicateLoopReturnEdges(sourceNode, targetNode, edgeToCheck));
                add(new NoDirectEndToStartControlflowEdges(sourceNode, targetNode, edgeToCheck));
              }
            };
        Optional<ValidationRule> violatedRule =
            rulesToCheck.stream()
                .filter(ValidationRule::isApplicable)
                .filter(ValidationRule::isViolated)
                .findAny();
        if (violatedRule.isPresent()) {
          return "Rule violated: " + violatedRule.get().getConstraintDescription();
        }
      }
      return super.getEdgeValidationError(edge, source, target);
    }

    public mxRectangle getLabelSize(Object cell) {
      mxCellState state = this.getView().getState(cell);
      return mxUtils.getLabelSize(
          this.getLabel(cell),
          (state != null) ? state.getStyle() : this.getCellStyle(cell),
          this.isHtmlLabel(cell),
          1);
    }

    public NESTWorkflowLayout getLayout() {
      return layout;
    }

    public void selectGraphItemsOfClass(DataClass targetDataClass) {
      super.setSelectionCells(
          Arrays.stream(super.getChildCells(super.getDefaultParent()))
              .map(mxICell.class::cast)
              .filter(
                  cell -> {
                    if (!(cell.getValue() instanceof NESTGraphItemObject)) {
                      return false;
                    }
                    DataClass dataClass = ((NESTGraphItemObject) cell.getValue()).getDataClass();
                    return targetDataClass == dataClass || dataClass.isSubclassOf(targetDataClass);
                  })
              .collect(Collectors.toList()));
    }

    public NESTAbstractWorkflowObject getNestWorkflow() {
      return nestWorkflow;
    }

    public void loadNESTWorkflow(NESTAbstractWorkflowObject nestWorkflow) {
      this.nestWorkflow = nestWorkflow;
      this.layout.setNestWorkflow(nestWorkflow);
      mxICell parent = (mxICell) this.getDefaultParent();
      this.setCellId(
          parent,
          "7c7bc177-cfc7-45fb-9d1b-5b29ebfb8570"); // prevent id conflict, mxGraph root cell always
      // has id "1" which could already be in use in
      // the NESTGraph
      //            this.setAutoSizeCells(true); // adapt cell size to label size
      //            this.getModel().beginUpdate();
      try {
        this.setEventsEnabled(false);
        nestWorkflow.getGraphNodes().forEach(node -> {
          mxICell insertedCell = (mxICell) this.insertVertex(parent, node.getId(), node, 80, 30,
              this.layout.getLayoutConfig().getNodeWidth(),
              this.layout.getLayoutConfig().getNodeHeight(),
              CellStyle.get(node));
          if (!node.isNESTControlflowNode()) {
            this.updateCellSize(insertedCell);
          }
        });
        this.loadNESTEdges();
      } finally {
        this.setEventsEnabled(true);
      }
      this.executeLayout();
    }

    private void loadNESTEdges() {
      Set<NESTEdgeObject> edges = this.nestWorkflow.getGraphEdges();
      Map<String, Object> graphCells = ((mxGraphModel) this.getModel()).getCells();
      List<Object> insertedEdges = new LinkedList();
      for (NESTEdgeObject edge : edges) {
        mxICell parentCell = (mxICell) this.getDefaultParent();
        mxICell tempConnectVertex =
            (mxICell)
                this.insertVertex(
                    parentCell,
                    "temp",
                    "null",
                    0,
                    0,
                    0,
                    0); // edges with a missing endpoint are connected to this vertex so they are
        // drawn
        insertedEdges.add(
            this.insertEdge(
                parentCell,
                edge.getId(),
                edge,
                edge.getPre() == null ? tempConnectVertex : graphCells.get(edge.getPre().getId()),
                edge.getPost() == null ? tempConnectVertex : graphCells.get(edge.getPost().getId()),
                CellStyle.get(edge)
                    + (edge.getPre() == null || edge.getPost() == null ? ";strokeColor=red" : "")));
        this.removeCells(new Object[]{tempConnectVertex}, false);
      }
      this.orderCells(true, insertedEdges.toArray());
      List<Object> insertedPartOfEdges =
          insertedEdges.stream()
              .filter(edge -> ((NESTEdgeObject) ((mxICell) edge).getValue()).isNESTPartOfEdge())
              .collect(Collectors.toList());
      this.orderCells(true, insertedPartOfEdges.toArray());
    }

    private Set<NESTSubWorkflowNodeObject> getTopLevelSubworkflowNodes(
        Set<NESTSubWorkflowNodeObject> subworkflowNodes) {
      return subworkflowNodes.stream().filter(node -> node.getOutgoingEdges().stream()
              .noneMatch(edge -> edge.isNESTPartOfEdge() && edge.getPost().isNESTSubWorkflowNode()))
          .collect(Collectors.toSet());
    }

    private void groupSubworkflows() {
      var topLevelSubWorkflows = this
          .getTopLevelSubworkflowNodes(this.nestWorkflow.getSubWorkflowNodes());
      topLevelSubWorkflows.forEach(subworkflowNode -> this
          .groupSubworkflow(subworkflowNode, (mxICell) this.getDefaultParent()));
    }

    private mxICell groupSubworkflow(NESTSubWorkflowNodeObject subWorkflowNode, mxICell parent) {
      Map<String, Object> graphCells = ((mxGraphModel) this.getModel()).getCells();
      // create a container cell for the elements of the subworkflow
      mxICell groupingCell =
          (mxICell)
              this.insertVertex(
                  parent, subWorkflowNode.getId() + "_GROUPING_CELL", null, 0, 0, 80, 80);
      // look for nested child subworkflows
      Set<NESTSubWorkflowNodeObject> childSubWorkflowNodes =
          subWorkflowNode.getIngoingEdges().stream()
              .filter(edge -> edge.isNESTPartOfEdge() && edge.getPre().isNESTSubWorkflowNode())
              .map(NESTEdgeObject::getPre)
              .map(NESTSubWorkflowNodeObject.class::cast)
              .collect(Collectors.toSet());
      // group the child subworkflows first
      childSubWorkflowNodes.forEach(
          childSubWorkflowNode -> {
            mxICell childSubWorkflowGroupingCell =
                this.groupSubworkflow(childSubWorkflowNode, groupingCell);
            // add grouped child subwf to parent grouping cell
            this.groupCells(groupingCell, 20, new Object[]{childSubWorkflowGroupingCell});
          });
      // look for non-subworkflow nodes that are part of the subWorkflow
      Set<NESTNodeObject> childNodes =
          subWorkflowNode.getIngoingEdges().stream()
              .filter(edge -> edge.isNESTPartOfEdge() && !edge.getPre().isNESTSubWorkflowNode())
              .map(NESTEdgeObject::getPre)
              .collect(Collectors.toSet());
      // change parent of non-subworkflow nodes to the container cell
      childNodes.forEach(
          childNode -> {
            mxICell childNodeCell = (mxICell) graphCells.get(childNode.getId());
            this.groupCells(groupingCell, 200, new Object[]{childNodeCell});
          });
      // add the subworkflow cell itself to the grouping cell
      mxICell subWorkflowNodeCell = (mxICell) graphCells.get(subWorkflowNode.getId());
      this.groupCells(groupingCell, 200, new Object[]{subWorkflowNodeCell});

      return groupingCell;
    }

    private void executeLayout(mxICell parent) {
      if (layout.isApplicable()) {
        layout.execute();
      } else {
        mxHierarchicalLayout layout = new mxHierarchicalLayout(this, SwingConstants.WEST);
        layout.setInterRankCellSpacing(30);
        layout.setIntraCellSpacing(30);
        layout.setParallelEdgeSpacing(15);
        layout.setFineTuning(true);
        layout.setDisableEdgeStyle(true);
        layout.execute(parent);
      }
      this.refresh();
    }

    public void executeLayout() {
      executeLayout((mxICell) this.getDefaultParent());
    }

    /**
     * Inserts the given nestEdge into the graph view when no cell with the id of the given nestEdge
     * is present in the mxGraph.
     *
     * @param nestEdge The {@link NESTEdgeObject} which should be synced to the graph view
     * @return
     */
    public mxICell syncEdge(NESTEdgeObject nestEdge) {
      mxGraphModel graphModel = (mxGraphModel) this.getModel();
      mxICell preCell = (mxICell) graphModel.getCell(nestEdge.getPre().getId());
      mxICell postCell = (mxICell) graphModel.getCell(nestEdge.getPost().getId());
      mxICell edgeCell = (mxICell) graphModel.getCell(nestEdge.getId());
      if (edgeCell == null) {
        edgeCell =
            (mxICell)
                this.insertEdge(
                    this.getDefaultParent(),
                    nestEdge.getId(),
                    nestEdge,
                    preCell,
                    postCell,
                    CellStyle.get(nestEdge));
      }
      return edgeCell;
    }

    /**
     * Inserts the outgoing edges (only when its id is missing in the mxGraph) of the given nestNode
     * into the graph view.
     *
     * @param nestNode
     */
    public void syncOutgoingEdges(NESTNodeObject nestNode) {
      nestNode.getOutgoingEdges().forEach(this::syncEdge);
    }

    /**
     * Sets the cells id to newId and updates the mapping of id to cell in the graph model by
     * removing the cell with the old id and adding it with the new id.. Is necessary for
     * model.getCell(id) to work later on (returns null when not updating the map).
     *
     * @param cell
     * @param newId
     */
    public void setCellId(mxICell cell, String newId) {
      Map<String, Object> cells = ((mxGraphModel) this.getModel()).getCells();
      cells.remove(cell.getId());
      cell.setId(newId);
      cells.put(newId, cell);
    }

    public mxICell getCellById(String id) {
      return (mxICell) ((mxGraphModel) this.getModel()).getCells().get(id);
    }

    /**
     * clears the graph of all cells with event firing disabled so the underlying NESTWorkflow is
     * not modified
     */
    public void clearWithoutEvents() {
      boolean eventsEnabled = this.isEventsEnabled();
      this.setEventsEnabled(false);
      Object[] cells =
          ((mxGraphModel) this.getModel())
              .getCells().values()
              .stream() // only model.getCells includes previously hidden cells. Downcast from
              // interface to implementation necessary to access the method
              .filter(
                  cell ->
                      cell != this.getDefaultParent()
                          && cell
                          != ((mxICell) this.getDefaultParent())
                          .getParent()) // do not remove the root nodes
              .map(mxICell.class::cast)
              .toArray(Object[]::new);
      this.removeCells(cells);
      this.setEventsEnabled(eventsEnabled);
    }

    @Override
    public mxRectangle getPreferredSizeForCell(Object cell) {
      // TODO: relocate constants or integrate into xml style
      final int MIN_WIDTH = 50;
      final int MIN_HEIGHT = 50;

      // Unclean "Hack" to not make Cells snap to grid after editing
      boolean previousGridEnabled = gridEnabled;

      gridEnabled = false;
      Object cellValue = ((mxCell) cell).getValue();
      // Don't remove next line

      if (cellValue instanceof NESTGraphItemObject && false) {
        return ((mxCell) cell).getGeometry();
      }

      mxRectangle result = super.getPreferredSizeForCell(cell);
      gridEnabled = previousGridEnabled;

      if (result != null) {
        result.setWidth(result.getWidth() < MIN_WIDTH ? MIN_WIDTH : result.getWidth());
        result.setHeight(result.getHeight() < MIN_HEIGHT ? MIN_HEIGHT : result.getHeight());
      }
      return result;
    }

    /**
     * restricts edge splitting to controlflow edges and sequence nodes / blocks also affects the
     * "dragging preview": only valid combinations of target edges and dragged cells are
     * highlighted
     *
     * @param target potential drop edge
     * @param cells  that are dragged
     * @return whether cells can be dropped on target
     */
    @Override
    public boolean isSplitTarget(Object target, Object[] cells) {
      return super.isSplitTarget(target, cells)
          && ((mxICell) target).getValue() instanceof NESTControlflowEdgeObject
          // only allow splitting of controlflow edges
          && !Utils.isEdgeLoopReturnEdge((NESTEdgeObject) ((mxICell) target).getValue())
          && (
          (((mxICell) cells[0]).getValue() instanceof NodeInsertType && Stream.of(
                  NodeInsertType.TASK,
                  NodeInsertType.AND_BLOCK,
                  NodeInsertType.XOR_BLOCK,
                  NodeInsertType.LOOP_BLOCK)
              .anyMatch(type -> type == ((mxICell) cells[0]).getValue()))
              // only allow sequence nodes(/blocks) from the palette to split an edge
              || ((mxICell) cells[0])
              .getValue() instanceof NESTTaskNodeObject); // or an already existing task
    }

    public CellLabelGenerator getCellLabelGenerator() {
      return cellLabelGenerator;
    }

  }

  public static class NESTWorkflowValidatorGUI extends JDialog {

    protected NESTWorkflowEditor editor;

    protected JTextArea errorLogArea = new JTextArea();
    protected JLabel validationResultLabel = new JLabel();

    public NESTWorkflowValidatorGUI(NESTWorkflowEditor editor) {
      super(editor.getEditorFrame(), "NESTWorkflow Validator");

      this.editor = editor;

      this.setDefaultCloseOperation(HIDE_ON_CLOSE);
      this.setLayout(new BorderLayout());

      JPanel validationResultPanel = new JPanel();
      validationResultPanel.setLayout(new BasicGridLayout(1, 2, 5, 5, 2, 10));
      JButton validateButton = new JButton("Validate:");
      validateButton.addActionListener(e -> validateNESTWorkflow());
      validationResultPanel.add(validateButton);
      validationResultPanel.add(validationResultLabel);
      this.add(validationResultPanel, BorderLayout.NORTH);

      JScrollPane scrollPane = new JScrollPane(errorLogArea);
      this.add(scrollPane, BorderLayout.CENTER);

      this.setPreferredSize(new Dimension(500, 200));
      this.pack();
    }

    public boolean validateNESTWorkflow() {
      if (editor.getNESTWorkflow().isNESTSequentialWorkflow()) {
        boolean isValid = new NESTSequentialWorkflowValidatorImpl(
            editor.getNESTWorkflow()).isValidSequentialWorkflow();

        if (!isValid) {
          errorLogArea.setText("INVALID sequential workflow");
        }

        validationResultLabel.setText("<html>"
            + (isValid ? "<font color='green'><b>VALID sequential workflow</b></font>"
            : "<font color='red'><b>INVALID sequential workflow</b></font>")
            + "</html>");

        return isValid;
      } else if (editor.getNESTWorkflow().isNESTWorkflow()) {
        NESTWorkflowValidatorImpl validator = new NESTWorkflowValidatorImpl(
            (NESTWorkflowObject) editor.getNESTWorkflow());
        validator.isBlockOrientedWorkflow();

        errorLogArea.setText(validator.getErrorMessage());
        validationResultLabel.setText("<html>"
            + (validator.isValidGraph() ? "<font color='green'><b>VALID graph</b></font>"
            : "<font color='red'><b>INVALID graph</b></font>")
            + (validator.isValidWorkflow() ? ", <font color='green'><b>VALID workflow</b></font>"
            : ", <font color='red'><b>INVALID workflow</b></font>")
            + (validator.isBlockOrientedWorkflow()
            ? ", <font color='green'><b>VALID block-oriented workflow</b></font>"
            : ", <font color='red'><b>INVALID block-oriented workflow</b></font>")
            + "</html>");

        return validator.isValidWorkflow();
      }

      return false;

    }
  }

  /**
   * A custom {@link mxVertexHandler} where Nodes can be resized without snapping to Grid
   */
  private static class CustomVertexHandler extends mxVertexHandler {

    /**
     * Constructor for a custom {@link mxVertexHandler} so Nodes can be resized without snapping to
     * Grid
     *
     * @param graphComponent Standard mxGraphComponent for this context
     * @param state          Standard mxCellState for this context
     */
    public CustomVertexHandler(mxGraphComponent graphComponent, mxCellState state) {
      super(graphComponent, state);
    }

    /**
     * draws preview resize without snapping to grid
     *
     * @param e standard Swing MouseEvent for this context
     */
    @Override
    public void mouseDragged(MouseEvent e) {
      if (!e.isConsumed() && first != null) {
        gridEnabledEvent = graphComponent.isGridEnabledEvent(e);
        constrainedEvent = graphComponent.isConstrainedEvent(e);

        double dx = e.getX() - first.x;
        double dy = e.getY() - first.y;

        if (isLabel(index)) {
          mxPoint pt = new mxPoint(e.getPoint());

          int idx = (int) Math.round(pt.getX() - first.x);
          int idy = (int) Math.round(pt.getY() - first.y);

          if (constrainedEvent) {
            if (Math.abs(idx) > Math.abs(idy)) {
              idy = 0;
            } else {
              idx = 0;
            }
          }

          Rectangle rect = state.getLabelBounds().getRectangle();
          rect.translate(idx, idy);
          preview.setBounds(rect);
        } else {
          mxGraph graph = graphComponent.getGraph();
          double scale = graph.getView().getScale();

          mxRectangle bounds = union(getState(), dx, dy, index);
          bounds.setWidth(bounds.getWidth() + 1);
          bounds.setHeight(bounds.getHeight() + 1);
          preview.setBounds(bounds.getRectangle());
        }

        if (!preview.isVisible() && graphComponent.isSignificant(dx, dy)) {
          preview.setVisible(true);
        }

        e.consume();
      }
    }

    /**
     * resizes Cell without snapping to Grid
     *
     * @param e standard Swing MouseEvent for this context
     */
    @Override
    protected void resizeCell(MouseEvent e) {
      mxGraph graph = graphComponent.getGraph();
      double scale = graph.getView().getScale();

      Object cell = state.getCell();
      mxGeometry geometry = graph.getModel().getGeometry(cell);

      if (geometry != null) {
        double dx = (e.getX() - first.x) / scale;
        double dy = (e.getY() - first.y) / scale;

        if (isLabel(index)) {
          geometry = (mxGeometry) geometry.clone();

          if (geometry.getOffset() != null) {
            dx += geometry.getOffset().getX();
            dy += geometry.getOffset().getY();
          }

          geometry.setOffset(new mxPoint(dx, dy));
          graph.getModel().setGeometry(cell, geometry);
        } else {
          mxRectangle bounds = union(geometry, dx, dy, index);
          Rectangle rect = bounds.getRectangle();

          graph.resizeCell(cell, new mxRectangle(rect));
        }
      }
    }
  }

}

