package de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor;

import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.object.base.AggregateObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTGraphItemObject;
import de.uni_trier.wi2.procake.gui.objecteditor.label.AggregateObjectLabelProvider;
import de.uni_trier.wi2.procake.gui.objecteditor.label.LabelProvider;
import de.uni_trier.wi2.procake.gui.objecteditor.label.LabelProviderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;

public class CellLabelGenerator {

  private final Logger logger = LoggerFactory.getLogger(CellLabelGenerator.class);

  private final Map<String, Queue<String>> dataClassNameToLabelPathMapping = new HashMap<>();
  private final Map<NESTGraphItemObject, LabelMode> graphItemToLabelModeMapping = new HashMap<>();
  private boolean showIds = false;
  private boolean showEdgeLabels = true;

  /**
   * @param nestGraphItemObject to generate a label for
   * @return label generated using a corresponding label provider
   */
  @SuppressWarnings("DataFlowIssue")
  public String getLabelFor(NESTGraphItemObject nestGraphItemObject) {

    StringBuilder label = new StringBuilder(
        this.showIds ? "ID: " + nestGraphItemObject.getId() + "\n" : "");

    boolean generatedLabel = false;

    DataObject semanticDescriptor = nestGraphItemObject.getSemanticDescriptor();
    if (semanticDescriptor != null && semanticDescriptor.isAggregate()) {

      LabelProvider labelProvider = LabelProviderFactory.getProvider(
          semanticDescriptor.getDataClass());
      if (labelProvider == null) {
        logger.warn("No label provider configured in composition for data class \"{}\"",
            semanticDescriptor.getDataClass().getName());
      } else {

        LabelMode labelMode = graphItemToLabelModeMapping.get(
            nestGraphItemObject);
        if (!(labelProvider instanceof AggregateObjectLabelProvider)) {
          label.append(
              Objects.requireNonNullElse(labelProvider.getLabel(nestGraphItemObject),
                  ""));
          generatedLabel = true;
        } else if (labelMode == LabelMode.FULL_SEMANTIC_DESCRIPTOR
            || labelMode == LabelMode.FULL_SEMANTIC_DESCRIPTOR_EXCLUDE_NULL) {

          AggregateObjectLabelProvider aggregateObjectLabelProvider = (AggregateObjectLabelProvider) labelProvider;
          aggregateObjectLabelProvider.setIncludeNullAttributes(
              labelMode != LabelMode.FULL_SEMANTIC_DESCRIPTOR_EXCLUDE_NULL);
          aggregateObjectLabelProvider.setIncludeAllAttributes(true);

          label.append(aggregateObjectLabelProvider.getLabel(semanticDescriptor));
          generatedLabel = true;

          //reset temporary style settings
          aggregateObjectLabelProvider.setIncludeNullAttributes(false);
          aggregateObjectLabelProvider.setIncludeAllAttributes(false);

        } else {
          var labelPath =
              dataClassNameToLabelPathMapping.get(semanticDescriptor.getDataClass().getName());
          if (labelPath != null) {
            LinkedList<String> copy = new LinkedList<>(labelPath); // clone
            DataObject labelObject =
                this.resolveAttributePath((AggregateObject) semanticDescriptor, copy);
            if (labelObject != null) {
              label.append(LabelProviderFactory.getLabel(labelObject));
              generatedLabel = true;
            } else {
              label.append("null");
              generatedLabel = true;
            }
          }
        }
      }
    }

    if (!generatedLabel) {
      label.append(LabelProviderFactory.getLabel(nestGraphItemObject));
    }

    if (!nestGraphItemObject.isNESTEdge()
        || (nestGraphItemObject.isNESTEdge() && showEdgeLabels)) {
      return label.toString();
    }

    return "";
  }

  private DataObject resolveAttributePath(
      AggregateObject aggregateObject, Queue<String> attributePath) {
    DataObject attributeValue = aggregateObject.getAttributeValue(attributePath.remove());
    if (attributePath.size() == 0
        || attributeValue == null
        || attributeValue
        .isVoid()) { // path is now empty, goal element reached OR element could not be
      // reached due to intermediary attributes being null/void
      return attributeValue;
    } else {
      return this.resolveAttributePath((AggregateObject) attributeValue, attributePath);
    }
  }

  public Map<String, Queue<String>> getDataClassNameToLabelPathMapping() {
    return dataClassNameToLabelPathMapping;
  }

  public boolean isShowIds() {
    return showIds;
  }

  public void setShowIds(boolean showIds) {
    this.showIds = showIds;
  }

  public boolean isShowEdgeLabels() {
    return showEdgeLabels;
  }

  public void setShowEdgeLabels(boolean showEdgeLabels) {
    this.showEdgeLabels = showEdgeLabels;
  }

  public void setLabelModeForItem(NESTGraphItemObject graphItemObject,
      LabelMode mode) {
    this.graphItemToLabelModeMapping.put(graphItemObject, mode);
  }
}
