package de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor;

import com.mxgraph.model.mxICell;
import com.mxgraph.util.mxEventObject;
import com.mxgraph.util.mxEventSource;
import de.uni_trier.wi2.procake.data.object.nest.NESTAbstractWorkflowObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTEdgeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTGraphItemObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTWorkflowObject;
import de.uni_trier.wi2.procake.data.object.nest.utils.NESTAbstractWorkflowModifier;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.swing.*;


public class CellAddListener implements mxEventSource.mxIEventListener {

  @Override
  public void invoke(Object sender, mxEventObject mxEventObject) {
    List<mxICell> newCells =
        Arrays.stream(((Object[]) mxEventObject.getProperty("cells")))
            .filter(Objects::nonNull)
            .map(mxICell.class::cast)
            .filter(
                cell ->
                    cell.getValue() instanceof NodeInsertType
                        || cell.getValue() instanceof NESTGraphItemObject)
            .collect(Collectors.toList());
    if (newCells.size() <= 0) {
      return;
    }

    NESTWorkflowEditor.CustomGraph customGraph = (NESTWorkflowEditor.CustomGraph) sender;
    customGraph.setEventsEnabled(
        false); // doInsertion itself can add new cells to the mxGraph. As we only wish to monitor
    // additions done from the GUI by the user, the now following  additions shall not
    // cause events.
    try {
      if (newCells.size() == 1
          && newCells.get(0).getValue() instanceof NodeInsertType) { // palette drop
        this.handlePaletteDrop(customGraph, newCells.get(0));
      } else if (newCells.size() == 1
          && ((NESTGraphItemObject) newCells.get(0).getValue())
          .getId()
          .equals(
              newCells
                  .get(0)
                  .getId())) { // NESTGraphItemObject ID == mxCell ID ->  existing node was
        // dropped on an edge
        this.handleEdgeSplitDrop(customGraph, newCells);
      } else {
        this.handleCopying(customGraph, newCells);
      }
    } finally {
      customGraph.orderCells(
          true,
          customGraph.getNestWorkflow().getGraphEdges().stream()
              .filter(NESTEdgeObject::isNESTPartOfEdge)
              .map(edge -> customGraph.getCellById(edge.getId()))
              .toArray(mxICell[]::new));
      customGraph.setEventsEnabled(true);
    }
  }

  private void handleCopying(NESTWorkflowEditor.CustomGraph customGraph, List<mxICell> newCells) {
    // cells were copied (ids should differ, mxGraph generates a new id on copy)
    this.handleCopiedWorkflowNode(
        customGraph,
        newCells); // remove potential copy of wf node if there already exists one in the graph

    // mxGraph does not reliably provide all the copied edge cells so we delete these which were
    // inserted.
    // Only provided node cells are processed which then will manually be connected with edges, if
    // necessary.
    // That means a particular selection of edges by the user is ignored by the copy process.
    newCells.stream().filter(mxICell::isEdge).forEach(edge -> customGraph.getModel().remove(edge));

    NESTAbstractWorkflowObject nestWorkflow = customGraph.getNestWorkflow();
    NESTAbstractWorkflowModifier nestWorkflowModifier = nestWorkflow.getModifier();
    Set<mxICell> newNodeCells =
        newCells.stream()
            .filter(cell -> ((NESTGraphItemObject) cell.getValue()).isNESTNode())
            .collect(Collectors.toSet());
    Set<NESTNodeObject> nestNodesToCopy =
        newNodeCells.stream()
            .map(mxICell::getValue)
            .map(NESTNodeObject.class::cast)
            .collect(Collectors.toSet());
    NESTWorkflowObject partialGraph =
        (NESTWorkflowObject) nestWorkflowModifier.extractPartialGraph(nestNodesToCopy);
    Map<String, String> oldToNewIDMapping =
        nestWorkflowModifier.insertSubgraph(partialGraph, null, null, null);
    Set<NESTNodeObject> copiedNESTNodes =
        newNodeCells.stream()
            .map(
                cell -> {
                  NESTNodeObject nestNode = (NESTNodeObject) cell.getValue();
                  String copiedNESTGraphItemId = oldToNewIDMapping.get(nestNode.getId());
                  NESTNodeObject copiedNESTNode = nestWorkflow.getGraphNode(copiedNESTGraphItemId);
                  customGraph.setCellId(cell, copiedNESTGraphItemId);
                  cell.setValue(copiedNESTNode);
                  return copiedNESTNode;
                })
            .collect(Collectors.toSet());
    copiedNESTNodes.forEach(customGraph::syncOutgoingEdges);
  }

  private void handleCopiedWorkflowNode(
      NESTWorkflowEditor.CustomGraph customGraph, List<mxICell> newCells) {
    mxICell workflowNodeCell =
        newCells.stream()
            .filter(cell -> ((NESTGraphItemObject) cell.getValue()).isNESTWorkflowNode())
            .findAny()
            .orElse(null);
    if (workflowNodeCell != null && customGraph.getNestWorkflow().getWorkflowNode() != null) {
      Set<mxICell> connectedEdges =
          IntStream.range(0, workflowNodeCell.getEdgeCount())
              .mapToObj(workflowNodeCell::getEdgeAt)
              .collect(Collectors.toSet());
      newCells.removeAll(connectedEdges);
      customGraph.getModel().remove(workflowNodeCell);
      this.showWorkflowNodeAlreadyExistsDialog();
    }
  }

  private void handleEdgeSplitDrop(
      NESTWorkflowEditor.CustomGraph customGraph, List<mxICell> newCells) {
    NESTNodeObject node =
        customGraph
            .getNestWorkflow()
            .getGraphNode(((NESTGraphItemObject) newCells.get(0).getValue()).getId());
    customGraph.setEventsEnabled(true);
    // remove the controlflow edges of the node as it will be assigned new ones due to the insertion
    // into the edge split
    customGraph.removeCells(
        node.getEdges(NESTEdgeObject::isNESTControlflowEdge).stream()
            .map(edge -> customGraph.getCellById(edge.getId()))
            .toArray(mxICell[]::new));
  }

  private void handlePaletteDrop(NESTWorkflowEditor.CustomGraph customGraph, mxICell newCell) {
    NodeInsertType insertType = (NodeInsertType) newCell.getValue();
    if (insertType == NodeInsertType.WORKFLOW
        && customGraph.getNestWorkflow().getWorkflowNode()
        != null) { // prevent multiple workflow nodes
      customGraph.getModel().remove(newCell);
      this.showWorkflowNodeAlreadyExistsDialog();
    } else {
      newCell.setValue(insertType.doInsertion(customGraph.getNestWorkflow(), customGraph, newCell));
    }
  }

  private void showWorkflowNodeAlreadyExistsDialog() {
    JOptionPane.showMessageDialog(null, "Only one workflow node is allowed.");
  }
}
