package de.uni_trier.wi2.procake.gui.objecteditor;

import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.object.base.AggregateObject;
import de.uni_trier.wi2.procake.data.object.base.CollectionObject;
import de.uni_trier.wi2.procake.data.object.base.ListObject;
import de.uni_trier.wi2.procake.data.object.base.SetObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTGraphItemObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTWorkflowObject;
import de.uni_trier.wi2.procake.data.objectpool.ReadableObjectPool;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Vector;
import java.util.stream.Collectors;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

public class ObjectEditorTreeModel implements TreeModel {

  DataObject dataObject;
  Vector<TreeModelListener> listener;

  public ObjectEditorTreeModel(DataObject dataObject) {
    this.dataObject = dataObject;
    this.listener = new Vector<>();
  }

  @Override
  public Object getRoot() {
    if (dataObject.isAtomic()) {
      return dataObject;
    }
    return dataObject;
  }

  @Override
  public Object getChild(Object o, int i) {
    if (o instanceof DataObject) {
      DataObject dataObject = (DataObject) o;
      if (dataObject.isAggregate()) {
        AggregateObject aggregateObject = (AggregateObject) dataObject;
        List<String> attributeNames =
            new ArrayList<>(aggregateObject.getAggregateClass().getAttributeNames());
        Collections.sort(attributeNames);
        return new ImmutablePair<>(
            attributeNames.get(i), aggregateObject.getAttributeValue(attributeNames.get(i)));
      } else if (dataObject.isList()) {
        return ((ListObject) dataObject).elementAt(i);
      } else if (dataObject.isSet()) {
        SetObject setObject = (SetObject) dataObject;
        ArrayList<DataObject> collectionList = new ArrayList<>(setObject.size());
        setObject
            .iterator()
            .forEachRemaining(setElement -> collectionList.add((DataObject) setElement));
        return collectionList.stream()
            .sorted(new ObjectPoolTreeModel.DataObjectComparator<>())
            .collect(Collectors.toList())
            .get(i);
      } else if (dataObject.isNESTEdge() || dataObject.isNESTNode()) {
        return this.getChild(((NESTGraphItemObject) dataObject).getSemanticDescriptor(), i);
      } else if (dataObject.isNESTWorkflow()) {
        switch (i) {
          case 0:
            Set<NESTNodeObject> graphNodes = ((NESTWorkflowObject) dataObject).getGraphNodes();
            return graphNodes.size() == 0
                ? "Nodes"
                : new HashSet<>(graphNodes) { // fix for empty node in tree view
                  @Override
                  public String toString() {
                    return "Nodes";
                  }
                };
          case 1:
            return new HashSet<>(((NESTWorkflowObject) dataObject).getGraphEdges()) {
              @Override
              public String toString() {
                return "Edges";
              }
            };
        }
        return i;
      }
    } else if (o instanceof Pair) { // aggregate attribute
      return this.getChild(((Pair<String, DataObject>) o).getValue(), i);
    } else if (o instanceof Set) { // nest edges or nodes
      return ((Set<NESTGraphItemObject>) o)
          .stream()
          .sorted(
              Comparator.comparing(
                      graphItem -> ((NESTGraphItemObject) graphItem).getDataClass().getName())
                  .thenComparing(graphItem -> ((NESTGraphItemObject) graphItem).getId()))
          .collect(Collectors.toList())
          .get(i);
    } else if (o instanceof ReadableObjectPool) {
      return new ArrayList<DataObject>(((ReadableObjectPool) o).getCollection())
          .stream().sorted(new ObjectPoolTreeModel.DataObjectComparator<>()).collect(Collectors.toList()).get(i);
    }
    return null;
  }

  @Override
  public int getChildCount(Object o) {
    if (o instanceof DataObject) {
      DataObject dataObject = (DataObject) o;
      if (dataObject.isAggregate()) {
        return ((AggregateObject) dataObject).getAggregateClass().getAttributeNames().size();
      } else if (dataObject.isCollection()) {
        return ((CollectionObject) dataObject).size();
      } else if (dataObject.isNESTEdge() || dataObject.isNESTNode()) {
        return this.getChildCount(((NESTGraphItemObject) dataObject).getSemanticDescriptor());
      } else if (dataObject.isNESTWorkflow()) {
        return 2; // edges and nodes
      }
    } else if (o instanceof Pair) {
      return this.getChildCount(((Pair<String, DataObject>) o).getValue());
    } else if (o instanceof Set) { // NESTGraph nodes or edges
      return ((Set) o).size();
    } else if (o instanceof ReadableObjectPool) {
      return ((ReadableObjectPool<DataObject>) o).size();
    }
    return 0;
  }

  @Override
  public boolean isLeaf(Object o) {
    if (o instanceof DataObject) {
      DataObject dataObject = (DataObject) o;
      if (dataObject.isAggregate()) {
        return ((AggregateObject) dataObject).getAggregateClass().getAttributeNames().size() <= 0;
      } else if (dataObject.isCollection()) {
        return ((CollectionObject) dataObject).size() <= 0;
      } else if (dataObject.isNESTEdge() || dataObject.isNESTNode()) {
        return this.getChildCount(((NESTGraphItemObject) dataObject).getSemanticDescriptor()) <= 0;
      } else if (dataObject.isNESTWorkflow()) {
        return false;
      }
    } else if (o instanceof Pair) {
      return this.isLeaf(((Pair<String, DataObject>) o).getValue());
    } else if (o instanceof Set) // NESTGraph nodes or edges
    {
      return ((Set) o).size() <= 0;
    } else if (o instanceof ReadableObjectPool) {
      return ((ReadableObjectPool) o).size() <= 0;
    }

    return true;
  }

  @Override
  public void valueForPathChanged(TreePath path, Object o) {
    System.out.println("valueForPathChanged: " + path + " -> " + o);
  }

  @Override
  public int getIndexOfChild(Object parent, Object child) {
    int max = this.getChildCount(parent);
    for (int i = 0; i < max; i++) {
      if (this.getChild(parent, i).equals(child)) {
        return i;
      }
    }
    return -1;
  }

  @Override
  public void addTreeModelListener(TreeModelListener l) {
    listener.addElement(l);
  }

  @Override
  public void removeTreeModelListener(TreeModelListener l) {
    listener.removeElement(l);
  }

  public void fireTreeStructureChanged(Object[] path) {
    TreeModelEvent e = new TreeModelEvent(this, path);
    listener.forEach(listener -> listener.treeStructureChanged(e));
  }
}
