package de.uni_trier.wi2.procake.gui.objecteditor;

import static de.uni_trier.wi2.procake.gui.Constants.NESTWORKFLOW_RESOURCES;

import com.mxgraph.util.mxResources;
import de.uni_trier.wi2.procake.data.io.xml.xerces_saxImpl.ObjectPoolReader;
import de.uni_trier.wi2.procake.data.io.xml.xerces_saxImpl.ObjectReader;
import de.uni_trier.wi2.procake.data.model.DataClass;
import de.uni_trier.wi2.procake.data.model.ModelFactory;
import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.object.base.AggregateObject;
import de.uni_trier.wi2.procake.data.object.base.CollectionObject;
import de.uni_trier.wi2.procake.data.object.base.ListObject;
import de.uni_trier.wi2.procake.data.object.base.SetObject;
import de.uni_trier.wi2.procake.data.object.base.StringObject;
import de.uni_trier.wi2.procake.data.objectpool.WriteableObjectPool;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.editor.BasicGraphEditor;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.editor.DefaultFileFilter;
import de.uni_trier.wi2.procake.utils.exception.ProCAKEInvalidTypeException;
import de.uni_trier.wi2.procake.utils.io.IOUtil;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.function.Supplier;
import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import javax.swing.tree.TreePath;

/**
 * Class containing many {@link AbstractAction}s to use in the context of {@link ObjectEditor}
 */
public class ObjectEditorActions {

  /**
   * {@link AbstractAction} for resetting the {@link ObjectEditor}s working object to the original
   * object
   */
  public static class RefreshObjectPoolAction extends AbstractAction {

    private ObjectEditor editor;

    public RefreshObjectPoolAction(ObjectEditor editor) {
      super("Load from Object");
      putValue(SHORT_DESCRIPTION,
          "<html><p width='300'>Copies the Object from the reference given on instantiation of the editor and loads this copy into the editor. Be aware that any unsaved changes made to the pool will be lost.</p></html>");
      this.editor = editor;
    }

    @Override
    public void actionPerformed(ActionEvent actionEvent) {
      JPanel barPanel = new JPanel();
      barPanel.setLayout(new GridBagLayout());
      JLabel text = new JLabel("Loading... ");
      JProgressBar bar = new JProgressBar();
      bar.setVisible(true);
      bar.setIndeterminate(true);
      barPanel.add(text);
      barPanel.add(bar);
      editor.setNewMainSplitPane(new JPanel(), barPanel);
      SwingUtilities.invokeLater(() -> {
        boolean useDataObjectMode = false;
        if (editor.getDataObject() != null && editor.getPool() == null) {
          useDataObjectMode = true;
          editor.loadObject(editor.getOriginalDataObject());
        } else if (editor.getDataObject() == null && editor.getPool() != null) {
          editor.loadObject(editor.getOriginalPool());
        }

        if (useDataObjectMode) {
          editor.loadObject(editor.getOriginalDataObject());
        } else {
          editor.loadObject(editor.getOriginalPool());
        }

        editor.setTree(editor.setUpTree());
        JPanel newLeftPane = editor.setUpLeftPane();
        JPanel newRightPane = editor.setUpRightPane();
        editor.setNewMainSplitPane(newLeftPane, newRightPane);

        editor.updateTreeView();
      });
    }
  }

  /**
   * {@link AbstractAction} for loading a .XML File into the {@link ObjectEditor}
   */
  public abstract static class OpenXMLFileAction extends AbstractAction {

    protected String lastDir;
    protected ObjectEditor editor;

    public OpenXMLFileAction(String lastDir, String actionTitle) {
      super(actionTitle, new ImageIcon(
          BasicGraphEditor.class.getResource(NESTWORKFLOW_RESOURCES + "images/open.gif")));
      this.lastDir = lastDir;
    }

    public OpenXMLFileAction(String lastDir) {
      this(lastDir, "Open File");
    }

    public OpenXMLFileAction(ObjectEditor editor) {
      this(editor.getCurrentFile() != null ? editor.getCurrentFile().getAbsolutePath() : null);
      this.editor = editor;
    }

    public void actionPerformed(ActionEvent e) {
      String wd = (lastDir != null) ? lastDir : System.getProperty("user.dir");

      JFileChooser fc = new JFileChooser(wd);

      // Adds file filter for supported file format
      DefaultFileFilter defaultFilter = new DefaultFileFilter(".xml", "XML File (.xml)");
      fc.addChoosableFileFilter(defaultFilter);
      fc.setFileFilter(defaultFilter);

      int rc = fc.showDialog(null, "Open File");
      if (rc == JFileChooser.APPROVE_OPTION) {
        lastDir = fc.getSelectedFile().getParent();
        this.consumeSelectedFile(fc.getSelectedFile());
      }
    }

    abstract void consumeSelectedFile(File selectedFile);
  }

  /**
   * Helper Class for loading .XML-Files with the {@link ObjectEditor}
   */
  public static class OpenPoolXMLFileAction extends OpenXMLFileAction {

    public OpenPoolXMLFileAction(ObjectEditor editor) {
      super(editor);
    }

    @Override
    void consumeSelectedFile(File selectedFile) {
      boolean useDataObjectMode = true;
      Object parsedObject = IOUtil.readFile(selectedFile.getAbsolutePath(), DataObject.class);
      if (parsedObject == null) {
        // useDataObjectMode CAN get false here!
        useDataObjectMode = false;
        parsedObject = IOUtil.readFile(selectedFile.getAbsolutePath(),
            ObjectPoolReader.READER_NAME);
        if (!(parsedObject instanceof WriteableObjectPool)) {
          JOptionPane.showMessageDialog(editor.getJframe(),
              "Chosen file does not contain a pool definition or the definition is invalid for the current CakeInstance.",
              "Error", JOptionPane.ERROR_MESSAGE);
          return;
        }
      }

      editor.setCurrentFile(selectedFile);
      if (useDataObjectMode) {
        editor.loadObject((DataObject) parsedObject);
      } else {
        editor.loadObject((WriteableObjectPool) parsedObject);
      }

      editor.setTree(editor.setUpTree());
      JPanel newLeftPane = editor.setUpLeftPane();
      JPanel newRightPane = editor.setUpRightPane();
      editor.setNewMainSplitPane(newLeftPane, newRightPane);
    }
  }

  /**
   * {@link AbstractAction} for loading a .XML File into the {@link ObjectEditor}
   */
  public abstract static class OpenDataObjectFromXMLFileAction extends OpenXMLFileAction {

    static String lastDir;
    Object target;
    Object parent;

    public OpenDataObjectFromXMLFileAction(ObjectEditor editor, String actionTitle, Object target,
        Object parent) {
      super(lastDir, actionTitle);
      this.target = target;
      this.parent = parent;
      this.editor = editor;
    }

    @Override
    void consumeSelectedFile(File selectedFile) {
      Object parsedObject = IOUtil.readFile(selectedFile.getAbsolutePath(),
          ObjectReader.READER_NAME);
      if (!(parsedObject instanceof DataObject)) {
        JOptionPane.showMessageDialog(editor.getJframe(),
            "Chosen file does not contain a data object definition or the definition is invalid for the current CakeInstance.",
            "Error", JOptionPane.ERROR_MESSAGE);
      } else {
        lastDir = selectedFile.getAbsolutePath();
        this.consumeLoadedDataObject((DataObject) parsedObject);
        editor.updateTreeView();
      }
    }

    abstract void consumeLoadedDataObject(DataObject dataObject);
  }

  /**
   * Replaces the target object with the object loaded from an xml file
   */
  public static class ReplaceWithDataObjectFromXMLFileAction extends
      OpenDataObjectFromXMLFileAction {

    public ReplaceWithDataObjectFromXMLFileAction(ObjectEditor editor, Object target,
        Object parent) {
      super(editor, "Replace by XML import", target, parent);
    }

    @Override
    void consumeLoadedDataObject(DataObject dataObject) {
      try {
        if (parent instanceof WriteableObjectPool) {
          WriteableObjectPool parentPool = (WriteableObjectPool) this.parent;
          parentPool.remove((DataObject) target);
          try {
            parentPool.store(dataObject);
          } catch (
              RuntimeException e) { // should be exception thrown due to non unique DataObject ID
            dataObject.setId(null); // Pool will generate new ID when ID is null
            parentPool.store(dataObject);
          }
        } else if (parent instanceof SetObject) {
          SetObject parentSet = (SetObject) this.parent;
          parentSet.removeValue((DataObject) target);
          parentSet.addValue(dataObject);
        } else if (parent instanceof ListObject) {
          ListObject parentList = (ListObject) this.parent;
          parentList.insertAt(dataObject, parentList.indexOf((DataObject) target));
          parentList.removeValue((DataObject) target);
        } else if (parent instanceof AggregateObject) {
          AggregateObject parentAggregate = (AggregateObject) this.parent;
          Map.Entry<String, DataObject> target = (Map.Entry<String, DataObject>) this.target;

          // convert subclasses of string to plain strings
          if (parentAggregate.getAggregateClass().getAttributeType(target.getKey())
              == ModelFactory.getDefaultModel().getStringSystemClass()) {
            StringObject stringObject = (StringObject) ModelFactory.getDefaultModel()
                .getStringSystemClass().newObject();
            stringObject.setNativeString(((StringObject) dataObject).getNativeString());
            parentAggregate.setAttributeValue(target.getKey(), stringObject);
          } else {
            parentAggregate.setAttributeValue(target.getKey(), dataObject);
          }
        } else {
          throw new ProCAKEInvalidTypeException(
              "Invalid parent type. Only collections, pools and aggregates are supported.",
              target.toString(),
              dataObject.getId());
        }
        editor.updateTreeView();
      } catch (ProCAKEInvalidTypeException e) {
        JOptionPane.showMessageDialog(editor.getJframe(), "Import failed due to invalid type of data object.",
            "Error", JOptionPane.ERROR_MESSAGE);
      } catch (Exception e) {
        JOptionPane.showMessageDialog(editor.getJframe(), "Import failed: " + e.getMessage(), "Error",
            JOptionPane.ERROR_MESSAGE);
      }
    }
  }

  /**
   * Replaces the target object with the object placed in the clipboard
   */
  public static class ReplaceWithDataObjectFromClipboardAction extends
      ReplaceWithDataObjectFromXMLFileAction {

    public ReplaceWithDataObjectFromClipboardAction(ObjectEditor editor, Object target,
        Object parent) {
      super(editor, target, parent);
      this.putValue("Name", "Replace by clipboard object");
      this.putValue("SmallIcon", new ImageIcon(BasicGraphEditor.class.getResource(
          NESTWORKFLOW_RESOURCES + "images/paste.gif")));
    }

    public void actionPerformed(ActionEvent e) {
      if (StoreDataObjectInClipboardAction.clipboardObject == null) {
        JOptionPane.showMessageDialog(editor.getJframe(), "Clipboard is empty.", "Error",
            JOptionPane.ERROR_MESSAGE);
      } else {
        this.consumeLoadedDataObject(StoreDataObjectInClipboardAction.clipboardObject.copy());
      }
    }
  }

  /**
   * Adds an object loaded from an xml file to the target collection/pool
   */
  public static class AddDataObjectFromXMLFileAction extends OpenDataObjectFromXMLFileAction {

    public AddDataObjectFromXMLFileAction(ObjectEditor editor, Object target) {
      super(editor, "Add from XML import", target, null);
    }

    @Override
    void consumeLoadedDataObject(DataObject dataObject) {
      if (target instanceof Map.Entry) {
        this.target = ((Map.Entry<String, DataObject>) this.target).getValue();
      }

      try {
        if (target instanceof WriteableObjectPool) {
          WriteableObjectPool targetPool = (WriteableObjectPool) this.target;
          try {
            targetPool.store(dataObject);
          } catch (
              RuntimeException e) { // should be exception thrown due to non unique DataObject ID
            dataObject.setId(null); // Pool will generate new ID when ID is null
            targetPool.store(dataObject);
          }
        } else if (target instanceof CollectionObject) {
          CollectionObject targetCollection = (CollectionObject) this.target;
          targetCollection.addValue(dataObject);
        } else {
          throw new ProCAKEInvalidTypeException(
              "Invalid target object. Only collections and pools are supported.", target.toString(),
              dataObject.getId());
        }
        editor.updateTreeView();
      } catch (ProCAKEInvalidTypeException e) {
        JOptionPane.showMessageDialog(editor.getJframe(), "Import failed due to invalid type of data object.",
            "Error", JOptionPane.ERROR_MESSAGE);
      } catch (Exception e) {
        JOptionPane.showMessageDialog(editor.getJframe(), "Import failed: " + e.getMessage(), "Error",
            JOptionPane.ERROR_MESSAGE);
      }
    }
  }

  /**
   * Adds the clipboard object to the target collection/pool
   */
  public static class AddDataObjectFromClipboardAction extends AddDataObjectFromXMLFileAction {

    public AddDataObjectFromClipboardAction(ObjectEditor editor, Object target) {
      super(editor, target);
      this.putValue("Name", "Add clipboard object");
      this.putValue("SmallIcon", new ImageIcon(BasicGraphEditor.class.getResource(
          NESTWORKFLOW_RESOURCES + "images/paste.gif")));
    }

    @Override
    public void actionPerformed(ActionEvent e) {
      if (StoreDataObjectInClipboardAction.clipboardObject == null) {
        JOptionPane.showMessageDialog(editor.getJframe(), "Clipboard is empty.", "Error",
            JOptionPane.ERROR_MESSAGE);
      } else {
        this.consumeLoadedDataObject(StoreDataObjectInClipboardAction.clipboardObject.copy());
      }
    }
  }

  /**
   * Stores a clone of the given DataObject in a global clipboard
   */
  public static class StoreDataObjectInClipboardAction extends AbstractAction {

    public static DataObject clipboardObject;
    private DataObject objectToStore;

    public StoreDataObjectInClipboardAction(DataObject dataObject) {
      super("Copy", new ImageIcon(BasicGraphEditor.class.getResource(
          NESTWORKFLOW_RESOURCES + "images/copy.gif")));
      this.objectToStore = dataObject;
    }

    @Override
    public void actionPerformed(ActionEvent actionEvent) {
      clipboardObject = objectToStore.copy();
    }
  }


  /**
   * {@link AbstractAction} to Export object from {@link ObjectEditor} to a .XML-File
   */
  public static class ExportAsAction extends AbstractAction {

    protected String lastDir = null;
    private ObjectEditor editor;
    private Supplier<Object> dataObjectToSaveSupplier, poolToSaveSupplier; // supplier to prevent potential null values due to order of
    // instantiation

    public ExportAsAction(ObjectEditor editor, Supplier<Object> dataObjectToSaveSupplier,
        Supplier<Object> poolToSaveSupplier) {
      super("Save to File", new ImageIcon(BasicGraphEditor.class.getResource(
          NESTWORKFLOW_RESOURCES + "images/saveas.gif")));
      this.editor = editor;
      this.dataObjectToSaveSupplier = dataObjectToSaveSupplier;
      this.poolToSaveSupplier = poolToSaveSupplier;
    }

    public ExportAsAction(ObjectEditor editor) {
      this(editor, editor::getDataObject, editor::getPool);
    }

    public void actionPerformed(ActionEvent e) {
      String filename;
      {
        String workingDir;

        if (lastDir != null) {
          workingDir = lastDir;
        } else if (editor.getCurrentFile() != null) {
          workingDir = editor.getCurrentFile().getParent();
        } else {
          workingDir = System.getProperty("user.dir");
        }

        JFileChooser fileChooser = new JFileChooser(workingDir);
        DefaultFileFilter defaultFileFilter = new DefaultFileFilter(".xml", "XML File (.xml)");
        fileChooser.addChoosableFileFilter(defaultFileFilter);
        fileChooser.setFileFilter(defaultFileFilter);

        int rc = fileChooser.showDialog(editor.getJframe(), "Save");
        if (rc != JFileChooser.APPROVE_OPTION) {
          return;
        } else {
          lastDir = fileChooser.getSelectedFile().getParent();
        }

        filename = fileChooser.getSelectedFile().getAbsolutePath();
        FileFilter selectedFilter = fileChooser.getFileFilter();

        if (selectedFilter instanceof DefaultFileFilter) {
          String ext = ((DefaultFileFilter) selectedFilter).getExtension();
          if (!filename.toLowerCase().endsWith(ext)) {
            filename += ext;
          }
        }

        if (new File(filename).exists()
            && JOptionPane.showConfirmDialog(editor.getJframe(), "Overwrite existing file?")
            != JOptionPane.YES_OPTION) {
          return;
        }
      }

//      editor.saveChanges();
      editor.updateTreeView();

      try {
        boolean writeSuccess = false;
        if (dataObjectToSaveSupplier.get() == null) {
          writeSuccess = IOUtil.writeFile(poolToSaveSupplier.get(), filename) != null;
        } else if (poolToSaveSupplier.get() == null) {
          writeSuccess = IOUtil.writeFile(dataObjectToSaveSupplier.get(), filename) != null;
        }
        if (!writeSuccess) {
          throw new IOException("XML serialization failed. Check stacktrace for details.");
        }
        editor.setCurrentFile(new File(filename));
      } catch (Throwable ex) {
        ex.printStackTrace();
        JOptionPane.showMessageDialog(editor.getJframe(), ex.toString(), "Error", JOptionPane.ERROR_MESSAGE);
      }
    }
  }

  /**
   * {@link AbstractAction} to open a new view with the same object and model in place
   */
  public static class OpenNewEditorWindowAction extends AbstractAction {

    private ObjectEditor editor;

    public OpenNewEditorWindowAction(ObjectEditor editor) {
      super("Open new editor window");
      this.editor = editor;
    }

    public void actionPerformed(ActionEvent e) {
      if (editor.getDataObject() != null && editor.getPool() == null) {
        // new instance creation by reflection is needed to keep overridden methods (e.g. in anonymous
        // subclasses)
        try {
          Constructor<? extends ObjectEditor> constructor = this.editor.getClass()
              .getDeclaredConstructor(DataObject.class);
          constructor.setAccessible(true);
          ObjectEditor newInstance = constructor.newInstance(this.editor.getDataObject());
          ObjectEditorTreeModel model = (ObjectEditorTreeModel) editor.getTree().getModel();
          newInstance.loadObject(this.editor.getDataObject());

          newInstance.setTreeModel(model);
          model.addTreeModelListener(newInstance.getTreeModelListener());
          newInstance.setDataObject(editor.getDataObject());
          newInstance.setOriginalDataObject(editor.getOriginalDataObject());

          TreePath selectionPath = this.editor.getTree().getSelectionPath();
          newInstance.getTree().scrollPathToVisible(selectionPath);
          newInstance.getTree().setSelectionPath(selectionPath);

        } catch (InstantiationException | IllegalAccessException | InvocationTargetException |
                 NoSuchMethodException exception) {
          JOptionPane.showMessageDialog(this.editor.getJframe(), exception.toString(), mxResources.get("error"),
              JOptionPane.ERROR_MESSAGE);
        }
      } else if (editor.getDataObject() == null && editor.getPool() != null) {
        try {
          Constructor<? extends ObjectEditor> constructor = this.editor.getClass()
              .getDeclaredConstructor(WriteableObjectPool.class);
          constructor.setAccessible(true);
          ObjectEditor newInstance = constructor.newInstance(this.editor.getPool());
          FilterableObjectPoolTreeModel<DataObject> model = (FilterableObjectPoolTreeModel<DataObject>) editor.getTree()
              .getModel();
          newInstance.setTreeModel(model);

          model.addTreeModelListener(newInstance.getTreeModelListener());
          newInstance.setPool(editor.getPool());
          newInstance.setOriginalPool(editor.getOriginalPool());

          TreePath selectionPath = this.editor.getTree().getSelectionPath();
          newInstance.getTree().scrollPathToVisible(selectionPath);
          newInstance.getTree().setSelectionPath(selectionPath);

        } catch (InstantiationException | IllegalAccessException | InvocationTargetException |
                 NoSuchMethodException exception) {
          JOptionPane.showMessageDialog(this.editor.getJframe(), exception.toString(), mxResources.get("error"),
              JOptionPane.ERROR_MESSAGE);
        }
      }
    }
  }

  /**
   * {@link AbstractAction} to add new data to the object of the {@link ObjectEditor}
   */
  public static class AddNewDataAction extends AbstractAction {

    private final ObjectEditor editor;
    private final JComboBox<DataClass> newDataObjectClassSelector;

    public AddNewDataAction(ObjectEditor editor, JComboBox<DataClass> newDataObjectClassSelector) {
      super("Add");
      this.editor = editor;
      this.newDataObjectClassSelector = newDataObjectClassSelector;
    }

    public void actionPerformed(ActionEvent e) {
      DataClass selectedDataClass = (DataClass) newDataObjectClassSelector.getSelectedItem();
      assert selectedDataClass != null;
      DataObject newDataObject = selectedDataClass.newObject();
      TreePath pathToNewObject;

      if (editor.getDataObject() != null) {
        DataObject obj = editor.getDataObject();
        if (obj instanceof SetObject) {
          ((SetObject) obj).addValue(newDataObject);
        }
        if (obj instanceof ListObject) {
          if (editor.getTree().getSelectionModel().isSelectionEmpty()) {
            ((ListObject) obj).addValue(newDataObject);
          } else {
            ((ListObject) obj).insertAt(newDataObject,
                editor.getTree().getSelectionModel().getLeadSelectionRow());
          }
        }

        // editor.updateTreeView();
        pathToNewObject = new TreePath(new Object[]{editor.getDataObject(), newDataObject});
      } else {
        editor.getPool().store(newDataObject);
//        editor.updateTreeView();
        pathToNewObject = new TreePath(new Object[]{editor.getPool(), newDataObject});
      }

      editor.updateTreeView();
      editor.updateTreeView(pathToNewObject.getPath());
      editor.getTree().scrollPathToVisible(pathToNewObject);
      editor.getTree().setSelectionPath(pathToNewObject);

      editor.setCurrentFile(null);
      if (!editor.getJframe().getTitle().endsWith(" *")) {
        editor.getJframe().setTitle(editor.getJframe().getTitle() + " *");
      }
    }
  }

  /**
   * {@link AbstractAction} to save data to the original object of the {@link ObjectEditor}
   */
  public static class SaveInObjectAction extends AbstractAction {

    private ObjectEditor editor;

    public SaveInObjectAction(ObjectEditor editor) {
      super("Save to Object", new ImageIcon(BasicGraphEditor.class.getResource(
           NESTWORKFLOW_RESOURCES + "images/save.gif")));
      this.editor = editor;
    }

    @Override
    public void actionPerformed(ActionEvent e) {
      editor.writeChangesToOriginalObject();
    }
  }

  /**
   * {@link AbstractAction} to delete data of the object of the {@link ObjectEditor}
   */
  public static class DeleteDataAction extends AbstractAction {

    private ObjectEditor editor;

    public DeleteDataAction(ObjectEditor editor) {
      super("Delete");
      this.editor = editor;
    }

    public void actionPerformed(ActionEvent e) {
      TreePath path = editor.getTree().getSelectionPath();

      if (path == null || editor == null) {
        JOptionPane.showMessageDialog(editor.getJframe(), "No Item selected", "Error",
            JOptionPane.ERROR_MESSAGE);
        return;
      }
      DataObject dataObject = editor.getDataObject();
      Object object = path.getLastPathComponent();

      editor.setCurrentFile(null);
      if (!editor.getJframe().getTitle().endsWith(" *")) {
        editor.getJframe().setTitle(editor.getJframe().getTitle() + " *");
      }

      if (editor.getPool() != null && editor.getDataObject() == null) {
        editor.getPool().remove((DataObject) object);
        editor.updateTreeView();
        return;
      }

      if (!dataObject.isCollection()) {
        return;
      }
      ((CollectionObject) dataObject).removeValue((DataObject) object);

      editor.updateTreeView();
    }
  }
}