package de.uni_trier.wi2.procake.gui.objecteditor;

import de.uni_trier.wi2.procake.data.model.DataClass;
import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTAbstractWorkflowObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTGraphItemObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTWorkflowObject;
import de.uni_trier.wi2.procake.data.objectpool.WriteableObjectPool;
import de.uni_trier.wi2.procake.gui.DataObjectLabelProvider;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.DataObjectEditorSaveable;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.NESTWorkflowEditor;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.SemanticDescriptorEditor;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.editor.EditorToolBar;
import de.uni_trier.wi2.procake.gui.objecteditor.nestworkfloweditor.utils.Utils;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.Map;
import java.util.Objects;
import javax.swing.*;
import javax.swing.event.MouseInputAdapter;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Editor for {@link DataObject}s and {@link WriteableObjectPool}s in ProCAKE
 *
 * @author osobaa
 */
public class ObjectEditor<T extends DataObject> extends DataObjectLabelProvider {

  protected static final Logger logger = LoggerFactory.getLogger(ObjectEditor.class);
  /**
   * Mutex for closing window, used in assigning the {@link WindowAdapter}
   */
  private static final Object windowCloseLock = new Object();
  /**
   * {@link JComboBox} for the selection of the different available {@link DataObject}s to add to
   * the {@link JTree}
   */
  JComboBox<DataClass> newDataObjectClassSelector;
  /**
   * The {@link JScrollPane} to contain the {@link JTree}
   */
  JScrollPane treeScrollPane;
  /**
   * JFrame
   */
  private JFrame Jframe = new JFrame();
  /**
   * original {@link DataObject} to revert to
   */
  private T originalDataObject;
  /**
   * Working {@link DataObject}, where changes are made
   */
  private T dataObject;
  /**
   * Working {@link WriteableObjectPool} of {@link DataObject}, where changes are made
   */

  private WriteableObjectPool<T> pool;
  /**
   * original {@link WriteableObjectPool} of {@link DataObject}s to revert to
   */
  private WriteableObjectPool<T> originalPool;
  /**
   * The {@link JTree} in which the {@link WriteableObjectPool} or {@link DataObject} is represented
   * and shown. The {@link JTree} is wrapped into {@link ObjectJTree} for convenience but can be
   * used just the same.
   */
  private ObjectJTree tree;
  /**
   * When changes are made to the {@link javax.swing.tree.TreeModel} and more than one views are
   * open, this method will be called after {@link #updateTreeView()} or
   * {@link #updateTreeView(Object[])}
   */
  TreeModelListener treeModelListener = new TreeModelListenerAdapter() {
    @Override
    public void treeStructureChanged(TreeModelEvent e) {
      tree.getTree().scrollPathToVisible(e.getTreePath());
    }
  };
  /**
   * The current {@link File} where the data should be saved. Can be null.
   */
  private File currentFile;
  /**
   * {@link JSplitPane} for showing the {@link JTree} on the left panel and the associated editor on
   * the right panel
   */
  private JSplitPane mainSplitPane;
  /**
   * The {@link JButton} to save changes in the {@link SemanticDescriptorEditor}. Can be null
   */
  private JButton saveButton;
  private boolean changesInObject = false;
  private DataObjectEditorSaveable activeDataObjectEditor;
  private NESTWorkflowEditor activeNestGraphEditor;

  // TreeModelListener
  private EditorToolBar extraToolBar = null;

  // Constructors
  public ObjectEditor(T dataObject, boolean blockThreadUntilEditorWindowClosed) {
    this.loadObject(dataObject);
    setUpApp(blockThreadUntilEditorWindowClosed);
  }

  public ObjectEditor(T dataObject) {
    this(dataObject, false);
  }

  public ObjectEditor(WriteableObjectPool<T> pool, boolean blockThreadUntilEditorWindowClosed) {
    this.loadObject(pool);
    setUpApp(blockThreadUntilEditorWindowClosed);
  }

  public ObjectEditor(WriteableObjectPool<T> pool) {
    this(pool, false);
  }

  private void setUpApp(boolean blockThreadUntilEditorWindowClosed) {

    this.setUpFrame();
    this.Jframe.setVisible(true);
    tree = this.setUpTree();

    JPanel leftPanel = this.setUpLeftPane();
    JPanel rightPanel = this.setUpRightPane();

    mainSplitPane = this.setUpSplitPane(leftPanel, rightPanel);

    this.Jframe.add(mainSplitPane);

    this.Jframe.setJMenuBar(new ObjectEditorMenuBar(this));

    // this.add(extraToolBar = new EditorToolBar(JToolBar.HORIZONTAL), BorderLayout.NORTH);
    JPanel topPanel = new JPanel(new BorderLayout());
    topPanel.add(new TreeSelectionPathBar(this, this.tree), BorderLayout.PAGE_START);
    topPanel.add(new PoolFilterBar(this), BorderLayout.PAGE_END);
    this.Jframe.add(topPanel, BorderLayout.PAGE_START);

    this.Jframe.setMinimumSize(new Dimension(1366, 768));
    // this.setExtendedState(this.getExtendedState() | Frame.MAXIMIZED_BOTH);

    if (pool == null && dataObject != null) {
      tree.getTree().setSelectionRow(0);
    } else if (pool != null && dataObject == null) {
      tree.getTree().setSelectionRow(1);
    }

    if (blockThreadUntilEditorWindowClosed) {
      this.waitUntilEditorWindowClosed();
    }
  }

  /**
   * Helper method to set up the JTree with the right listeners and actions
   *
   * @return {@link ObjectJTree} ready to use Tree
   */
  ObjectJTree setUpTree() {
    ObjectJTree tree = new ObjectJTree(this);
    if (pool == null) {
      tree.getTree().setModel(new ObjectEditorTreeModel(dataObject));
    } else if (dataObject == null) {
      tree.getTree().setModel(new FilterableObjectPoolTreeModel<>(pool, this));
    }

    tree.getTree().getModel().addTreeModelListener(treeModelListener);

    // React to specific mouse inputs for the {@link ObjectJTree}
    tree.getTree().addMouseListener(new MouseInputAdapter() {
      @Override
      public void mousePressed(MouseEvent e) {
        if (SwingUtilities.isRightMouseButton(e)) {

          // https://stackoverflow.com/questions/27468337/jtree-select-item-on-right-click
          int selRow = tree.getTree().getRowForLocation(e.getX(), e.getY());
          TreePath selPath = tree.getTree().getPathForLocation(e.getX(), e.getY());
          tree.getTree().setSelectionPath(selPath);
          if (selRow > -1) {
            tree.getTree().setSelectionRow(selRow);
          }

          Point pt = SwingUtilities.convertPoint(e.getComponent(), e.getPoint(),
              ObjectEditor.this.Jframe);
          JPopupMenu menu = new JPopupMenu();
          menu.add(new ObjectEditorActions.AddNewDataAction(ObjectEditor.this,
              ObjectEditor.this.newDataObjectClassSelector));
          menu.add(new ObjectEditorActions.DeleteDataAction(ObjectEditor.this));
          if (tree.getTree().getSelectionPath() == null) {
            for (var element : menu.getComponents()) {
              element.setEnabled(false);
            }
          }
          menu.show(ObjectEditor.this.Jframe, pt.x, pt.y);
        }
      }
    });

    tree.getTree().getSelectionModel().

        setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

    // ObjectPoolEditor @valueChanged
    tree.getTree().addTreeSelectionListener(e -> {

      TreePath selectionPath = tree.getTree().getSelectionPath();
      if (selectionPath == null) {
        return;
      }

      Object[] path = selectionPath.getPath();
      Object selectedElement = path[path.length - 1];

      // Remove extraToolbar for NESTWorkflows, if it exists

      if (extraToolBar != null) {
        ObjectEditor.this.Jframe.remove(extraToolBar);
        ObjectEditor.this.Jframe.add(extraToolBar = new EditorToolBar(JToolBar.HORIZONTAL),
            BorderLayout.NORTH);
      }

      DataObject dataObject = null;
      if (selectedElement instanceof DataObject) {
        dataObject = (DataObject) selectedElement;
      } else if (selectedElement instanceof Map.Entry) { // aggregate attribute
        dataObject = ((Map.Entry<String, DataObject>) selectedElement).getValue();
        if (dataObject == null) {
          ObjectEditor.this.mainSplitPane.setRightComponent(new JLabel(
              ("Creation and deletion of aggregate attributes have to be done in the editor of the corresponding aggregate object.")));
          return;
        }
      }

      if (dataObject != null) {
        if (dataObject.isNESTWorkflow() || dataObject.isNESTSequentialWorkflow()) {
          activeNestGraphEditor = new NESTWorkflowEditor((NESTAbstractWorkflowObject) dataObject,
              true,
              true, true, true);
          activeNestGraphEditor.addGraphSaveListener(
              ObjectEditor.this::updateForNESTGraphEditorSave);
          ObjectEditor.this.mainSplitPane.setRightComponent(activeNestGraphEditor);

          // Add extra Toolbar for NESTWorkflowEditor
          ObjectEditor.this.Jframe.add(
              extraToolBar = new EditorToolBar(activeNestGraphEditor, JToolBar.HORIZONTAL),
              BorderLayout.NORTH);

        } else if (dataObject.isNESTEdge() || dataObject.isNESTNode()) {
          NESTGraphItemObject graphItemObject = (NESTGraphItemObject) dataObject;
          if (graphItemObject.getSemanticDescriptor() != null) {
            ObjectEditor.this.mainSplitPane.setRightComponent(new JScrollPane(
                ObjectEditor.this.createSaveableDataObjectEditorPanel(
                    graphItemObject.getSemanticDescriptor())));
          }
        } else {
          ObjectEditor.this.mainSplitPane.setRightComponent(
              new JScrollPane(ObjectEditor.this.createSaveableDataObjectEditorPanel(dataObject)));
        }
      } else {
        ObjectEditor.this.mainSplitPane.setRightComponent(new JLabel(("")));
      }

      Object[] oldPath =
          e.getOldLeadSelectionPath() != null ? e.getOldLeadSelectionPath().getPath() : null;

      SwingUtilities.invokeLater(() -> {
        SwingUtilities.updateComponentTreeUI(ObjectEditor.this.Jframe);
      });

      if (oldPath == null) {
        return;
      }

      ObjectEditor.this.updateTreeView(oldPath);
    });
    return tree;
  }

  /**
   * Method for creating the right editor for the selected {@link DataObject}
   *
   * @param dataObject the {@link DataObject} to get the editor for
   * @return correct {@link JPanel} containing the editor
   */
  private JPanel createSaveableDataObjectEditorPanel(DataObject dataObject) {
    JPanel panel = new JPanel(new BorderLayout());
    activeDataObjectEditor = new DataObjectEditorSaveable(dataObject, true);
    panel.add(activeDataObjectEditor, BorderLayout.CENTER);

    if (!dataObject.isCollection()) {
      if (this.dataObject != null && (!this.dataObject.isCollection() && this.pool == null)) {
//    if (!this.dataObject.isCollection() && this.pool == null) {
        saveButton = new JButton("Save changes");
        saveButton.addActionListener(e -> {
          this.setCurrentFile(null);
          if (!this.Jframe.getTitle().endsWith(" *")) {
            this.Jframe.setTitle(this.Jframe.getTitle() + " *");
            activeDataObjectEditor.saveChanges();
            this.updateTreeView(
                Objects.requireNonNull(tree.getTree().getSelectionPath()).getPath());
          }
        });
        JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        buttonPanel.add(saveButton);
        panel.add(buttonPanel, BorderLayout.PAGE_START);
      }
    }

    return panel;
  }

//  public void saveChanges() {
//    if (activeDataObjectEditor != null && activeNestGraphEditor == null) {
////      this.setCurrentFile(null);
//      System.out.println();
//      System.out.println(activeDataObjectEditor.getDataObject().equals(activeDataObjectEditor.getOriginal()));
//      if(activeDataObjectEditor.getDataObject().equals(activeDataObjectEditor.getOriginal()))
//      {
//        return;
//      }
//
//      if (!this.getTitle().endsWith(" *")) {
//        this.setTitle(this.getTitle() + " *");
//      }
//      changesInObject = true;
//      activeDataObjectEditor.saveChanges();
//    } else if (activeNestGraphEditor != null && activeDataObjectEditor == null) {
//      NESTWorkflowObject nestWorkflowObject = (NESTWorkflowObject) activeNestGraphEditor.getNESTWorkflow();
//      if (nestWorkflowObject == null || nestWorkflowObject.equals(activeNestGraphEditor.getOriginalNestWorkflow())) {
//        return;
//      }
//      changesInObject = true;
//      activeNestGraphEditor.saveInObjectNESTWorkflow(nestWorkflowObject);
//      activeNestGraphEditor.setModified(false);
//      activeNestGraphEditor.fireGraphSaved();
//    }
//
//    activeNestGraphEditor = null;
//    activeDataObjectEditor = null;
//  }

  /**
   * Helper Method to update the {@link JTree} View and scroll to the provided path
   *
   * @param path The {@link TreePath} to which to scroll to
   */
  void updateTreeView(Object[] path) {
    TreePath selectionPath = tree.getTree().getSelectionPath();
    if (pool == null) {
      ((ObjectEditorTreeModel) tree.getTree().getModel()).fireTreeStructureChanged(path);
    } else if (dataObject == null) {
      ((FilterableObjectPoolTreeModel<?>) tree.getTree().getModel()).fireTreeStructureChanged(path);
    }
    tree.getTree().scrollPathToVisible(selectionPath);
    tree.getTree().setSelectionPath(selectionPath);
  }

  /**
   * Helper Method to update the {@link JTree} View
   */
  void updateTreeView() {
    TreePath selectionPath = tree.getTree().getSelectionPath();
    if (pool == null) {
      ((ObjectEditorTreeModel) tree.getTree().getModel()).fireTreeStructureChanged(
          new Object[]{this.dataObject});
    } else if (dataObject == null) {
      ((ObjectPoolTreeModel<?>) tree.getTree().getModel()).fireTreeStructureChanged(
          new Object[]{this.getPool()});
      tree.getTree().scrollPathToVisible(selectionPath);
      tree.getTree().setSelectionPath(selectionPath);
    }
    tree.getTree().scrollPathToVisible(selectionPath);
    tree.getTree().setSelectionPath(selectionPath);
  }

  /**
   * Helper Method to receive the correct right pane for the {@link JSplitPane}
   *
   * @return Currently returns empty {@link JPanel}
   */
  JPanel setUpRightPane() {
    return new JPanel();
  }

  @SuppressWarnings("BusyWait")
  private void waitUntilEditorWindowClosed() {
    try {
      while (!this.Jframe.isVisible()) {
        Thread.sleep(100);
      }
    } catch (InterruptedException e) {
      logger.error("Exception: ", e);
    }
    synchronized (windowCloseLock) {
      while (this.Jframe.isVisible()) {
        try {
          windowCloseLock.wait();
        } catch (InterruptedException e) {
          logger.error("Exception: ", e);
        }
      }
    }
  }

  /**
   * Helper Method to receive the correct left pane for the {@link JSplitPane}. Has to be called
   * after {@link #setUpTree()}, so that the {@link JScrollPane} for the {@link JTree} can be set
   * up
   *
   * @return Returns {@link JPanel} with the {@link JScrollPane} containing the {@link JTree} and a
   * {@link JPanel} containing elements for adding new Objects to the {@link JTree}
   */
  JPanel setUpLeftPane() {
    JPanel leftPanel = new JPanel(new BorderLayout());
    treeScrollPane = new JScrollPane();
    leftPanel.add(treeScrollPane, BorderLayout.CENTER);
    leftPanel.setMinimumSize(new Dimension(250, 250));
    treeScrollPane.setViewportView(tree.getTree());

    leftPanel.add(new AddDataPanel(this), BorderLayout.PAGE_END);

    return leftPanel;
  }

  /**
   * Helper method to correctly set up the {@link JFrame} of the whole View
   */
  private void setUpFrame() {
    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      Utils.setDefaultFontSize(11.5f); // resolves dpi scaling issue (label fonts too small)
      SwingUtilities.updateComponentTreeUI(this.getJframe());
    } catch (Exception e) {
      logger.error("Exception: ", e);
    }

    this.Jframe.setIconImage(Toolkit.getDefaultToolkit().getImage(
        getClass().getResource("/images/ProCAKE_Icon.png")));
    this.Jframe.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    this.Jframe.setLayout(new BorderLayout());

    this.Jframe.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        synchronized (ObjectEditor.windowCloseLock) {
          JFrame frame = ((JFrame) e.getComponent());
          if (changesInObject) {
            frame.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
            int n = JOptionPane.showConfirmDialog(ObjectEditor.this.getJframe(),
                "You may have unsaved changes." + " Do you want to continue closing the window?",
                "Unsaved Changes", JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
            if (n == JOptionPane.NO_OPTION || n == JOptionPane.CLOSED_OPTION) {
              frame.setVisible(true);
              return;
            }
          }
          frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
          ObjectEditor.this.Jframe.setVisible(false);
          ObjectEditor.this.Jframe.dispose();
          ObjectEditor.windowCloseLock.notify();
        }
      }
    });
    this.Jframe.pack();
  }

  /**
   * Load a {@link DataObject} into the editor. This will replace the old
   * {@link #originalDataObject} and will also remove {@link #originalPool} and {@link #pool}
   *
   * @param dataObject The {@link DataObject} to load
   */
  void loadObject(T dataObject) {
    pool = null;
    originalPool = null;
    this.originalDataObject = dataObject;
    this.dataObject = (T) originalDataObject.copy();
    this.Jframe.setTitle(dataObject.getId());
  }

  /**
   * Load a {@link WriteableObjectPool} into the editor. This will replace the old
   * {@link #originalPool} and will also remove {@link #originalDataObject} and {@link #dataObject}
   *
   * @param pool The {@link WriteableObjectPool} to load
   */
  void loadObject(WriteableObjectPool<T> pool) {
    dataObject = null;
    originalDataObject = null;
//    this.tree.getModel().removeTreeModelListener(this);
    this.originalPool = pool;
    this.pool = pool.copy();

    this.Jframe.setTitle(this.pool.getId());
  }

  /**
   * Sets the correct layout of the left and right component for the main {@link JSplitPane} of the
   * {@link ObjectEditor}
   *
   * @param leftPanel  {@link JPanel} to set as left side of the {@link JSplitPane}. Please use the
   *                   value of {@link #setUpLeftPane()}
   * @param rightPanel {@link JPanel} to set as right side of the {@link JSplitPane}. Please use the
   *                   value of {@link #setUpRightPane()}
   * @return Main {@link JSplitPane} to use in the {@link JFrame} of the {@link ObjectEditor}
   */
  private JSplitPane setUpSplitPane(JPanel leftPanel, JPanel rightPanel) {
    JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
    splitPane.setOneTouchExpandable(true);

    splitPane.setLeftComponent(leftPanel);
    splitPane.setRightComponent(rightPanel);

    splitPane.setDividerSize(12);
    this.Jframe.add(splitPane, BorderLayout.CENTER);

    return splitPane;
  }

  // Getter & Setter

  /**
   * Getter for the {@link #originalDataObject}
   *
   * @return {@link #originalDataObject}
   */
  protected DataObject getOriginalDataObject() {
    return originalDataObject;
  }

  /**
   * Set {@link #originalPool}
   *
   * @param dataObject {@link DataObject} to set as {@link #originalDataObject}
   */
  protected void setOriginalDataObject(T dataObject) {
    this.originalDataObject = dataObject;
  }

  /**
   * Getter for the {@link #currentFile}
   *
   * @return {@link #currentFile}
   */
  protected File getCurrentFile() {
    return currentFile;
  }

  /**
   * Public setter for the {@link #currentFile} and to update the title of the {@link JFrame}
   *
   * @param file {@link File} to set {@link #currentFile} to
   */
  protected void setCurrentFile(File file) {
    this.currentFile = file;
    this.updateTitle();
  }

  /**
   * Helper method to set the correct title for the {@link JFrame}
   */
  private void updateTitle() {
    String ID = dataObject == null ? pool.getId() : dataObject.getId();
    this.Jframe.setTitle(
        ID + ((currentFile != null) ? " - file: " + currentFile.getAbsolutePath() : ""));
  }

  /**
   * Getter for the {@link #dataObject}
   *
   * @return {@link #dataObject}
   */
  protected DataObject getDataObject() {
    return dataObject;
  }

  /**
   * Set {@link #dataObject} to work on
   *
   * @param dataObject {@link DataObject} to work on
   */
  protected void setDataObject(T dataObject) {
    this.dataObject = dataObject;
  }

  /**
   * Saves the changes made in {@link #dataObject} to {@link #originalDataObject} in overwriting it.
   * When the current selected {@link DataObject} uses a {@link DataObjectEditorSaveable} as editor
   * {@link DataObjectEditorSaveable#saveChanges()} will be executed.
   */
  protected void writeChangesToOriginalObject() {
//    saveChanges();
    if (saveButton != null) {
      saveButton.doClick();
    }
    try {
      if (dataObject != null && pool == null) {
        originalDataObject = (T) dataObject.copy();
      } else if (pool != null && dataObject == null) {
        originalPool = pool.copy();
      }
      this.Jframe.setTitle(this.Jframe.getTitle().replace("*", " ").trim());
    } catch (Exception e) {
      JOptionPane.showMessageDialog(this.getJframe(),
          "Error in saving, please refer to the stack trace",
          "Error", JOptionPane.ERROR_MESSAGE);
      throw new RuntimeException(e);
    }

  }

  /**
   * Getter for the {@link #Jframe}
   *
   * @return {@link #Jframe}
   */
  public JFrame getJframe() {
    return Jframe;
  }

  /**
   * Getter for the {@link #tree}
   *
   * @return {@link #tree}
   */
  protected JTree getTree() {
    return tree.getTree();
  }

  /**
   * Setter for the {@link #tree}
   *
   * @param setUpTree {@link ObjectJTree} to set
   */
  protected void setTree(ObjectJTree setUpTree) {
    this.tree = setUpTree;
  }

  /**
   * Getter for the {@link #treeModelListener}
   *
   * @return {@link #treeModelListener}
   */
  protected TreeModelListener getTreeModelListener() {
    return treeModelListener;
  }

  /**
   * Getter for the {@link #pool}
   *
   * @return {@link #pool}
   */
  protected WriteableObjectPool<T> getPool() {
    return pool;
  }

  /**
   * Set {@link #pool} to work on
   *
   * @param pool {@link WriteableObjectPool} to work on
   */
  protected void setPool(WriteableObjectPool<T> pool) {
    this.pool = pool;
  }

  /**
   * Getter for the {@link #originalPool}
   *
   * @return {@link #originalPool}
   */
  protected WriteableObjectPool<T> getOriginalPool() {
    return originalPool;
  }

  /**
   * Set {@link #originalPool}
   *
   * @param pool {@link WriteableObjectPool} to set as {@link #originalPool}
   */
  protected void setOriginalPool(WriteableObjectPool<T> pool) {
    this.originalPool = pool;
  }

  /**
   * Setter for new {@link javax.swing.tree.TreeModel} in case of current object is
   * {@link DataObject}
   *
   * @param model Use object of {@link ObjectPoolTreeModel} to set as new model
   */
  protected void setTreeModel(ObjectEditorTreeModel model) {
    this.tree.getTree().setModel(model);
  }

  /**
   * Setter for new {@link javax.swing.tree.TreeModel} in case of current object is
   * {@link WriteableObjectPool}
   *
   * @param model Use object of {@link FilterableObjectPoolTreeModel} to set as new model
   */
  protected void setTreeModel(FilterableObjectPoolTreeModel<?> model) {
    this.tree.getTree().setModel(model);
  }

  /**
   * Setter for new content of the main {@link JSplitPane} without creating a new one
   *
   * @param l {@link Component} for left side
   * @param r {@link Component} for right side
   */
  void setNewMainSplitPane(Component l, Component r) {
    mainSplitPane.setLeftComponent(l);
    mainSplitPane.setRightComponent(r);
  }

  /**
   * Callback method to apply changes made in a {@link NESTWorkflowEditor} to the current
   * {@link #tree}
   *
   * @param nestWorkflowObject {@link NESTWorkflowObject} that was changed
   */
  private void updateForNESTGraphEditorSave(NESTAbstractWorkflowObject nestWorkflowObject) {
    Object[] path = tree.getTree().getSelectionPath().getPath();
    writeChangesToOriginalObject();
    SwingUtilities.invokeLater(() -> {
      updateTreeView(path);
    });
  }

}