package de.uni_trier.wi2.procake.gui.objecteditor;

import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.objectpool.ReadableObjectPool;
import de.uni_trier.wi2.procake.data.objectpool.WriteableObjectPool;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import javax.swing.tree.TreePath;

public class FilterableObjectPoolTreeModel<T extends DataObject> extends ObjectPoolTreeModel<T> {

  private FilterStatus filterStatus = new FilterStatus();
  private List<Object[]> listOfPaths;
  private ObjectEditor editor;

  public FilterableObjectPoolTreeModel(WriteableObjectPool<T> pool, ObjectEditor editor) {
    super(pool);
    listOfPaths = getAllPaths(pool);
    this.editor = editor;
  }

  @Override
  public Object getChild(Object o, int i) {
    if (o instanceof ReadableObjectPool) {
      return new ArrayList<T>(((ReadableObjectPool) o).getCollection())
          .stream()
          //.sorted(new DataObjectComparator<>())
          .collect(Collectors.toList())
          .get(i);
    }
    return super.getChild(o, i);
  }

  @Override
  public int getChildCount(Object o) {
    if (o instanceof ReadableObjectPool) {
      return (int)
          new ArrayList<T>(((ReadableObjectPool) o).getCollection())
              .stream().count();
    }
    return super.getChildCount(o);
  }

  @Override
  public boolean isLeaf(Object o) {
    if (o instanceof ReadableObjectPool) {
      return this.getChildCount(o) <= 0;
    }
    return super.isLeaf(o);
  }

  public FilterStatus setFilter(String targetText, boolean contextChecked, String filterOptions) {

    String[] targetFilters = targetText.split("&");
    targetFilters = Arrays.stream(targetFilters).map(x -> convertStringToFormat(x))
        .toArray(String[]::new);
    TreePath currentPath = editor.getTree().getSelectionPath();

    if (targetText.isBlank()) {
      filterStatus.resetData(targetText, contextChecked);
      filterStatus.setFilterSuccess(true);
      TreePath root = new TreePath(currentPath.getPathComponent(0));
      editor.getTree().setSelectionPath(root);
      return filterStatus;
    }

    if (isSameFilterFunction(targetText) && isSameInContext(contextChecked) && isSameContextPath(
        currentPath)) {
      getNextFilteredPath(filterOptions);
      return filterStatus;
    }

    filterStatus.resetData(targetText, contextChecked);

    for (int arrPos = 0; arrPos < listOfPaths.size(); arrPos++) {

      Object[] path = listOfPaths.get(arrPos);

      if (contextChecked && !checkInContext(path, currentPath)) {
        continue;
      }

      String[] pathStringsIds = Arrays.stream(path)
          .map(elem -> convertStringToFormat(elem.toString())).toArray(String[]::new);
      String[] pathStringsSemantics = Arrays.stream(path).map(elem -> convertStringToFormat(
          editor.getLabel(elem))).toArray(String[]::new);
      int index = findMatchingFoundedElementIndex(pathStringsIds, pathStringsSemantics,
          targetFilters);

      if (index != -1) {
        filterStatus.getFilteredPaths().add(Arrays.copyOf(listOfPaths.get(arrPos), index + 1));
      }
    }

    if (checkIfFilteredPathExist()) {
      editor.getTree().setSelectionPath(new TreePath(filterStatus.getFilteredPaths().get(0)));
      filterStatus.setOldContextPath(editor.getTree().getSelectionPath());
    }

    return filterStatus;
  }

  private static int findMatchingFoundedElementIndex(String[] pathIds, String[] pathSemantics,
      String[] filterList) {
    int maxIndex = -1;

    boolean pathContainAllFilters = Arrays.stream(filterList)
        .allMatch(filter -> (Arrays.stream(pathSemantics)
            .anyMatch(pathElement -> pathElement.contains(filter)) || Arrays.stream(pathIds)
            .anyMatch(pathElement -> pathElement.contains(filter))
        ));

    if (pathContainAllFilters) {
      for (int i = 0; i < pathIds.length; i++) {
        int finalI = i;
        boolean containsInPathId = Arrays.stream(filterList)
            .anyMatch(filter -> pathIds[finalI].contains(filter));
        boolean containsInPathSemantics = Arrays.stream(filterList)
            .anyMatch(filter -> pathSemantics[finalI].contains(filter));
        if (containsInPathId || containsInPathSemantics) {
          maxIndex = finalI;
        }
      }
    }

    return maxIndex;
  }

  private List<Object[]> getAllPaths(Object parent) {
    List<Object[]> listOfObjectIDs = new ArrayList<>();

    getDeepPaths(parent, new ArrayList<>(), listOfObjectIDs);
    return listOfObjectIDs;
  }

  private void getDeepPaths(Object parent, ArrayList<Object> currentPath,
      List<Object[]> listOfObjectIDs) {
    if (parent == null) {
      return;
    }

    currentPath.add(parent);

    if (getChildCount(parent) == 0) {
      listOfObjectIDs.add(currentPath.toArray());
    } else {
      for (int i = 0; i < getChildCount(parent); i++) {
        getDeepPaths(getChild(parent, i), new ArrayList<>(currentPath), listOfObjectIDs);
      }
    }
  }

  private boolean isSameFilterFunction(String newTargetText) {

    if (filterStatus.getProcTargetText() == null) {
      return false;
    }

    String oldFilterText = filterStatus.getProcTargetText();
    return oldFilterText.equals(convertStringToFormat(newTargetText));
  }

  private boolean isSameInContext(boolean newContextChecked) {
    return newContextChecked == filterStatus.isOldCheckboxStatus();
  }

  private boolean isSameContextPath(TreePath newPath) {
    if (filterStatus.getOldContextPath() == null) {
      return false;
    }
    return newPath == filterStatus.getOldContextPath();
  }

  private void getNextFilteredPath(String backOrNext) {
    int nextPos;
    int oldIndex = filterStatus.getOldFilterPos();
    int filteredPathsSize = filterStatus.getFilteredPaths().size();

    if (backOrNext.equals("Back")) {
      nextPos = ((oldIndex - 1) % filteredPathsSize + filteredPathsSize) % filteredPathsSize;
    } else {
      nextPos = (oldIndex + 1) % filteredPathsSize;
    }

    if (checkIfFilteredPathExist()) {
      editor.getTree().setSelectionPath(new TreePath(filterStatus.getFilteredPaths().get(nextPos)));
      filterStatus.setOldFilterPos(nextPos);
      filterStatus.setOldContextPath(editor.getTree().getSelectionPath());
    }
  }

  private boolean checkIfFilteredPathExist() {
    if (filterStatus.getFilteredPaths().size() == 0) {
      filterStatus.setFilterSuccess(false);
      return false;
    }
    filterStatus.setFilterSuccess(true);
    return true;
  }

  private boolean checkInContext(Object[] path, TreePath currentPath) {
    for (int i = 0; i < currentPath.getPathCount(); i++) {
      if (path[i] != currentPath.getPathComponent(i)) {
        if (editor.getLabel(path[i]) != editor.getLabel(
            currentPath.getPathComponent(i))) {
          return false;
        }
      }
    }

    return true;
  }

  private static String convertStringToFormat(String text) {
    return text.replaceAll("\\s+", "").trim().toLowerCase();
  }

  protected class FilterStatus {

    private boolean filterSuccess;
    private String procTargetText;
    private int oldFilterPos;
    private boolean oldCheckboxStatus = false;
    private TreePath oldContextPath;

    public List<Object[]> getFilteredPaths() {
      return filteredPaths;
    }

    public void setFilteredPaths(List<Object[]> filteredPaths) {
      this.filteredPaths = filteredPaths;
    }

    private List<Object[]> filteredPaths;

    public void setFilterSuccess(boolean filterSuccess) {
      this.filterSuccess = filterSuccess;
    }

    public boolean isFilterSuccess() {
      return filterSuccess;
    }

    public String getProcTargetText() {
      return procTargetText;
    }

    public boolean isOldCheckboxStatus() {
      return oldCheckboxStatus;
    }

    public int getOldFilterPos() {
      return oldFilterPos;
    }

    public void setOldFilterPos(int oldFilterPos) {
      this.oldFilterPos = oldFilterPos;
    }

    public TreePath getOldContextPath() {
      return oldContextPath;
    }

    public void setOldContextPath(TreePath currentContextPath) {
      this.oldContextPath = currentContextPath;
    }

    public void resetData(String targetText, boolean contextChecked) {
      this.procTargetText = convertStringToFormat(targetText);
      this.oldFilterPos = 0;
      this.oldCheckboxStatus = contextChecked;
      this.oldContextPath = null;
      this.filterSuccess = false;
      this.filteredPaths = new ArrayList<>();
    }
  }

}
