package de.uni_trier.wi2.procake.gui;

import de.uni_trier.wi2.procake.data.object.DataObject;
import de.uni_trier.wi2.procake.data.object.base.AggregateObject;
import de.uni_trier.wi2.procake.data.object.base.AtomicObject;
import de.uni_trier.wi2.procake.data.object.base.CollectionObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTEdgeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTGraphObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTNodeObject;
import de.uni_trier.wi2.procake.data.object.nest.NESTWorkflowObject;
import de.uni_trier.wi2.procake.data.objectpool.ReadableObjectPool;
import de.uni_trier.wi2.procake.gui.objecteditor.label.AggregateObjectLabelProvider;
import de.uni_trier.wi2.procake.gui.objecteditor.label.LabelProviderFactory;
import java.util.Map;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class DataObjectLabelProvider {

  private final Logger logger = LoggerFactory.getLogger(DataObjectLabelProvider.class);

  /**
   * Get the label for the corresponding Object
   *
   * @param object The Object to get the label from. Preferably a {@link DataObject}
   * @return Label of type {@link String} from the Object
   */
  public String getLabel(Object object) {
    try {
      if (object == null) {
        return "null";
      } else if (object instanceof AtomicObject) {
        return getLabel((AtomicObject) object);
      } else if (object instanceof AggregateObject) {
        return getLabel((AggregateObject) object);
      } else if (object instanceof NESTWorkflowObject) {
        return getLabel((NESTWorkflowObject) object);
      } else if (object instanceof NESTGraphObject) {
        return getLabel((NESTGraphObject) object);
      } else if (object instanceof Map.Entry) { // attribute of an aggregate
        return getLabel((Map.Entry<String, Object>) object);
      } else if (object instanceof CollectionObject) {
        return getLabel((CollectionObject) object);
      } else if (object instanceof NESTEdgeObject) {
        return getLabel((NESTEdgeObject) object);
      } else if (object instanceof NESTNodeObject) {
        return getLabel((NESTNodeObject) object);
      } else if (object instanceof ReadableObjectPool) {
        return getLabel((ReadableObjectPool<DataObject>) object);
      } else if (object instanceof DataObject) {
        return getLabel((DataObject) object);
      }
    } catch (RuntimeException e) {
      logger.error("Could not get label for object: {}", object);
      logger.error("Exception: ", e);
    }
    return Objects.toString(object);
  }

  /**
   * Get the Label/ID of a given {@link DataObject}
   *
   * @param dataObject {@link DataObject} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(DataObject dataObject) {
    return dataObject.getId();
  }

  /**
   * Get the Label/ID of a given {@link ReadableObjectPool}
   *
   * @param pool {@link ReadableObjectPool} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(ReadableObjectPool<DataObject> pool) {
    return pool.getId();
  }

  /**
   * Get the Label/ID of a given {@link Map.Entry}
   *
   * @param attribute {@link Map.Entry} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(Map.Entry<String, Object> attribute) {
    return attribute.getKey() + " = " + getLabel(attribute.getValue());
  }

  /**
   * Get the Label/ID of a given {@link AtomicObject}
   *
   * @param atomicObject {@link AtomicObject} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(AtomicObject atomicObject) {
    return atomicObject.getNativeObject().toString();
  }

  /**
   * Get the Label/ID of a given {@link AggregateObject}
   *
   * @param aggregateObject {@link AggregateObject} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(AggregateObject aggregateObject) {
    AggregateObjectLabelProvider labelProvider = LabelProviderFactory.getProvider(
        aggregateObject.getDataClass());
    return labelProvider == null ? aggregateObject.getId()
        : labelProvider.getLabel(aggregateObject);
  }

  /**
   * Get the Label/ID of a given {@link CollectionObject}
   *
   * @param collectionObject {@link CollectionObject} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(CollectionObject collectionObject) {
    return collectionObject.getDataClass() + " of " + collectionObject.size() + " "
        + collectionObject.getCollectionClass().getElementClass().getName();
  }

  /**
   * Get the Label/ID of a given {@link NESTEdgeObject}
   *
   * @param nestEdgeObject {@link NESTEdgeObject} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(NESTEdgeObject nestEdgeObject) {
    return nestEdgeObject.getDataClass().getName() + ": " + nestEdgeObject;
  }

  /**
   * Get the Label/ID of a given {@link NESTNodeObject}
   *
   * @param nestNodeObject {@link NESTNodeObject} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(NESTNodeObject nestNodeObject) {
    return nestNodeObject.getDataClass().getName() + ": " + nestNodeObject.getId();
  }

  /**
   * Get the Label/ID of a given {@link NESTGraphObject}
   *
   * @param nestGraphObject {@link NESTGraphObject} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(NESTGraphObject nestGraphObject) {
    return nestGraphObject.getId();
  }

  /**
   * Get the Label/ID of a given {@link NESTWorkflowObject}
   *
   * @param nestWorkflowObject {@link NESTWorkflowObject} to get the ID from
   * @return ID in form of {@link String}
   */
  protected String getLabel(NESTWorkflowObject nestWorkflowObject) {
    return nestWorkflowObject.getId();
  }

}
