/*
 * Decompiled with CFR 0.152.
 */
package migratedb.v1.core.internal.command;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import migratedb.v1.core.api.MigrateDbException;
import migratedb.v1.core.api.MigrationInfo;
import migratedb.v1.core.api.MigrationState;
import migratedb.v1.core.api.TargetVersion;
import migratedb.v1.core.api.Version;
import migratedb.v1.core.api.callback.Event;
import migratedb.v1.core.api.configuration.Configuration;
import migratedb.v1.core.api.executor.Context;
import migratedb.v1.core.api.executor.MigrationExecutor;
import migratedb.v1.core.api.internal.callback.CallbackExecutor;
import migratedb.v1.core.api.internal.database.base.Database;
import migratedb.v1.core.api.internal.database.base.Schema;
import migratedb.v1.core.api.internal.database.base.Session;
import migratedb.v1.core.api.logging.Log;
import migratedb.v1.core.api.output.CommandResultFactory;
import migratedb.v1.core.api.output.MigrateResult;
import migratedb.v1.core.api.resolver.MigrationResolver;
import migratedb.v1.core.api.resolver.ResolvedMigration;
import migratedb.v1.core.internal.info.MigrationInfoServiceImpl;
import migratedb.v1.core.internal.info.ValidationContext;
import migratedb.v1.core.internal.info.ValidationMatch;
import migratedb.v1.core.internal.jdbc.ExecutionTemplateFactory;
import migratedb.v1.core.internal.schemahistory.SchemaHistory;
import migratedb.v1.core.internal.util.DateTimeUtils;
import migratedb.v1.core.internal.util.ExceptionUtils;
import migratedb.v1.core.internal.util.StopWatch;
import migratedb.v1.core.internal.util.StringUtils;
import org.checkerframework.checker.nullness.qual.Nullable;

public class DbMigrate {
    private static final Log LOG = Log.getLog(DbMigrate.class);
    private final Database database;
    private final SchemaHistory schemaHistory;
    private final Schema schema;
    private final MigrationResolver migrationResolver;
    private final Configuration configuration;
    private final CallbackExecutor callbackExecutor;
    private final Session session;
    private MigrateResult migrateResult;
    private boolean isPreviousVersioned;
    private final List<ResolvedMigration> appliedResolvedMigrations = new ArrayList<ResolvedMigration>();

    public DbMigrate(Database database, SchemaHistory schemaHistory, Schema schema, MigrationResolver migrationResolver, Configuration configuration, CallbackExecutor callbackExecutor) {
        this.database = database;
        this.session = database.getMigrationSession();
        this.schemaHistory = schemaHistory;
        this.schema = schema;
        this.migrationResolver = migrationResolver;
        this.configuration = configuration;
        this.callbackExecutor = callbackExecutor;
    }

    public MigrateResult migrate() throws MigrateDbException {
        int count;
        this.callbackExecutor.onMigrateEvent(Event.BEFORE_MIGRATE);
        this.migrateResult = CommandResultFactory.createMigrateResult(this.database.getCatalog(), this.configuration);
        try {
            StopWatch stopWatch = new StopWatch();
            stopWatch.start();
            count = this.configuration.isGroup() ? this.schemaHistory.withLock(this::migrateAll).intValue() : this.migrateAll();
            stopWatch.stop();
            this.migrateResult.targetSchemaVersion = this.getTargetVersion();
            this.migrateResult.migrationsExecuted = count;
            this.logSummary(count, stopWatch.getTotalTimeMillis(), this.migrateResult.targetSchemaVersion);
        }
        catch (MigrateDbException e) {
            this.callbackExecutor.onMigrateEvent(Event.AFTER_MIGRATE_ERROR);
            throw e;
        }
        if (count > 0) {
            this.callbackExecutor.onMigrateEvent(Event.AFTER_MIGRATE_APPLIED);
        }
        this.callbackExecutor.onMigrateEvent(Event.AFTER_MIGRATE);
        return this.migrateResult;
    }

    private String getTargetVersion() {
        if (!this.migrateResult.migrations.isEmpty()) {
            for (int i = this.migrateResult.migrations.size() - 1; i >= 0; --i) {
                String targetVersion = this.migrateResult.migrations.get((int)i).version;
                if (targetVersion.isEmpty()) continue;
                return targetVersion;
            }
        }
        return null;
    }

    private int migrateAll() {
        int count;
        int total = 0;
        this.isPreviousVersioned = true;
        do {
            boolean firstRun = total == 0;
            count = this.configuration.isGroup() ? this.migrateGroup(firstRun).intValue() : this.schemaHistory.withLock(() -> this.migrateGroup(firstRun)).intValue();
            total += count;
        } while (count != 0 && !Objects.equals(this.configuration.getTarget(), TargetVersion.NEXT));
        if (this.isPreviousVersioned) {
            this.callbackExecutor.onMigrateEvent(Event.AFTER_VERSIONED);
        }
        return total;
    }

    private Integer migrateGroup(boolean firstRun) {
        MigrationInfo[] failed;
        MigrationInfo[] future;
        String currentSchemaVersionString;
        EnumSet<ValidationMatch> allowedMatches = EnumSet.allOf(ValidationMatch.class);
        if (!this.configuration.isOutOfOrder()) {
            allowedMatches.remove((Object)ValidationMatch.OUT_OF_ORDER);
        }
        MigrationInfoServiceImpl infoService = new MigrationInfoServiceImpl(this.migrationResolver, this.schemaHistory, this.database, this.configuration, this.configuration.getTarget(), this.configuration.getCherryPick(), new ValidationContext(allowedMatches));
        infoService.refresh();
        MigrationInfo current = infoService.current();
        Version currentSchemaVersion = current == null ? null : current.getVersion();
        String string = currentSchemaVersionString = currentSchemaVersion == null ? "<< Empty Schema >>" : currentSchemaVersion.toString();
        if (firstRun) {
            LOG.info("Current version of schema " + this.schema + ": " + currentSchemaVersionString);
            this.migrateResult.initialSchemaVersion = currentSchemaVersionString;
            if (this.configuration.isOutOfOrder()) {
                String outOfOrderWarning = "outOfOrder mode is active. Migration of schema " + this.schema + " may not be reproducible.";
                LOG.warn(outOfOrderWarning);
                this.migrateResult.addWarning(outOfOrderWarning);
            }
        }
        if ((future = infoService.future()).length > 0) {
            List<MigrationInfo> resolved = Arrays.asList(infoService.resolved());
            Collections.reverse(resolved);
            if (resolved.isEmpty()) {
                LOG.error("Schema " + this.schema + " has version " + currentSchemaVersionString + ", but no migration could be resolved in the configured locations !");
            } else {
                for (MigrationInfo migrationInfo : resolved) {
                    if (migrationInfo.getVersion() == null) continue;
                    LOG.warn("Schema " + this.schema + " has a version (" + currentSchemaVersionString + ") that is newer than the latest available migration (" + migrationInfo.getVersion() + ") !");
                    break;
                }
            }
        }
        if ((failed = infoService.failed()).length > 0) {
            if (failed.length == 1 && failed[0].getState() == MigrationState.FUTURE_FAILED && this.configuration.isIgnoreFutureMigrations()) {
                LOG.warn("Schema " + this.schema + " contains a failed future migration to version " + failed[0].getVersion() + " !");
            } else {
                if (failed[0].getVersion() == null) {
                    throw new MigrateDbException("Schema " + this.schema + " contains a failed repeatable migration (" + this.doQuote(failed[0].getDescription()) + ") !");
                }
                throw new MigrateDbException("Schema " + this.schema + " contains a failed migration to version " + failed[0].getVersion() + " !");
            }
        }
        LinkedHashMap<MigrationInfo, Boolean> group = new LinkedHashMap<MigrationInfo, Boolean>();
        for (MigrationInfo pendingMigration : infoService.pending()) {
            if (this.appliedResolvedMigrations.contains(pendingMigration.getResolvedMigration())) continue;
            boolean isOutOfOrder = this.isOutOfOrder(pendingMigration, currentSchemaVersion);
            group.put(pendingMigration, isOutOfOrder);
            if (!this.configuration.isGroup()) break;
        }
        if (!group.isEmpty()) {
            boolean bl = this.configuration.isSkipExecutingMigrations();
            this.applyMigrations(group, bl);
        }
        return group.size();
    }

    private boolean isOutOfOrder(MigrationInfo pendingMigration, @Nullable Version currentSchemaVersion) {
        Version pendingVersion = pendingMigration.getVersion();
        if (pendingVersion == null) {
            return false;
        }
        return currentSchemaVersion != null && pendingVersion.compareTo(currentSchemaVersion) < 0;
    }

    private void logSummary(int migrationSuccessCount, long executionTime, String targetVersion) {
        if (migrationSuccessCount == 0) {
            LOG.info("Schema " + this.schema + " is up to date. No migration necessary.");
            return;
        }
        Object targetText = targetVersion != null ? ", now at version v" + targetVersion : "";
        String migrationText = migrationSuccessCount == 1 ? "migration" : "migrations";
        LOG.info("Successfully applied " + migrationSuccessCount + " " + migrationText + " to schema " + this.schema + (String)targetText + " (execution time " + DateTimeUtils.formatDuration(executionTime) + ")");
    }

    private void applyMigrations(Map<MigrationInfo, Boolean> group, boolean skipExecutingMigrations) {
        boolean executeGroupInTransaction = this.isExecuteGroupInTransaction(group);
        StopWatch stopWatch = new StopWatch();
        try {
            if (executeGroupInTransaction) {
                ExecutionTemplateFactory.createExecutionTemplate(this.session.getJdbcConnection(), this.database).execute(() -> {
                    this.doMigrateGroup(group, stopWatch, skipExecutingMigrations, true);
                    return null;
                });
            } else {
                this.doMigrateGroup(group, stopWatch, skipExecutingMigrations, false);
            }
        }
        catch (MigrateDbMigrateException e) {
            MigrationInfo migration = e.getMigration();
            ResolvedMigration resolvedMigration = migration.getResolvedMigration();
            assert (resolvedMigration != null);
            String failedMsg = "Migration of " + this.toMigrationText(migration, e.isOutOfOrder()) + " failed!";
            if (this.database.supportsDdlTransactions() && executeGroupInTransaction) {
                LOG.error(failedMsg + " Changes successfully rolled back.");
            } else {
                LOG.error(failedMsg + " Please restore backups and roll back database and code!");
                stopWatch.stop();
                int executionTime = (int)stopWatch.getTotalTimeMillis();
                this.schemaHistory.addAppliedMigration(migration.getVersion(), migration.getDescription(), migration.getType(), migration.getScript(), resolvedMigration.getChecksum(), executionTime, false);
            }
            throw e;
        }
    }

    private boolean isExecuteGroupInTransaction(Map<MigrationInfo, Boolean> group) {
        boolean executeGroupInTransaction = true;
        boolean first = true;
        for (Map.Entry<MigrationInfo, Boolean> entry : group.entrySet()) {
            ResolvedMigration resolvedMigration = entry.getKey().getResolvedMigration();
            assert (resolvedMigration != null);
            boolean inTransaction = resolvedMigration.getExecutor().canExecuteInTransaction();
            if (first) {
                executeGroupInTransaction = inTransaction;
                first = false;
                continue;
            }
            if (!this.configuration.isMixed() && executeGroupInTransaction != inTransaction) {
                throw new MigrateDbException("Detected both transactional and non-transactional migrations within the same migration group (even though mixed is false). First offending migration: " + this.doQuote((Comparable)(resolvedMigration.getVersion() == null ? "" : resolvedMigration.getVersion()) + (String)(StringUtils.hasLength(resolvedMigration.getDescription()) ? " " + resolvedMigration.getDescription() : "")) + (inTransaction ? "" : " [non-transactional]"));
            }
            executeGroupInTransaction &= inTransaction;
        }
        return executeGroupInTransaction;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doMigrateGroup(Map<MigrationInfo, Boolean> group, StopWatch stopWatch, boolean skipExecutingMigrations, boolean isExecuteInTransaction) {
        Context context = new Context(){

            @Override
            public Configuration getConfiguration() {
                return DbMigrate.this.configuration;
            }

            @Override
            public Connection getConnection() {
                return DbMigrate.this.session.getJdbcConnection();
            }
        };
        for (Map.Entry<MigrationInfo, Boolean> entry : group.entrySet()) {
            MigrationInfo migrationInfo = entry.getKey();
            ResolvedMigration resolvedMigration = migrationInfo.getResolvedMigration();
            assert (resolvedMigration != null);
            boolean isOutOfOrder = entry.getValue();
            String migrationText = this.toMigrationText(migrationInfo, isOutOfOrder);
            stopWatch.start();
            if (this.isPreviousVersioned && migrationInfo.getVersion() == null) {
                this.callbackExecutor.onMigrateEvent(Event.AFTER_VERSIONED);
                this.callbackExecutor.onMigrateEvent(Event.BEFORE_REPEATABLES);
                this.isPreviousVersioned = false;
            }
            if (skipExecutingMigrations) {
                LOG.debug("Skipping execution of migration of " + migrationText);
            } else {
                LOG.debug("Starting migration of " + migrationText + " ...");
                this.session.restoreOriginalState();
                this.session.changeCurrentSchemaTo(this.schema);
                try {
                    this.callbackExecutor.setMigrationInfo(migrationInfo);
                    this.callbackExecutor.onEachMigrateEvent(Event.BEFORE_EACH_MIGRATE);
                    try {
                        LOG.info("Migrating " + migrationText);
                        boolean oldAutoCommit = context.getConnection().getAutoCommit();
                        if (this.database.usesSingleSession() && !isExecuteInTransaction) {
                            context.getConnection().setAutoCommit(true);
                        }
                        resolvedMigration.getExecutor().execute(context);
                        if (this.database.usesSingleSession() && !isExecuteInTransaction) {
                            context.getConnection().setAutoCommit(oldAutoCommit);
                        }
                        this.appliedResolvedMigrations.add(resolvedMigration);
                    }
                    catch (MigrateDbException e) {
                        this.callbackExecutor.onEachMigrateEvent(Event.AFTER_EACH_MIGRATE_ERROR);
                        throw new MigrateDbMigrateException(migrationInfo, isOutOfOrder, e);
                    }
                    catch (SQLException e) {
                        this.callbackExecutor.onEachMigrateEvent(Event.AFTER_EACH_MIGRATE_ERROR);
                        throw new MigrateDbMigrateException(migrationInfo, isOutOfOrder, e);
                    }
                    LOG.debug("Successfully completed migration of " + migrationText);
                    this.callbackExecutor.onEachMigrateEvent(Event.AFTER_EACH_MIGRATE);
                }
                finally {
                    this.callbackExecutor.setMigrationInfo(null);
                }
            }
            stopWatch.stop();
            int executionTime = (int)stopWatch.getTotalTimeMillis();
            this.migrateResult.migrations.add(CommandResultFactory.createMigrateOutput(migrationInfo, executionTime));
            this.schemaHistory.addAppliedMigration(migrationInfo.getVersion(), migrationInfo.getDescription(), migrationInfo.getType(), migrationInfo.getScript(), resolvedMigration.getChecksum(), executionTime, true);
        }
    }

    private String toMigrationText(MigrationInfo migration, boolean isOutOfOrder) {
        ResolvedMigration resolvedMigration = migration.getResolvedMigration();
        assert (resolvedMigration != null);
        MigrationExecutor migrationExecutor = resolvedMigration.getExecutor();
        String migrationText = migration.getVersion() != null ? "schema " + this.schema + " to version " + this.doQuote(migration.getVersion() + (String)(StringUtils.hasLength(migration.getDescription()) ? " - " + migration.getDescription() : "")) + (isOutOfOrder ? " [out of order]" : "") + (migrationExecutor.canExecuteInTransaction() ? "" : " [non-transactional]") : "schema " + this.schema + " with repeatable migration " + this.doQuote(migration.getDescription()) + (migrationExecutor.canExecuteInTransaction() ? "" : " [non-transactional]");
        return migrationText;
    }

    private String doQuote(String text) {
        return "\"" + text + "\"";
    }

    public static class MigrateDbMigrateException
    extends MigrateDbException {
        private final MigrationInfo migration;
        private final boolean outOfOrder;

        MigrateDbMigrateException(MigrationInfo migration, boolean outOfOrder, SQLException e) {
            super(ExceptionUtils.toMessage(e), e);
            this.migration = migration;
            this.outOfOrder = outOfOrder;
        }

        MigrateDbMigrateException(MigrationInfo migration, boolean outOfOrder, MigrateDbException e) {
            super(e.getMessage(), e);
            this.migration = migration;
            this.outOfOrder = outOfOrder;
        }

        public MigrationInfo getMigration() {
            return this.migration;
        }

        public boolean isOutOfOrder() {
            return this.outOfOrder;
        }
    }
}

