/*
 * Decompiled with CFR 0.152.
 */
package migratedb.v1.core.internal.command;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;
import migratedb.v1.core.api.MigrateDbException;
import migratedb.v1.core.api.MigrationType;
import migratedb.v1.core.api.Version;
import migratedb.v1.core.api.callback.Event;
import migratedb.v1.core.api.configuration.Configuration;
import migratedb.v1.core.api.internal.callback.CallbackExecutor;
import migratedb.v1.core.api.internal.database.base.Database;
import migratedb.v1.core.api.internal.database.base.Schema;
import migratedb.v1.core.api.internal.database.base.Table;
import migratedb.v1.core.api.internal.jdbc.JdbcTemplate;
import migratedb.v1.core.api.logging.Log;
import migratedb.v1.core.api.output.CommandResultFactory;
import migratedb.v1.core.api.output.LiberateAction;
import migratedb.v1.core.api.output.LiberateResult;
import migratedb.v1.core.internal.exception.MigrateDbSqlException;
import migratedb.v1.core.internal.jdbc.JdbcNullTypes;
import migratedb.v1.core.internal.schemahistory.SchemaHistory;
import org.checkerframework.checker.nullness.qual.Nullable;

public class DbLiberate {
    private static final Log LOG = Log.getLog(DbLiberate.class);
    private final SchemaHistory schemaHistory;
    private final Configuration configuration;
    private final Database database;
    private final Schema defaultSchema;
    private final Schema[] schemas;
    private final CallbackExecutor callbackExecutor;
    private final JdbcTemplate jdbcTemplate;
    private final boolean failOnNonEmptySchemaHistoryTable;

    public DbLiberate(SchemaHistory schemaHistory, Configuration configuration, Database database, Schema defaultSchema, Schema[] schemas, CallbackExecutor callbackExecutor, boolean failOnNonEmptySchemaHistoryTable) {
        this.schemaHistory = schemaHistory;
        this.configuration = configuration;
        this.database = database;
        this.defaultSchema = defaultSchema;
        this.schemas = schemas;
        this.callbackExecutor = callbackExecutor;
        this.jdbcTemplate = database.getMainSession().getJdbcTemplate();
        this.failOnNonEmptySchemaHistoryTable = failOnNonEmptySchemaHistoryTable;
    }

    public LiberateResult liberate() {
        LiberateResult result;
        try {
            this.callbackExecutor.onEvent(Event.BEFORE_LIBERATE);
            result = this.doLiberate();
        }
        catch (MigrateDbException e) {
            this.callbackExecutor.onEvent(Event.AFTER_LIBERATE_ERROR);
            throw e;
        }
        this.callbackExecutor.onEvent(Event.AFTER_LIBERATE);
        return result;
    }

    private LiberateResult doLiberate() {
        Table fromTable = Stream.of(Stream.of(this.defaultSchema), Arrays.stream(this.schemas)).flatMap(it -> it).map(it -> it.getTable(this.configuration.getOldTable())).filter(Table::exists).findFirst().orElse(null);
        if (fromTable == null) {
            throw new MigrateDbException("The table " + this.configuration.getOldTable() + " was not found in any schema");
        }
        return this.database.getMainSession().lock(fromTable, () -> {
            if (!this.schemaHistory.exists()) {
                this.schemaHistory.create(false);
            } else if (this.schemaHistory.hasAppliedMigrations()) {
                String message = "Cannot convert old schema history since target table " + this.schemaHistory.getTable() + " already has applied migrations";
                if (this.failOnNonEmptySchemaHistoryTable) {
                    throw new MigrateDbException(message);
                }
                return CommandResultFactory.createLiberateResult(this.configuration, this.database, this.schemaHistory.getTable().getSchema().getName(), this.schemaHistory.getTable().getName(), List.of(new LiberateAction("aborted", message)));
            }
            List changes = this.database.getMainSession().lock(this.schemaHistory.getTable(), () -> this.convertToMigrateDb(fromTable));
            return CommandResultFactory.createLiberateResult(this.configuration, this.database, this.schemaHistory.getTable().getSchema().getName(), this.schemaHistory.getTable().getName(), changes);
        });
    }

    private List<LiberateAction> convertToMigrateDb(Table fromTable) {
        try {
            ArrayList<LiberateAction> output = new ArrayList<LiberateAction>();
            List<OldSchemaHistoryRow> oldSchemaHistory = this.collectOldSchemaHistory(fromTable, output);
            this.copyToCurrentSchemaHistory(oldSchemaHistory, output);
            return output;
        }
        catch (SQLException e) {
            throw new MigrateDbSqlException("Failed to convert old schema history", e);
        }
    }

    private List<OldSchemaHistoryRow> collectOldSchemaHistory(Table fromTable, List<LiberateAction> output) throws SQLException {
        LinkedList<OldSchemaHistoryRow> oldSchemaHistory = new LinkedList<OldSchemaHistoryRow>();
        ArrayList<OldSchemaHistoryRow> rows = new ArrayList<OldSchemaHistoryRow>(this.jdbcTemplate.query("select * from " + fromTable, this::readOldSchemaHistoryRow, new Object[0]));
        rows.sort(Comparator.comparing(it -> it.installedRank));
        for (OldSchemaHistoryRow row : rows) {
            if (row.isUndo) {
                output.add(new LiberateAction("skipped_undo_migration", "Skipped undo migration: " + row));
                this.removeUndoneMigration(oldSchemaHistory, row, output);
                continue;
            }
            if (row.isTableMarker) {
                output.add(new LiberateAction("skipped_table_marker", "Skipped table creation marker: " + row));
                continue;
            }
            assert (row.type != null);
            String rowType = row.type.name().toLowerCase(Locale.ROOT);
            if (row.type.equals((Object)MigrationType.DELETED)) {
                output.add(new LiberateAction("skipped_" + rowType + "_migration", "Skipped deleted migration: " + row));
                this.removeDeletedMigration(oldSchemaHistory, row);
                continue;
            }
            if (row.type.equals((Object)MigrationType.SCHEMA)) {
                output.add(new LiberateAction("skipped_" + rowType + "_marker", "Skipped schema creation marker: " + row));
                continue;
            }
            oldSchemaHistory.add(row);
        }
        oldSchemaHistory.sort(Comparator.comparing(it -> it.installedRank));
        return oldSchemaHistory;
    }

    private void removeDeletedMigration(List<OldSchemaHistoryRow> oldSchemaHistory, OldSchemaHistoryRow row) {
        if (row.version != null) {
            oldSchemaHistory.removeIf(it -> row.version.equals(it.version));
        } else {
            oldSchemaHistory.removeIf(it -> it.version == null && Objects.equals(it.description, row.description));
        }
    }

    private void removeUndoneMigration(List<OldSchemaHistoryRow> oldSchemaHistory, OldSchemaHistoryRow undoRow, List<LiberateAction> output) {
        assert (undoRow.isUndo);
        if (undoRow.success && undoRow.version != null) {
            ListIterator<OldSchemaHistoryRow> iter = oldSchemaHistory.listIterator();
            while (iter.hasNext()) {
                OldSchemaHistoryRow next = iter.next();
                if (!Objects.equals(undoRow.version, next.version) || undoRow.installedRank <= next.installedRank) continue;
                iter.remove();
                output.add(new LiberateAction("skipped_undone_migration", "Skipped undone migration: " + next));
            }
        }
    }

    private void copyToCurrentSchemaHistory(List<OldSchemaHistoryRow> rowsInInsertionOrder, List<LiberateAction> output) throws SQLException {
        for (OldSchemaHistoryRow row : rowsInInsertionOrder) {
            assert (row.type != null);
            String versionStr = row.version == null ? null : row.version.toString();
            String description = row.description;
            if (!this.database.supportsEmptyMigrationDescription() && "".equals(description)) {
                description = "<< no description >>";
            }
            Object versionObj = versionStr == null ? JdbcNullTypes.StringNull : versionStr;
            JdbcNullTypes checksumObj = JdbcNullTypes.StringNull;
            this.jdbcTemplate.update(this.database.getInsertStatement(this.schemaHistory.getTable()), new Object[]{row.installedRank, versionObj, description, row.type.name(), row.script, checksumObj, row.installedBy, row.executionTime, row.success});
            output.add(new LiberateAction("copied_" + row.type.toString().toLowerCase(Locale.ROOT) + "_migration", "Copied to MigrateDB schema history: " + row));
            LOG.info("Copied " + row + " to MigrateDB Schema History");
        }
    }

    private OldSchemaHistoryRow readOldSchemaHistoryRow(ResultSet rs) throws SQLException {
        ResultSetReader reader = new ResultSetReader(rs);
        int installedRank = reader.requireInt("installed_rank");
        String description = reader.requireString("description");
        Version version = reader.readString("version").map(Version::parse).orElse(null);
        String typeAsString = reader.requireString("type");
        String script = reader.readString("script").orElse("");
        String installedBy = reader.readString("installed_by").orElse(this.database.getInstalledBy());
        Integer executionTime = reader.readInt("execution_time").orElse(0);
        boolean success = reader.requireBoolean("success");
        boolean isUndo = typeAsString.contains("UNDO");
        boolean isTableMarker = typeAsString.equals("TABLE");
        MigrationType type = null;
        try {
            if (!isUndo && !isTableMarker) {
                typeAsString = typeAsString.replace("STATE_SCRIPT", "BASELINE");
                typeAsString = typeAsString.replace("CUSTOM", "JDBC");
                typeAsString = typeAsString.replace("SCRIPT", "SQL");
                type = MigrationType.fromString(typeAsString);
            }
        }
        catch (IllegalArgumentException e) {
            throw new MigrateDbException("Conversion failed: Unsupported migration type '" + typeAsString + "'");
        }
        return new OldSchemaHistoryRow(installedRank, description, version, type, isUndo, isTableMarker, script, installedBy, executionTime, success);
    }

    private static final class OldSchemaHistoryRow {
        public final int installedRank;
        public final String description;
        public final @Nullable Version version;
        public final @Nullable MigrationType type;
        public final boolean isUndo;
        public final boolean isTableMarker;
        public final String script;
        public final String installedBy;
        public final int executionTime;
        public final boolean success;

        private OldSchemaHistoryRow(int installedRank, String description, @Nullable Version version, @Nullable MigrationType type, boolean isUndo, boolean isTableMarker, String script, String installedBy, int executionTime, boolean success) {
            this.installedRank = installedRank;
            this.description = description;
            this.version = version;
            this.type = type;
            this.isUndo = isUndo;
            this.isTableMarker = isTableMarker;
            this.script = script;
            this.installedBy = installedBy;
            this.executionTime = executionTime;
            this.success = success;
        }

        public String toString() {
            return this.type + "{version=" + this.version + ",description=" + this.description + ",success=" + this.success + "}";
        }
    }

    private static final class ResultSetReader {
        private final ResultSet rs;
        private final Map<String, Integer> columnPositions;

        ResultSetReader(ResultSet rs) throws SQLException {
            this.rs = rs;
            ResultSetMetaData meta = rs.getMetaData();
            int colCount = meta.getColumnCount();
            this.columnPositions = new HashMap<String, Integer>(colCount, 2.0f);
            for (int pos = 1; pos <= colCount; ++pos) {
                this.columnPositions.put(meta.getColumnLabel(pos).toLowerCase(Locale.ROOT), pos);
            }
        }

        boolean requireBoolean(String column) throws SQLException {
            boolean result = this.rs.getBoolean(this.positionOf(column));
            if (this.rs.wasNull()) {
                throw this.columnWasNull(column);
            }
            return result;
        }

        Optional<Integer> readInt(String column) throws SQLException {
            int result = this.rs.getInt(this.positionOf(column));
            if (this.rs.wasNull()) {
                return Optional.empty();
            }
            return Optional.of(result);
        }

        int requireInt(String column) throws SQLException {
            return this.readInt(column).orElseThrow(() -> this.columnWasNull(column));
        }

        Optional<String> readString(String column) throws SQLException {
            return Optional.ofNullable(this.rs.getString(this.positionOf(column)));
        }

        String requireString(String column) throws SQLException {
            return this.readString(column).orElseThrow(() -> this.columnWasNull(column));
        }

        private int positionOf(String lowerCaseLabel) {
            Integer position = this.columnPositions.get(lowerCaseLabel);
            if (position == null) {
                throw new MigrateDbException("Conversion failed: Column '" + lowerCaseLabel + "' not present in old schema history table");
            }
            return position;
        }

        private MigrateDbException columnWasNull(String column) {
            return new MigrateDbException("Conversion failed: Value of '" + column + "' was null");
        }
    }
}

