/*
 * Decompiled with CFR 0.152.
 */
package migratedb.v1.commandline;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Console;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Reader;
import java.lang.invoke.CallSite;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.NotDirectoryException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import migratedb.v1.commandline.Arguments;
import migratedb.v1.commandline.CommandLineConfigKey;
import migratedb.v1.commandline.DownloadDriversCommand;
import migratedb.v1.commandline.DriverSupport;
import migratedb.v1.commandline.DriverSupportDataSource;
import migratedb.v1.commandline.DriverSupportRegistry;
import migratedb.v1.commandline.EnvironmentMapper;
import migratedb.v1.commandline.ErrorOutput;
import migratedb.v1.commandline.LogLevel;
import migratedb.v1.core.MigrateDb;
import migratedb.v1.core.api.ConnectionProvider;
import migratedb.v1.core.api.ErrorCode;
import migratedb.v1.core.api.MigrateDbException;
import migratedb.v1.core.api.MigrateDbExtension;
import migratedb.v1.core.api.MigrationInfo;
import migratedb.v1.core.api.MigrationInfoService;
import migratedb.v1.core.api.Version;
import migratedb.v1.core.api.configuration.Configuration;
import migratedb.v1.core.api.configuration.DefaultConfiguration;
import migratedb.v1.core.api.logging.Log;
import migratedb.v1.core.api.output.CompositeResult;
import migratedb.v1.core.api.output.OperationResult;
import migratedb.v1.core.internal.configuration.ConfigUtils;
import migratedb.v1.core.internal.info.MigrationInfoDumper;
import migratedb.v1.core.internal.util.ClassUtils;
import migratedb.v1.core.internal.util.StringUtils;
import org.checkerframework.checker.nullness.qual.MonotonicNonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.yaml.snakeyaml.Yaml;

class MigrateDbCommand {
    private static final Log LOG = Log.getLog(MigrateDbCommand.class);
    public static final String CONFIG_FILE_NAME = "migratedb.conf";
    private final Arguments arguments;
    private final Console console;
    private final PrintStream stdout;
    private final PrintStream stderr;
    private final @Nullable InputStream stdin;
    private final Map<String, String> environment;
    private final FileSystem fileSystem;
    private final Path installationDir;
    private final Path driversDir;
    private final Path configDir;
    private @MonotonicNonNull Supplier<MigrateDb> migrateDb;

    MigrateDbCommand(Arguments arguments, @Nullable Console console, PrintStream stdout, PrintStream stderr, @Nullable InputStream stdin, Map<String, String> environment) {
        this.arguments = arguments;
        this.console = console;
        this.stdout = stdout;
        this.stderr = stderr;
        this.stdin = stdin;
        this.environment = environment;
        this.fileSystem = arguments.getFileSystem();
        this.installationDir = this.fileSystem.getPath(arguments.getInstallationDirectory(), new String[0]);
        this.driversDir = this.installationDir.resolve("drivers");
        this.configDir = this.installationDir.resolve("conf");
    }

    int run() throws Exception {
        String currentOperation = null;
        try {
            OperationResult result;
            this.arguments.validate();
            if (this.arguments.shouldPrintVersionAndExit()) {
                this.printVersion();
                return 0;
            }
            if (this.arguments.hasOperation("help") || this.arguments.shouldPrintUsage()) {
                this.printUsage();
                return 0;
            }
            if (this.arguments.getOperations().size() == 1) {
                currentOperation = this.arguments.getOperations().get(0);
                result = this.executeOperation(currentOperation);
            } else {
                CompositeResult compositeResult = new CompositeResult();
                Iterator<String> iterator = this.arguments.getOperations().iterator();
                while (iterator.hasNext()) {
                    String operation;
                    currentOperation = operation = iterator.next();
                    OperationResult individualResult = this.executeOperation(operation);
                    compositeResult.individualResults.add(individualResult);
                }
                result = compositeResult;
            }
            if (this.arguments.shouldOutputJson()) {
                this.printJson(result);
            }
            return 0;
        }
        catch (Exception e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            if (this.arguments.shouldOutputJson()) {
                this.printJson(this.unhandledExceptionErrorOutput(currentOperation, e));
            } else if (this.arguments.getLogLevel() == LogLevel.DEBUG) {
                LOG.error("Unexpected error", e);
            } else {
                LOG.error(this.getMessagesFromException(e));
            }
            return 1;
        }
    }

    private OperationResult unhandledExceptionErrorOutput(@Nullable String currentOperation, Exception exception) {
        String message = exception.getMessage();
        if (exception instanceof MigrateDbException) {
            MigrateDbException migratedbException = (MigrateDbException)((Object)exception);
            return new ErrorOutput(migratedbException.getErrorCode(), message == null ? "Error occurred" : message, this.arguments.getLogLevel() == LogLevel.DEBUG ? MigrateDbCommand.getStackTrace(exception) : null, currentOperation);
        }
        return new ErrorOutput(ErrorCode.FAULT, message == null ? "Fault occurred" : message, MigrateDbCommand.getStackTrace(exception), currentOperation);
    }

    private static String getStackTrace(Exception exception) {
        ByteArrayOutputStream output = new ByteArrayOutputStream(4096);
        PrintStream printStream = new PrintStream((OutputStream)output, true, StandardCharsets.UTF_8);
        exception.printStackTrace(printStream);
        return output.toString(StandardCharsets.UTF_8);
    }

    private MigrateDb createMigrateDb() throws IOException {
        Map<String, String> envVars = this.environmentVariablesToPropertyMap();
        Map<String, String> configProps = new HashMap<String, String>();
        this.initializeDefaults(configProps);
        this.loadConfigurationFromConfigFiles(configProps, envVars);
        configProps.putAll(envVars);
        configProps = this.overrideConfiguration(configProps, this.arguments.getConfiguration());
        String url = configProps.get("migratedb.url");
        String user = configProps.get("migratedb.user");
        String password = configProps.get("migratedb.password");
        String driverClassName = configProps.get("migratedb.driver");
        if (this.arguments.isWorkingDirectorySet()) {
            this.makeRelativeLocationsBasedOnWorkingDirectory(configProps);
        }
        ClassLoader classLoader = this.createClassLoader(configProps);
        List spiExtensions = ServiceLoader.load(MigrateDbExtension.class, classLoader).stream().map(ServiceLoader.Provider::get).collect(Collectors.toList());
        DriverSupport driverSupport = this.getDriverSupport(url, classLoader);
        if (!this.arguments.shouldSuppressPrompt()) {
            this.promptForCredentialsIfMissing(configProps, driverSupport);
        }
        this.dumpConfiguration(configProps, driverSupport);
        Map<String, String> commandLineProperties = this.filterProperties(configProps);
        DriverSupportDataSource dataSource = new DriverSupportDataSource(classLoader, driverClassName, url, user, password, CommandLineConfigKey.getJdbcProperties(commandLineProperties), driverSupport);
        DefaultConfiguration configuration = new DefaultConfiguration(classLoader);
        configuration.configure(configProps);
        configuration.useExtensions(spiExtensions);
        configuration.setDataSource((ConnectionProvider)dataSource);
        return new MigrateDb((Configuration)configuration);
    }

    private ClassLoader createClassLoader(Map<String, String> configProps) throws IOException {
        ClassLoader classLoader = ClassUtils.defaultClassLoader();
        ArrayList<Path> jarFiles = new ArrayList<Path>();
        jarFiles.addAll(this.getJdbcDriverJarFiles());
        jarFiles.addAll(this.getJavaMigrationJarFiles(configProps));
        if (!jarFiles.isEmpty()) {
            classLoader = this.addJarsOrDirectoriesToClasspath(classLoader, jarFiles);
        }
        return classLoader;
    }

    private DriverSupport getDriverSupport(String url, ClassLoader classLoader) {
        DriverSupportRegistry registry = new DriverSupportRegistry();
        ServiceLoader.load(DriverSupport.class, classLoader).forEach(registry::register);
        return registry.getDriverSupportForUrl(url);
    }

    private Map<String, String> environmentVariablesToPropertyMap() {
        HashMap<String, String> result = new HashMap<String, String>();
        for (Map.Entry<String, String> entry : this.environment.entrySet()) {
            String convertedKey = EnvironmentMapper.convertKey(entry.getKey());
            if (convertedKey == null) continue;
            result.put(EnvironmentMapper.convertKey(entry.getKey()), this.expandEnvironmentVariables(entry.getValue()));
        }
        return result;
    }

    private String expandEnvironmentVariables(String value) {
        Pattern pattern = Pattern.compile("\\$\\{(\\w+)}");
        Matcher matcher = pattern.matcher(value);
        String expandedValue = value;
        while (matcher.find()) {
            String variableName = matcher.group(1);
            String variableValue = this.environment.getOrDefault(variableName, "");
            LOG.debug("Expanding environment variable in config: " + variableName + " -> " + variableValue);
            expandedValue = expandedValue.replaceAll(Pattern.quote(matcher.group(0)), Matcher.quoteReplacement(variableValue));
        }
        return expandedValue;
    }

    private ClassLoader addJarsOrDirectoriesToClasspath(ClassLoader classLoader, List<Path> jarFiles) {
        ArrayList<URL> urls = new ArrayList<URL>();
        for (Path jarFile : jarFiles) {
            try {
                urls.add(jarFile.toUri().toURL());
            }
            catch (RuntimeException | MalformedURLException e) {
                throw new MigrateDbException("Unable to load " + jarFile, (Throwable)e);
            }
        }
        return new URLClassLoader(urls.toArray(new URL[0]), classLoader);
    }

    private void dumpConfiguration(Map<String, String> config, DriverSupport driverSupport) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Using configuration:");
            for (Map.Entry<String, String> entry : new TreeMap<String, String>(config).entrySet()) {
                String key = entry.getKey();
                String value = entry.getValue();
                if (key.toLowerCase(Locale.ROOT).endsWith("password")) {
                    value = StringUtils.trimOrPad((String)"", (int)value.length(), (char)'*');
                } else if (key.equals("migratedb.url")) {
                    value = driverSupport.redactJdbcUrl(value);
                }
                LOG.debug(key + " -> " + value);
            }
        }
    }

    private Map<String, String> loadDefaultConfigurationFiles(String encoding) {
        HashMap<String, String> configMap = new HashMap<String, String>();
        configMap.putAll(this.loadConfigurationFile(this.configDir.resolve(CONFIG_FILE_NAME), encoding, false));
        configMap.putAll(this.loadConfigurationFile(this.fileSystem.getPath(System.getProperty("user.home"), CONFIG_FILE_NAME), encoding, false));
        configMap.putAll(this.loadConfigurationFile(this.fileSystem.getPath(CONFIG_FILE_NAME, new String[0]), encoding, false));
        return configMap;
    }

    private Map<String, String> loadConfigurationFile(Path configFile, String encoding, boolean failIfMissing) throws MigrateDbException {
        String errorMessage = "Unable to load config file: " + configFile.toAbsolutePath();
        if ("-".equals(configFile.getFileName().toString())) {
            return this.loadConfigurationFromStdin();
        }
        if (!Files.isRegularFile(configFile, new LinkOption[0]) || !Files.isReadable(configFile)) {
            if (!failIfMissing) {
                LOG.debug(errorMessage);
                return new HashMap<String, String>();
            }
            throw new MigrateDbException(errorMessage);
        }
        LOG.debug("Loading config file: " + configFile.toAbsolutePath());
        try {
            return ConfigUtils.loadConfiguration((Reader)new InputStreamReader(Files.newInputStream(configFile, new OpenOption[0]), encoding));
        }
        catch (IOException | MigrateDbException e) {
            throw new MigrateDbException(errorMessage, e);
        }
    }

    private Map<String, String> loadConfigurationFromStdin() {
        HashMap<String, String> config;
        block10: {
            config = new HashMap<String, String>();
            try {
                if (this.stdin == null) break block10;
                try (Reader reader = this.waitForStdin();){
                    LOG.debug("Attempting to load configuration from standard input");
                    Map configFromStdin = ConfigUtils.loadConfiguration((Reader)reader);
                    if (configFromStdin.isEmpty()) {
                        LOG.warn("Configuration from standard input is empty");
                    }
                    config.putAll(configFromStdin);
                }
            }
            catch (IOException | InterruptedException | RuntimeException | ExecutionException e) {
                if (e instanceof InterruptedException) {
                    Thread.currentThread().interrupt();
                }
                LOG.debug("Could not load configuration from standard input " + e.getMessage());
            }
        }
        return config;
    }

    private Reader waitForStdin() throws ExecutionException, InterruptedException {
        ExecutorService exec = Executors.newSingleThreadExecutor(r -> {
            Thread thread = new Thread(r);
            thread.setName("Waiting for STDIN");
            thread.setDaemon(true);
            return thread;
        });
        InputStream stream = this.stdin;
        assert (stream != null);
        Future<InputStreamReader> future = exec.submit(() -> {
            if (stream.available() == 0) {
                BufferedInputStream markSupport = new BufferedInputStream(stream);
                markSupport.mark(2);
                if (markSupport.read() == -1) {
                    throw new MigrateDbException("No input provided");
                }
                markSupport.reset();
                return new InputStreamReader((InputStream)markSupport, StandardCharsets.UTF_8);
            }
            return new InputStreamReader(stream, StandardCharsets.UTF_8);
        });
        try {
            return future.get(10L, TimeUnit.SECONDS);
        }
        catch (TimeoutException e) {
            throw new MigrateDbException("Timeout while waiting for STDIN");
        }
    }

    private void makeRelativeLocationsBasedOnWorkingDirectory(Map<String, String> config) {
        CharSequence[] locations = config.get("migratedb.locations").split(",");
        for (int i = 0; i < locations.length; ++i) {
            if (!locations[i].startsWith("filesystem:")) continue;
            String newLocation = ((String)locations[i]).substring("filesystem:".length());
            Path file = this.fileSystem.getPath(newLocation, new String[0]);
            if (!file.isAbsolute()) {
                file = this.fileSystem.getPath(this.arguments.getWorkingDirectory(), newLocation);
            }
            locations[i] = "filesystem:" + file.toAbsolutePath();
        }
        config.put("migratedb.locations", String.join((CharSequence)",", locations));
    }

    private Map<String, String> overrideConfiguration(Map<String, String> existingConfiguration, Map<String, String> newConfiguration) {
        HashMap<String, String> combinedConfiguration = new HashMap<String, String>(existingConfiguration);
        combinedConfiguration.putAll(newConfiguration);
        return combinedConfiguration;
    }

    private String getMessagesFromException(Throwable e) {
        StringBuilder condensedMessages = new StringBuilder();
        String preamble = "";
        while (e != null) {
            if (e instanceof MigrateDbException) {
                condensedMessages.append(preamble).append(e.getMessage());
            } else {
                condensedMessages.append(preamble).append(e);
            }
            preamble = "\nCaused by: ";
            e = e.getCause();
        }
        return condensedMessages.toString();
    }

    private @Nullable OperationResult executeOperation(String operation) throws IOException {
        Object result = null;
        if ("download-drivers".equals(operation)) {
            result = new DownloadDriversCommand(MigrateDbCommand.parseDriverDefinitions(this.configDir.resolve("drivers.yaml")), this.driversDir, this.arguments.getDriverNames()).run();
            this.migrateDb = MigrateDbCommand.lazy(this::createMigrateDb);
            return result;
        }
        if (this.migrateDb == null) {
            this.migrateDb = MigrateDbCommand.lazy(this::createMigrateDb);
        }
        if ("baseline".equals(operation)) {
            result = this.migrateDb.get().baseline();
        } else if ("migrate".equals(operation)) {
            result = this.migrateDb.get().migrate();
        } else if ("validate".equals(operation)) {
            if (this.arguments.shouldOutputJson()) {
                result = this.migrateDb.get().validateWithResult();
            } else {
                this.migrateDb.get().validate();
            }
        } else if ("info".equals(operation)) {
            Version currentSchemaVersion;
            MigrationInfoService info = this.migrateDb.get().info();
            MigrationInfo current = info.current();
            Version version = currentSchemaVersion = current == null ? null : current.getVersion();
            if (!this.arguments.shouldOutputJson()) {
                this.stdout.println("Schema version: " + (Comparable)((Object)(currentSchemaVersion == null ? "<< Empty Schema >>" : currentSchemaVersion)));
                this.stdout.println();
                this.stdout.println(MigrationInfoDumper.dumpToAsciiTable((MigrationInfo[])info.all()));
            }
            result = info.getInfoResult();
        } else if ("repair".equals(operation)) {
            result = this.migrateDb.get().repair();
        } else if ("liberate".equals(operation)) {
            result = this.migrateDb.get().liberate();
        } else {
            LOG.error("Invalid operation: " + operation);
            this.printUsage();
            throw new MigrateDbException("Invalid operation");
        }
        return result;
    }

    private void printJson(@Nullable OperationResult object) throws IOException {
        String json = this.convertObjectToJsonString(object);
        byte[] bytes = json.getBytes(StandardCharsets.UTF_8);
        if (this.arguments.isOutputFileSet()) {
            Path path = this.fileSystem.getPath(this.arguments.getOutputFile(), new String[0]);
            try {
                Files.write(path, bytes, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.CREATE);
            }
            catch (IOException e) {
                throw new MigrateDbException("Could not write to output file " + this.arguments.getOutputFile(), (Throwable)e);
            }
        }
        this.stdout.write(bytes);
    }

    private String convertObjectToJsonString(@Nullable Object object) {
        Gson gson = new GsonBuilder().setPrettyPrinting().disableHtmlEscaping().serializeNulls().create();
        return gson.toJson(object);
    }

    private void initializeDefaults(Map<String, String> config) {
        String workingDirectory = this.arguments.isWorkingDirectorySet() ? this.arguments.getWorkingDirectory() : this.installationDir.toString();
        config.put("migratedb.locations", "filesystem:" + this.fileSystem.getPath(workingDirectory, "sql").toAbsolutePath());
        config.put("migratedb.jarDirs", this.fileSystem.getPath(workingDirectory, "jars").toAbsolutePath().toString());
    }

    private Map<String, String> filterProperties(Map<String, String> config) {
        HashMap<String, String> result = new HashMap<String, String>();
        for (String key : List.of("migratedb.jarDirs", "migratedb.configFiles", "migratedb.configFileEncoding", "migratedb.url", "migratedb.user", "migratedb.password", "migratedb.driver")) {
            String removed = config.remove(key);
            if (removed == null) continue;
            result.put(key, removed);
        }
        Iterator<Map.Entry<String, String>> entries = config.entrySet().iterator();
        while (entries.hasNext()) {
            Map.Entry<String, String> entry = entries.next();
            if (!entry.getKey().startsWith("migratedb.jdbcProperties.")) continue;
            result.put(entry.getKey(), entry.getValue());
            entries.remove();
        }
        return result;
    }

    private void printVersion() throws IOException {
        List<CallSite> lines = List.of("MigrateDB version 1.2.0", "Java " + System.getProperty("java.version") + " (" + System.getProperty("java.vendor") + ")", System.getProperty("os.name") + " " + System.getProperty("os.version") + " " + System.getProperty("os.arch") + "\n");
        if (this.arguments.shouldOutputJson()) {
            this.printJson(new ErrorOutput(ErrorCode.CLI_USAGE, String.join((CharSequence)"\n", lines), null, null));
        } else {
            lines.forEach(this.stdout::println);
        }
    }

    private void printUsage() throws IOException {
        List<String> usageLines = List.of("Usage", "=====", "", "migratedb [options] command", "", "By default, the configuration will be read from conf/migratedb.conf.", "Options passed from the command-line override the configuration.", "", "Commands", "--------", "migrate  : Migrates the database", "info     : Prints the information about applied, current and pending migrations", "validate : Validates the applied migrations against the ones on the classpath", "baseline : Baselines an existing database at the baselineVersion", "repair   : Repairs the schema history table", "repair   : Repairs the schema history table", "liberate : Converts from Flyway to MigrateDB", "", "Options (Format: -key=value)", "-------", "driver                       : Fully qualified classname of the JDBC driver", "url                          : JDBC url to use to connect to the database", "user                         : User to use to connect to the database", "password                     : Password to use to connect to the database", "connectRetries               : Maximum number of retries when attempting to connect to the database", "initSql                      : SQL statements to run to initialize a new database connection", "schemas                      : Comma-separated list of the schemas managed by MigrateDb", "table                        : Name of MigrateDB's schema history table", "oldTable                     : Name of previous schema history table", "locations                    : Classpath locations to scan recursively for migrations", "failOnMissingLocations       : Whether to fail if a location specified in the migratedb.locations option doesn't exist", "resolvers                    : Comma-separated list of custom MigrationResolvers", "skipDefaultResolvers         : Skips default resolvers (jdbc, sql and Spring-jdbc)", "sqlMigrationPrefix           : File name prefix for versioned SQL migrations", "repeatableSqlMigrationPrefix : File name prefix for repeatable SQL migrations", "sqlMigrationSeparator        : File name separator for SQL migrations", "sqlMigrationSuffixes         : Comma-separated list of file name suffixes for SQL migrations", "mixed                        : Allow mixing transactional and non-transactional statements", "encoding                     : Encoding of SQL migrations", "placeholderReplacement       : Whether placeholders should be replaced", "placeholders                 : Placeholders to replace in sql migrations", "placeholderPrefix            : Prefix of every placeholder", "placeholderSuffix            : Suffix of every placeholder", "scriptPlaceholderPrefix      : Prefix of every script placeholder", "scriptPlaceholderSuffix      : Suffix of every script placeholder", "lockRetryCount               : The maximum number of retries when trying to obtain a lock", "jdbcProperties               : Properties to pass to the JDBC driver object", "installedBy                  : Username that will be recorded in the schema history table", "target                       : Target version up to which MigrateDB should use migrations", "outOfOrder                   : Allows migrations to be run \"out of order\"", "callbacks                    : Comma-separated list of MigrateDbCallback classes", "skipDefaultCallbacks         : Skips default callbacks (sql)", "validateOnMigrate            : Validate when running migrate", "validateMigrationNaming      : Validate file names of SQL migrations (including callbacks)", "ignoreMissingMigrations      : Allow missing migrations when validating", "ignoreIgnoredMigrations      : Allow ignored migrations when validating", "ignorePendingMigrations      : Allow pending migrations when validating", "ignoreFutureMigrations       : Allow future migrations when validating", "baselineVersion              : Version to tag schema with when executing baseline", "baselineDescription          : Description to tag schema with when executing baseline", "baselineOnMigrate            : Baseline on migrate against uninitialized non-empty schema", "configFiles                  : Comma-separated list of config files to use", "configFileEncoding           : Encoding to use when loading the config files", "jarDirs                      : Comma-separated list of dirs for Jdbc drivers & Java migrations", "createSchemas                : Whether MigrateDB should attempt to create the schemas specified in the schemas property", "outputFile                   : Send output to the specified file alongside the console", "outputType                   : Serialise the output in the given format, Values: json", "", "Flags", "-----", "-X              : Print debug output", "-q              : Suppress all output, except for errors and warnings", "-n              : Suppress prompting for a user and password", "--version, -v   : Print the MigrateDB version and exit", "--help, -h, -?  : Print this usage info and exit", "", "Example", "-------", "migratedb -user=myuser -password=s3cr3t -url=jdbc:h2:mem -placeholders.abc=def migrate");
        if (this.arguments.shouldOutputJson()) {
            this.printJson(new ErrorOutput(ErrorCode.CLI_USAGE, String.join((CharSequence)"\n", usageLines), null, null));
        } else {
            usageLines.forEach(this.stdout::println);
        }
    }

    private List<Path> getJdbcDriverJarFiles() throws IOException {
        List<Path> list;
        block8: {
            Stream<Path> fileStream = Files.list(this.driversDir);
            try {
                list = fileStream.filter(it -> it.getFileName().toString().endsWith(".jar")).collect(Collectors.toList());
                if (fileStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (fileStream != null) {
                        try {
                            fileStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (NoSuchFileException | NotDirectoryException e) {
                    LOG.warn("Directory for JDBC Drivers not found: " + this.driversDir.toAbsolutePath());
                    return Collections.emptyList();
                }
            }
            fileStream.close();
        }
        return list;
    }

    private List<Path> getJavaMigrationJarFiles(Map<String, String> config) throws IOException {
        String jarDirs = config.get("migratedb.jarDirs");
        if (!StringUtils.hasLength((String)jarDirs)) {
            return Collections.emptyList();
        }
        jarDirs = jarDirs.replace(File.pathSeparator, ",");
        String[] dirs = StringUtils.tokenizeToStringArray((String)jarDirs, (String)",");
        ArrayList<Path> jarFiles = new ArrayList<Path>();
        for (String dirName : dirs) {
            try (Stream<Path> fileStream = Files.list(this.fileSystem.getPath(dirName, new String[0]));){
                jarFiles.addAll(fileStream.filter(it -> it.getFileName().toString().endsWith(".jar")).collect(Collectors.toList()));
            }
            catch (NoSuchFileException | NotDirectoryException e) {
                LOG.warn("Directory for Java migrations not found: " + dirName);
            }
        }
        return jarFiles;
    }

    private void loadConfigurationFromConfigFiles(Map<String, String> config, Map<String, String> envVars) {
        String encoding = this.determineConfigurationFileEncoding(this.arguments, envVars);
        config.putAll(this.loadDefaultConfigurationFiles(encoding));
        for (Path configFile : this.determineConfigFilesFromArgs(this.arguments, envVars)) {
            config.putAll(this.loadConfigurationFile(configFile, encoding, true));
        }
    }

    private void promptForCredentialsIfMissing(Map<String, String> config, DriverSupport driverSupport) {
        if (this.console == null) {
            return;
        }
        if (!config.containsKey("migratedb.url")) {
            return;
        }
        String url = config.get("migratedb.url");
        if (!config.containsKey("migratedb.user") && this.needsUser(url, driverSupport)) {
            config.put("migratedb.user", this.console.readLine("Database user: ", new Object[0]));
        }
        if (!config.containsKey("migratedb.password") && this.needsPassword(url, driverSupport)) {
            char[] password = this.console.readPassword("Database password: ", new Object[0]);
            config.put("migratedb.password", password == null ? "" : String.valueOf(password));
        }
    }

    private boolean needsUser(String url, DriverSupport driverSupport) {
        return driverSupport.detectUserRequiredByUrl(url);
    }

    private boolean needsPassword(String url, DriverSupport driverSupport) {
        return driverSupport.detectPasswordRequiredByUrl(url);
    }

    private List<Path> determineConfigFilesFromArgs(Arguments arguments, Map<String, String> envVars) {
        String workingDirectory;
        ArrayList<Path> configFiles = new ArrayList<Path>();
        String string = workingDirectory = arguments.isWorkingDirectorySet() ? arguments.getWorkingDirectory() : ".";
        if (envVars.containsKey("migratedb.configFiles")) {
            for (String file : StringUtils.tokenizeToStringArray((String)envVars.get("migratedb.configFiles"), (String)",")) {
                configFiles.add(this.fileSystem.getPath(workingDirectory, file));
            }
            return configFiles;
        }
        for (String file : arguments.getConfigFiles()) {
            configFiles.add(this.fileSystem.getPath(workingDirectory, file));
        }
        return configFiles;
    }

    private String determineConfigurationFileEncoding(Arguments arguments, Map<String, String> envVars) {
        if (envVars.containsKey("migratedb.configFileEncoding")) {
            return envVars.get("migratedb.configFileEncoding");
        }
        if (arguments.isConfigFileEncodingSet()) {
            return arguments.getConfigFileEncoding();
        }
        return "UTF-8";
    }

    private static <T> Supplier<T> lazy(final Callable<T> factory) {
        return new Supplier<T>(){
            private @MonotonicNonNull T lazyValue;

            @Override
            public T get() {
                if (this.lazyValue == null) {
                    try {
                        this.lazyValue = factory.call();
                    }
                    catch (Exception e) {
                        if (e instanceof InterruptedException) {
                            Thread.currentThread().interrupt();
                        }
                        throw new MigrateDbException((Throwable)e);
                    }
                }
                return this.lazyValue;
            }
        };
    }

    private static DownloadDriversCommand.DriverDefinitions parseDriverDefinitions(Path file) {
        DownloadDriversCommand.DriverDefinitions driverDefinitions;
        BufferedInputStream stream = new BufferedInputStream(Files.newInputStream(file, new OpenOption[0]));
        try {
            driverDefinitions = (DownloadDriversCommand.DriverDefinitions)new Yaml().loadAs((InputStream)stream, DownloadDriversCommand.DriverDefinitions.class);
        }
        catch (Throwable throwable) {
            try {
                try {
                    stream.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new MigrateDbException("Cannot parse driver definitions from '" + file + "'");
            }
        }
        stream.close();
        return driverDefinitions;
    }
}

