/*
 * Decompiled with CFR 0.152.
 */
package migratedb.v1.commandline;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import migratedb.v1.core.api.MigrateDbException;
import migratedb.v1.core.api.logging.Log;
import migratedb.v1.core.api.output.OperationResult;
import migratedb.v1.core.internal.util.StringUtils;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.text.StringSubstitutor;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.nodes.Tag;

public class DownloadDriversCommand {
    private final DriverDefinitions driverDefinitions;
    private final Set<String> driversToDownload;
    private final Path targetDir;
    private final HttpClient httpClient;
    private static final Log LOG = Log.getLog(DownloadDriversCommand.class);

    DownloadDriversCommand(DriverDefinitions driverDefinitions, Path targetDir, Collection<String> driversToDownload) {
        this.driverDefinitions = driverDefinitions;
        this.driversToDownload = Set.copyOf(driversToDownload);
        this.targetDir = targetDir;
        this.httpClient = HttpClient.newBuilder().version(HttpClient.Version.HTTP_1_1).connectTimeout(Duration.ofSeconds(10L)).build();
    }

    Result run() {
        DownloadDriversCommand.resolvePlaceholders(this.driverDefinitions);
        Set<String> downloadedDrivers = this.driversToDownload.isEmpty() ? this.allSupportedDriverAliases() : this.driversToDownload;
        List driverDefinitions = downloadedDrivers.stream().map(this::findDriverDefinition).collect(Collectors.toList());
        for (DriverDefinition driverDefinition : driverDefinitions) {
            LOG.info("Processing artifacts for '" + driverDefinition.name + "'");
            this.downloadArtifacts(this.driverDefinitions.repo, driverDefinition);
        }
        LOG.info("Successfully downloaded " + downloadedDrivers.size() + " drivers");
        return new Result(downloadedDrivers);
    }

    private void downloadArtifacts(String repo, DriverDefinition driverDefinition) {
        for (ArtifactDefinition artifact : driverDefinition.artifacts) {
            JarCoordinates coordinates = new JarCoordinates(artifact.m2);
            Path targetFile = this.targetDir.resolve(coordinates.targetFileName());
            if (Files.isRegularFile(targetFile, new LinkOption[0])) {
                LOG.info("Already exists: '" + targetFile.getFileName() + "'");
                try {
                    this.validate(targetFile, artifact.sha256);
                    continue;
                }
                catch (RuntimeException e) {
                    LOG.info("Please delete '" + targetFile.getFileName() + "' and try again");
                    throw e;
                }
            }
            this.downloadFromRemoteRepository(repo, artifact, coordinates, targetFile);
        }
    }

    private void downloadFromRemoteRepository(String repo, ArtifactDefinition artifact, JarCoordinates coordinates, Path targetFile) {
        Path tempFile = targetFile.resolveSibling(targetFile.getFileName() + ".tmp");
        try {
            Files.deleteIfExists(tempFile);
            this.httpGet(coordinates.toUrl(repo), tempFile, artifact.sha256);
            try {
                Files.move(tempFile, targetFile, StandardCopyOption.ATOMIC_MOVE);
            }
            catch (AtomicMoveNotSupportedException e) {
                Files.move(tempFile, targetFile, StandardCopyOption.REPLACE_EXISTING);
            }
        }
        catch (URISyntaxException e) {
            throw new MigrateDbException("Invalid download URL for artifact '" + artifact.m2 + "'", (Throwable)e);
        }
        catch (IOException e) {
            throw new MigrateDbException("Unable to fetch driver artifact '" + artifact.m2 + "'", (Throwable)e);
        }
        finally {
            try {
                Files.deleteIfExists(tempFile);
            }
            catch (IOException e) {
                LOG.warn("Failed to remove temporary file '" + tempFile + "'");
            }
        }
    }

    private NavigableSet<String> allSupportedDriverAliases() {
        return this.driverDefinitions.drivers.stream().map(it -> it.alias).collect(Collectors.toCollection(TreeSet::new));
    }

    private void httpGet(URI url, Path toFile, String sha256) throws IOException {
        LOG.info("Downloading '" + url + "'");
        try {
            HttpResponse<Path> response;
            HttpRequest request = HttpRequest.newBuilder(url).GET().timeout(Duration.ofMinutes(10L)).build();
            if (toFile.getParent() != null) {
                Files.createDirectories(toFile.getParent(), new FileAttribute[0]);
            }
            if ((response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofFile(toFile))).statusCode() != 200) {
                throw new MigrateDbException("GET request to '" + url + "' failed with status code " + response.statusCode());
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new MigrateDbException("Interrupted");
        }
        this.validate(toFile, sha256);
    }

    private void validate(Path targetFile, String sha256) {
        LOG.info("Validating '" + targetFile.getFileName() + "'");
        MessageDigest digest = DownloadDriversCommand.newSha256();
        byte[] buf = new byte[4096];
        try (DigestInputStream stream = new DigestInputStream(Files.newInputStream(targetFile, new OpenOption[0]), digest);){
            while (stream.read(buf) > 0) {
            }
        }
        catch (IOException e) {
            throw new MigrateDbException("Failed to validate '" + targetFile + "'", (Throwable)e);
        }
        try {
            byte[] actual = digest.digest();
            byte[] expected = Hex.decodeHex((String)sha256);
            if (!Arrays.equals(expected, actual)) {
                throw new MigrateDbException("Hash mismatch for file '" + targetFile + "'\n   Actual:   '" + Hex.encodeHexString((byte[])actual) + "'\n   Expected: '" + sha256 + "'\nPlease try again. If the error persists, the remote driver file may have been corrupted.");
            }
        }
        catch (DecoderException e) {
            throw new MigrateDbException("Not a hexadecimal value: '" + sha256 + "'");
        }
    }

    private DriverDefinition findDriverDefinition(String alias) {
        return this.driverDefinitions.drivers.stream().filter(it -> it.alias.equalsIgnoreCase(alias)).findFirst().orElseThrow(() -> new MigrateDbException("No such driver '" + alias + "'. The available drivers are: " + this.allSupportedDriverAliases()));
    }

    private static MessageDigest newSha256() {
        try {
            return MessageDigest.getInstance("SHA-256");
        }
        catch (NoSuchAlgorithmException e) {
            throw new AssertionError();
        }
    }

    private static void resolvePlaceholders(DriverDefinitions driverDefinitions) {
        StringSubstitutor subst = new StringSubstitutor(driverDefinitions.properties, "${", "}");
        subst.setEnableUndefinedVariableException(true);
        driverDefinitions.repo = subst.replace(driverDefinitions.repo);
        for (DriverDefinition driver : driverDefinitions.drivers) {
            driver.name = subst.replace(driver.name);
            for (ArtifactDefinition artifact : driver.artifacts) {
                artifact.m2 = subst.replace(artifact.m2);
            }
        }
    }

    public static void main(String[] args) {
        Path driversDotYaml = Paths.get(args[0], new String[0]);
        MessageDigest sha256 = DownloadDriversCommand.newSha256();
        HttpClient httpClient = HttpClient.newBuilder().version(HttpClient.Version.HTTP_1_1).connectTimeout(Duration.ofSeconds(10L)).build();
        try (BufferedInputStream stream = new BufferedInputStream(Files.newInputStream(driversDotYaml, new OpenOption[0]));){
            Yaml yaml = new Yaml();
            DriverDefinitions def1 = (DriverDefinitions)yaml.loadAs((InputStream)stream, DriverDefinitions.class);
            DriverDefinitions def2 = (DriverDefinitions)yaml.loadAs(yaml.dumpAs((Object)def1, Tag.MAP, DumperOptions.FlowStyle.AUTO), DriverDefinitions.class);
            DownloadDriversCommand.resolvePlaceholders(def2);
            Iterator<DriverDefinition> drivers1 = def1.drivers.iterator();
            Iterator<DriverDefinition> drivers2 = def2.drivers.iterator();
            while (drivers1.hasNext()) {
                DriverDefinition d1 = drivers1.next();
                DriverDefinition d2 = drivers2.next();
                Iterator<ArtifactDefinition> artifacts1 = d1.artifacts.iterator();
                Iterator<ArtifactDefinition> artifacts2 = d2.artifacts.iterator();
                while (artifacts1.hasNext()) {
                    String hash;
                    ArtifactDefinition a1 = artifacts1.next();
                    ArtifactDefinition a2 = artifacts2.next();
                    URI url = new JarCoordinates(a2.m2).toUrl(def2.repo);
                    System.out.println("Accepting current hash of");
                    System.out.print(url);
                    HttpRequest request = HttpRequest.newBuilder(url).GET().timeout(Duration.ofMinutes(10L)).build();
                    HttpResponse<InputStream> response = httpClient.send(request, HttpResponse.BodyHandlers.ofInputStream());
                    if (response.statusCode() != 200) {
                        throw new IOException(response.toString());
                    }
                    sha256.reset();
                    try (DigestInputStream digestStream = new DigestInputStream(response.body(), sha256);){
                        digestStream.transferTo(OutputStream.nullOutputStream());
                        hash = Hex.encodeHexString((byte[])sha256.digest(), (boolean)true);
                    }
                    System.out.println(" as " + hash);
                    a1.sha256 = hash;
                }
            }
            Path updatedYaml = driversDotYaml.resolveSibling(driversDotYaml.getFileName().toString() + ".updated");
            Files.writeString(updatedYaml, (CharSequence)yaml.dumpAs((Object)def1, Tag.MAP, DumperOptions.FlowStyle.BLOCK), new OpenOption[0]);
            System.out.println("Updated definitions at: " + updatedYaml.toAbsolutePath());
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    static class JarCoordinates {
        public final String groupId;
        public final String artifactId;
        public final String version;
        public final String extension = "jar";

        JarCoordinates(String coordinates) {
            String[] groupArtifactVersion = StringUtils.tokenizeToStringArray((String)coordinates, (String)":");
            if (groupArtifactVersion.length != 3) {
                throw new MigrateDbException("Unsupported artifact coordinates: '" + coordinates + "'");
            }
            this.groupId = groupArtifactVersion[0];
            this.artifactId = groupArtifactVersion[1];
            this.version = groupArtifactVersion[2];
        }

        String targetFileName() {
            return this.groupId + "." + this.artifactId + "-" + this.version + ".jar";
        }

        URI toUrl(String repo) throws URISyntaxException {
            URI repoUrl = new URI(repo);
            String newPath = repoUrl.getPath() + (repoUrl.getPath().endsWith("/") ? "" : "/") + this.groupId.replace('.', '/') + "/" + this.artifactId + "/" + this.version + "/" + this.artifactId + "-" + this.version + ".jar";
            return new URI(repoUrl.getScheme(), repoUrl.getUserInfo(), repoUrl.getHost(), repoUrl.getPort(), newPath, null, null);
        }
    }

    public static class DriverDefinition {
        public String name;
        public String alias;
        public List<ArtifactDefinition> artifacts;
    }

    public static class ArtifactDefinition {
        public String m2;
        public String sha256;
    }

    public static class DriverDefinitions {
        public Map<String, String> properties;
        public List<DriverDefinition> drivers;
        public String repo;
    }

    public static final class Result
    extends OperationResult {
        public Set<String> downloadedDrivers;

        Result(Set<String> downloadedDrivers) {
            this.downloadedDrivers = Set.copyOf(downloadedDrivers);
            this.migratedbVersion = "1.2.0";
            this.operation = "download-drivers";
        }
    }
}

