package de.undercouch.citeproc.output;

import java.util.Arrays;
import java.util.Map;
import java.util.Objects;

/**
 * A generated bibliography consisting of bibliography entries and some
 * formatting parameters
 * 
 * @author Michel Kraemer
 */
public class Bibliography {
    private final String[] entries;

    private final String bibStart;
    private final String bibEnd;
    private final String[] entryIds;
    private final Integer maxOffset;
    private final Integer entrySpacing;
    private final Integer lineSpacing;
    private final Boolean hangingIndent;
    private final Boolean done;
    private final SecondFieldAlign secondFieldAlign;

    public Bibliography(String... entries) {
        this.entries = entries;

        this.bibStart = null;
        this.bibEnd = null;
        this.entryIds = null;
        this.maxOffset = null;
        this.entrySpacing = null;
        this.lineSpacing = null;
        this.hangingIndent = null;
        this.done = null;
        this.secondFieldAlign = null;

    }

    public Bibliography(String[] entries, String bibStart, String bibEnd, String[] entryIds, Integer maxOffset,
            Integer entrySpacing, Integer lineSpacing, Boolean hangingIndent, Boolean done,
            SecondFieldAlign secondFieldAlign) {
        this.entries = entries;

        this.bibStart = bibStart;
        this.bibEnd = bibEnd;
        this.entryIds = entryIds;
        this.maxOffset = maxOffset;
        this.entrySpacing = entrySpacing;
        this.lineSpacing = lineSpacing;
        this.hangingIndent = hangingIndent;
        this.done = done;
        this.secondFieldAlign = secondFieldAlign;

    }

    /**
     * @return the bibliography's entries
     */
    public String[] getEntries() {
        return entries;
    }

    /**
     * @return the bibliography's bibstart
     */
    public String getBibStart() {
        return bibStart;
    }
    /**
     * @return the bibliography's bibend
     */
    public String getBibEnd() {
        return bibEnd;
    }
    /**
     * @return the bibliography's entry_ids
     */
    public String[] getEntryIds() {
        return entryIds;
    }
    /**
     * @return the bibliography's maxoffset
     */
    public Integer getMaxOffset() {
        return maxOffset;
    }
    /**
     * @return the bibliography's entryspacing
     */
    public Integer getEntrySpacing() {
        return entrySpacing;
    }
    /**
     * @return the bibliography's linespacing
     */
    public Integer getLineSpacing() {
        return lineSpacing;
    }
    /**
     * @return the bibliography's hangingindent
     */
    public Boolean getHangingIndent() {
        return hangingIndent;
    }
    /**
     * @return the bibliography's done
     */
    public Boolean getDone() {
        return done;
    }
    /**
     * @return the bibliography's second-field-align
     */
    public SecondFieldAlign getSecondFieldAlign() {
        return secondFieldAlign;
    }

    /**
     * Converts this bibliography to a String that can be inserted into the text
     * 
     * @return the String representing the bibliography
     */
    public String makeString() {
        StringBuilder b = new StringBuilder();
        if (getBibStart() != null)
            b.append(getBibStart());
        if (entries != null) {
            for (String e : entries)
                b.append(e);
        }
        if (getBibEnd() != null)
            b.append(getBibEnd());
        return b.toString();
    }

    @Override
    public int hashCode() {
        int result = 1;

        result = 31 * result + ((bibStart == null) ? 0 : bibStart.hashCode());
        result = 31 * result + ((bibEnd == null) ? 0 : bibEnd.hashCode());
        result = 31 * result + Arrays.hashCode(entryIds);
        result = 31 * result + ((maxOffset == null) ? 0 : maxOffset.hashCode());
        result = 31 * result + ((entrySpacing == null) ? 0 : entrySpacing.hashCode());
        result = 31 * result + ((lineSpacing == null) ? 0 : lineSpacing.hashCode());
        result = 31 * result + ((hangingIndent == null) ? 0 : hangingIndent.hashCode());
        result = 31 * result + ((done == null) ? 0 : done.hashCode());
        result = 31 * result + ((secondFieldAlign == null) ? 0 : secondFieldAlign.hashCode());
        result = 31 * result + Arrays.hashCode(entries);

        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof Bibliography))
            return false;
        Bibliography other = (Bibliography) obj;

        if (bibStart == null) {
            if (other.bibStart != null)
                return false;
        } else if (!bibStart.equals(other.bibStart))
            return false;

        if (bibEnd == null) {
            if (other.bibEnd != null)
                return false;
        } else if (!bibEnd.equals(other.bibEnd))
            return false;

        if (!Arrays.equals(entryIds, other.entryIds))
            return false;

        if (maxOffset == null) {
            if (other.maxOffset != null)
                return false;
        } else if (!maxOffset.equals(other.maxOffset))
            return false;

        if (entrySpacing == null) {
            if (other.entrySpacing != null)
                return false;
        } else if (!entrySpacing.equals(other.entrySpacing))
            return false;

        if (lineSpacing == null) {
            if (other.lineSpacing != null)
                return false;
        } else if (!lineSpacing.equals(other.lineSpacing))
            return false;

        if (hangingIndent == null) {
            if (other.hangingIndent != null)
                return false;
        } else if (!hangingIndent.equals(other.hangingIndent))
            return false;

        if (done == null) {
            if (other.done != null)
                return false;
        } else if (!done.equals(other.done))
            return false;

        if (secondFieldAlign == null) {
            if (other.secondFieldAlign != null)
                return false;
        } else if (!secondFieldAlign.equals(other.secondFieldAlign))
            return false;

        if (!Arrays.equals(entries, other.entries))
            return false;

        return true;
    }
}
