package de.undercouch.citeproc.csl;

import java.util.Arrays;
import java.util.Map;
import java.util.Objects;

import java.util.Collection;

import de.undercouch.citeproc.helper.json.JsonBuilder;
import de.undercouch.citeproc.helper.json.JsonObject;

/**
 * A citation date.
 * 
 * @author Michel Kraemer
 */
public class CSLDate implements JsonObject {

    private final int[][] dateParts;
    private final String season;
    private final Boolean circa;
    private final String literal;
    private final String raw;

    public CSLDate() {

        this.dateParts = null;
        this.season = null;
        this.circa = null;
        this.literal = null;
        this.raw = null;

    }

    public CSLDate(int[][] dateParts, String season, Boolean circa, String literal, String raw) {

        this.dateParts = dateParts;
        this.season = season;
        this.circa = circa;
        this.literal = literal;
        this.raw = raw;

    }

    /**
     * @return the date's date-parts
     */
    public int[][] getDateParts() {
        return dateParts;
    }
    /**
     * @return the date's season
     */
    public String getSeason() {
        return season;
    }
    /**
     * @return the date's circa
     */
    public Boolean getCirca() {
        return circa;
    }
    /**
     * @return the date's literal
     */
    public String getLiteral() {
        return literal;
    }
    /**
     * @return the date's raw
     */
    public String getRaw() {
        return raw;
    }

    @Override
    public Object toJson(JsonBuilder builder) {

        if (dateParts != null) {
            builder.add("date-parts", dateParts);
        }
        if (season != null) {
            builder.add("season", season);
        }
        if (circa != null) {
            builder.add("circa", circa);
        }
        if (literal != null) {
            builder.add("literal", literal);
        }
        if (raw != null) {
            builder.add("raw", raw);
        }

        return builder.build();
    }

    /**
     * Converts a JSON object to a CSLDate object.
     * 
     * @param obj
     *            the JSON object to convert
     * @return the converted CSLDate object
     */
    @SuppressWarnings("unchecked")
    public static CSLDate fromJson(Map<String, Object> obj) {

        {
            Object v = obj.get("date-parts");
            if (v instanceof Map)
                v = ((Map<?, ?>) v).values();
            if (v instanceof Collection) {
                Collection<?> cv = (Collection<?>) v;
                for (Object vo : cv) {
                    if (vo instanceof Map)
                        vo = ((Map<?, ?>) vo).values();
                    if (vo instanceof Collection) {
                        Collection<?> icv = (Collection<?>) vo;
                        java.util.Iterator<?> i = icv.iterator();
                        while (i.hasNext()) {
                            Object s = i.next();
                            if (s instanceof String && ((String) s).isEmpty()) {
                                i.remove();
                                while (i.hasNext()) {
                                    i.next();
                                    i.remove();
                                }
                            }
                        }
                    }
                }
            }
        }

        CSLDateBuilder builder = new CSLDateBuilder();

        {
            Object v = obj.get("date-parts");
            if (v != null) {
                int[][] dateParts;
                if (v instanceof Map) {
                    v = ((Map<?, ?>) v).values();
                } else if (!(v instanceof Collection)) {
                    throw new IllegalArgumentException("`date-parts' must be an array");
                }
                Collection<?> cv = (Collection<?>) v;
                dateParts = new int[cv.size()][];
                int i = 0;
                for (Object vo : cv) {
                    if (vo instanceof Map) {
                        vo = ((Map<?, ?>) vo).values();
                    } else if (!(vo instanceof Collection)) {
                        throw new IllegalArgumentException("`date-parts' must be an array of arrays");
                    }
                    Collection<?> icv = (Collection<?>) vo;
                    dateParts[i] = new int[icv.size()];
                    int j = 0;
                    for (Object ivo : icv) {
                        dateParts[i][j] = toInt(ivo);
                        ++j;
                    }
                    ++i;
                }
                builder.dateParts(dateParts);
            }
        }
        {
            Object v = obj.get("season");
            if (v != null) {
                String season;
                season = v.toString();
                builder.season(season);
            }
        }
        {
            Object v = obj.get("circa");
            if (v != null) {
                Boolean circa;
                circa = toBool(v);
                builder.circa(circa);
            }
        }
        {
            Object v = obj.get("literal");
            if (v != null) {
                String literal;
                literal = v.toString();
                builder.literal(literal);
            }
        }
        {
            Object v = obj.get("raw");
            if (v != null) {
                String raw;
                raw = v.toString();
                builder.raw(raw);
            }
        }

        return builder.build();
    }

    private static boolean isFalsy(Object o) {
        if (o == null) {
            return true;
        }
        if (Boolean.FALSE.equals(o)) {
            return true;
        }
        if ("".equals(o)) {
            return true;
        }
        if (Integer.valueOf(0).equals(o)) {
            return true;
        }
        if (Long.valueOf(0L).equals(o)) {
            return true;
        }
        if (o instanceof Float && (Float.valueOf(0f).equals(o) || ((Float) o).isNaN())) {
            return true;
        }
        if (o instanceof Double && (Double.valueOf(0d).equals(o) || ((Double) o).isNaN())) {
            return true;
        }
        if (Byte.valueOf((byte) 0).equals(o)) {
            return true;
        }
        if (Short.valueOf((short) 0).equals(o)) {
            return true;
        }
        return false;
    }

    private static int toInt(Object o) {
        if (o instanceof CharSequence) {
            return Integer.parseInt(o.toString());
        }
        return ((Number) o).intValue();
    }

    private static boolean toBool(Object o) {
        if (o instanceof String) {
            return Boolean.parseBoolean((String) o);
        } else if (o instanceof Number) {
            return ((Number) o).intValue() != 0;
        }
        return (Boolean) o;
    }

    @Override
    public int hashCode() {
        int result = 1;

        result = 31 * result + Arrays.deepHashCode(dateParts);
        result = 31 * result + ((season == null) ? 0 : season.hashCode());
        result = 31 * result + ((circa == null) ? 0 : circa.hashCode());
        result = 31 * result + ((literal == null) ? 0 : literal.hashCode());
        result = 31 * result + ((raw == null) ? 0 : raw.hashCode());

        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof CSLDate))
            return false;
        CSLDate other = (CSLDate) obj;

        if (!Arrays.deepEquals(dateParts, other.dateParts))
            return false;

        if (season == null) {
            if (other.season != null)
                return false;
        } else if (!season.equals(other.season))
            return false;

        if (circa == null) {
            if (other.circa != null)
                return false;
        } else if (!circa.equals(other.circa))
            return false;

        if (literal == null) {
            if (other.literal != null)
                return false;
        } else if (!literal.equals(other.literal))
            return false;

        if (raw == null) {
            if (other.raw != null)
                return false;
        } else if (!raw.equals(other.raw))
            return false;

        return true;
    }
}
